/**
 * Copyright (c) 2020-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.stext.validation

import com.google.common.collect.Lists
import com.google.common.collect.Sets
import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.ExpressionsPackage
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.PostFixUnaryExpression
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.TypesPackage
import com.yakindu.base.types.Package
import com.yakindu.base.types.scoping.IPackageImport2URIMapper
import com.yakindu.base.types.scoping.IPackageImport2URIMapper.PackageImport
import com.yakindu.sct.model.sgraph.SGraphPackage
import com.yakindu.sct.model.sgraph.Scope
import com.yakindu.sct.model.sgraph.ScopedElement
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.util.ContextElementAdapter
import com.yakindu.sct.model.stext.concepts.CompletionTransition
import com.yakindu.sct.model.stext.extensions.STextExtensions
import com.yakindu.sct.model.stext.services.STextGrammarAccess
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.ImportScope
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.model.stext.stext.InternalScope
import com.yakindu.sct.model.stext.stext.OperationDefinition
import com.yakindu.sct.model.stext.stext.StextPackage
import com.yakindu.sct.model.stext.stext.VariableDefinition
import java.util.LinkedList
import java.util.Optional
import java.util.Set
import org.eclipse.emf.common.util.EList
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EStructuralFeature
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.Keyword
import org.eclipse.xtext.naming.IQualifiedNameProvider
import org.eclipse.xtext.nodemodel.ICompositeNode
import org.eclipse.xtext.nodemodel.INode
import org.eclipse.xtext.nodemodel.util.NodeModelUtils
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.CheckType
import org.eclipse.xtext.validation.ValidationMessageAcceptor

import static extension com.yakindu.base.types.TypesUtil.*

class ScopeValidator extends STextBaseValidator {

	@Inject
	extension IQualifiedNameProvider nameProvider
	@Inject
	extension ExpressionExtensions
	@Inject(optional=true)
	IPackageImport2URIMapper mapper
	@Inject
	STextGrammarAccess grammarAccess
	@Inject protected extension CompletionTransition
	@Inject	protected extension STextExtensions
	
	public static final String WARNING_AMBIGOUS_MSG = "%s is ambiguous. Please consider to use fully qualified name.";
	public static final String WARNING_AMBIGOUS_CODE = "ambigousType";

	@Check(CheckType.FAST)
	def void checkAmbigous(TypeSpecifier typedElement) {
		val type = typedElement.getType()
		if(EcoreUtil2.getRootContainer(type) instanceof Package){
			val node = NodeModelUtils.getNode(typedElement)
			val packageContainedTypes = (EcoreUtil2.getRootContainer(type) as Package).collectPackageContainedTypes(newHashSet)
			if(packageContainedTypes.exists[it !== type && name !== null && name.equals(type.name)] && node.text.contains(type.name) && !node.text.contains(type.toQualifiedName.toString))
				warning(String.format(WARNING_AMBIGOUS_MSG, type.name), TypesPackage.Literals.TYPE_SPECIFIER__TYPE, WARNING_AMBIGOUS_CODE)
		}
	}

	public static final String FEATURE_CALL_TO_SCOPE = "A variable, event or operation is required.";

	@Check(CheckType.FAST)
	def void checkFeatureCall(FeatureCall call) {
		if (call.eContainer() instanceof FeatureCall) {
			return;
		}
		if (call.getFeature() instanceof Scope) {
			error(FEATURE_CALL_TO_SCOPE, ExpressionsPackage.Literals.FEATURE_CALL__FEATURE, INSIGNIFICANT_INDEX)
		}
	}

	@Check(CheckType.FAST)
	def void checkFeatureCall(ElementReferenceExpression call) {
		if (call.eContainer() instanceof FeatureCall) {
			return;
		}
		if (call.getReference() instanceof Scope) {
			error(FEATURE_CALL_TO_SCOPE, ExpressionsPackage.Literals.ELEMENT_REFERENCE_EXPRESSION__REFERENCE,
				INSIGNIFICANT_INDEX)
		}
	}

	public static final String ONLY_ONE_INTERFACE = "Only one default/unnamed interface is allowed.";

	@Check(CheckType.FAST)
	def void checkInterfaceScope(ScopedElement statechart) {
		var defaultInterfaces = new LinkedList()
		for (Scope scope : statechart.getScopes()) {
			if (scope instanceof InterfaceScope && ((scope as InterfaceScope)).getName() === null) {
				defaultInterfaces.add((scope as InterfaceScope))
			}
		}
		if (defaultInterfaces.size() > 1) {
			for (InterfaceScope scope : defaultInterfaces) {
				error(ONLY_ONE_INTERFACE, scope, grammarAccess.getInterfaceScopeAccess().getInterfaceKeyword_1(),
					ValidationMessageAcceptor.INSIGNIFICANT_INDEX, ONLY_ONE_INTERFACE)
			}
		}
	}

	public static final String DUPLICATE_IMPORT = "Duplicate import '%s'.";

	@Check
	def void checkDuplicateImport(ImportScope importScope) {
		var allScopes = EcoreUtil2.getContainerOfType(importScope, ScopedElement).getScopes().filter(ImportScope)
		for (String importToCheck : importScope.getImports()) {
			var Set<String> allImports = Sets.newHashSet()
			for (scope : allScopes) {
				for (anImport : scope.getImports()) {
					if (anImport.equals(importToCheck) && !allImports.add(anImport)) {
						warning(String.format(DUPLICATE_IMPORT, importToCheck), importScope,
							StextPackage.Literals.IMPORT_SCOPE__IMPORTS,
							importScope.getImports().indexOf(importToCheck))
					}
				}
			}
		}
	}

	public static final String ERROR_ASSIGNMENT_TO_READONLY_CODE = "AssignmentToReadonly"
	public static final String ERROR_ASSIGNMENT_TO_READONLY_MSG = "Property '%s' is readonly."

	@Check(CheckType.FAST)
	def void checkAssignmentToReadonlyVariable(AssignmentExpression exp) {
		var referencedObject = exp.getVarRef().featureOrReference
		if (referencedObject instanceof Property) {
			if (referencedObject.readonly && referencedObject.resource != exp.resource) {
				error(String.format(ERROR_ASSIGNMENT_TO_READONLY_MSG, referencedObject.name),
					ExpressionsPackage.Literals.ASSIGNMENT_EXPRESSION__VAR_REF, ERROR_ASSIGNMENT_TO_READONLY_CODE)
			}
		}
	}

	@Check(CheckType.FAST)
	def void checkPostFixUnaryExpressionToReadonlyVariable(PostFixUnaryExpression exp) {
		var referencedObject = exp.getOperand().featureOrReference
		if (referencedObject instanceof Property) {
			if (referencedObject.readonly && referencedObject.resource != exp.resource) {
				error(String.format(ERROR_ASSIGNMENT_TO_READONLY_MSG, referencedObject.name),
					ExpressionsPackage.Literals.UNARY_EXPRESSION__OPERAND, ERROR_ASSIGNMENT_TO_READONLY_CODE)
			}
		}
	}

	public static final String IMPORT_NOT_RESOLVED_MSG = "Import '%s' cannot be resolved.";
	public static final String IMPORT_NOT_RESOLVED_MSG_EXT = " Please check the statechart domain in the statechart properties, e.g. set the C/C++ domain to enable header file imports."

	@Check
	def void checkImportExists(ImportScope scope) {
		var EList<String> imports = scope.getImports()
		for (String packageImport : imports) {
			var Optional<PackageImport> pkImport = mapper.findPackageImport(scope.eResource(), packageImport)

			if (!pkImport.isPresent()) {
				val msg = if (packageImport.endsWith(".ysc") || packageImport.endsWith(".sct"))
						IMPORT_NOT_RESOLVED_MSG
					else
						IMPORT_NOT_RESOLVED_MSG + IMPORT_NOT_RESOLVED_MSG_EXT
				error(String.format(msg, packageImport), scope, StextPackage.Literals.IMPORT_SCOPE__IMPORTS,
					imports.indexOf(packageImport))
			}
		}
	}

	def protected Resource getResource(EObject context) {
		val ContextElementAdapter provider = (EcoreUtil.getExistingAdapter(context.eResource(),
			ContextElementAdapter) as ContextElementAdapter)
		if (provider === null) {
			return context.eResource()
		} else {
			return provider.getElement().eResource()
		}
	}

	public static final String REFERENCE_CONSTANT_BEFORE_DEFINED = "Cannot reference a constant from different scope or before it is defined. '%s' is referring '%s'";

	@Check(CheckType.NORMAL)
	def void checkValueReferenedBeforeDefined(Scope scope) {
		var defined = Sets.newHashSet()
		for (declaration : scope.declarations) {
			if (declaration instanceof VariableDefinition) {
				if(!declaration.isConst()) return;
				var initialValue = declaration.getInitialValue()
				var toCheck = Lists.newArrayList(initialValue)
				var eAllContents = initialValue.eAllContents()
				while (eAllContents.hasNext()) {
					var EObject next = eAllContents.next()
					if(next instanceof Expression) toCheck.add(next)
				}
				for (Expression expression : toCheck) {
					var EObject referencedObject = null
					if (expression instanceof FeatureCall)
						referencedObject = expression.getFeature()
					else if (expression instanceof ElementReferenceExpression)
						referencedObject = expression.getReference()
					if (referencedObject instanceof VariableDefinition) {
						if (!defined.contains(nameProvider.getFullyQualifiedName(referencedObject)))
							error(String.format(REFERENCE_CONSTANT_BEFORE_DEFINED, declaration.name, referencedObject.name), declaration,
								TypesPackage.Literals.PROPERTY__INITIAL_VALUE)
					}
				}
				defined.add(nameProvider.getFullyQualifiedName(declaration))
			}
		}
	}

	public static final String INTERNAL_DECLARATION_UNUSED = "Internal declaration '%s' is not used in statechart.";

	@Check(CheckType.FAST)
	def void checkUnusedVariablesInInternalScope(InternalScope scope) {
		var rootContainer = EcoreUtil.getRootContainer(scope)
		var rootRes = getResource(rootContainer)

		var Statechart statechart = EcoreUtil.getObjectByType(rootRes.getContents(),
			SGraphPackage.Literals.STATECHART) as Statechart

		if (statechart === null)
			return;

		var allUsedElementReferences = EcoreUtil2.getAllContentsOfType(statechart, ElementReferenceExpression)

		if (statechart.getSpecification() !== null) {
			for (Declaration internalDeclaration : scope.declarations) {
				var boolean internalDeclarationUsed = false
				for (ElementReferenceExpression elementReference : allUsedElementReferences) {
					if (elementReference.getReference().eContainer() instanceof InternalScope) {
						if (elementReference.getReference() instanceof Property) {
							if (((elementReference.getReference() as Property)).getName().equals(
								internalDeclaration.getName()) && internalDeclaration instanceof Property) {
								internalDeclarationUsed = true
							// TODO end loop
							}
						} else if (elementReference.getReference() instanceof EventDefinition) {
							if (((elementReference.getReference() as EventDefinition)).getName().equals(
								internalDeclaration.getName()) && internalDeclaration instanceof EventDefinition) {
								internalDeclarationUsed = true
							// TODO end loop
							}
						} else if (elementReference.getReference() instanceof OperationDefinition) {
							if (((elementReference.getReference() as OperationDefinition)).getName().equals(
								internalDeclaration.getName()) && internalDeclaration instanceof OperationDefinition) {
								internalDeclarationUsed = true
							// TODO end loop
							}
						}
					}
				}
				if (!internalDeclarationUsed) {
					if (internalDeclaration instanceof VariableDefinition ||
						internalDeclaration instanceof EventDefinition ||
						internalDeclaration instanceof OperationDefinition)
						warning(String.format(INTERNAL_DECLARATION_UNUSED, internalDeclaration.name), internalDeclaration, null, -1)
				}
			}
		}
	}

	// FIXME WTF
	def protected void error(String message, EObject source, Keyword keyword, int index, String code) {
		val String[] issueData = null
		var ICompositeNode rootNode = NodeModelUtils.findActualNodeFor(source)
		if (rootNode !== null) {
			var INode child = findNode(source, false, rootNode, keyword, #[index])
			if (child !== null) {
				var int offset = child.getTotalOffset()
				var int length = child.getTotalLength()
				getMessageAcceptor().acceptError(message, source, offset, length, code, issueData)
				return;
			}
		}
		error(message, source, (null as EStructuralFeature), ValidationMessageAcceptor.INSIGNIFICANT_INDEX, code)
	}

	// FIXME WTF
	def protected INode findNode(EObject source, boolean sourceFound_finalParam_, INode root, Keyword keyword,
		int[] index) {
		var sourceFound = sourceFound_finalParam_
		if (sourceFound && root.getSemanticElement() !== source) {
			return null
		}
		if (root.getSemanticElement() === source) {
			sourceFound = true
		}
		var EObject grammarElement = root.getGrammarElement()
		// .equals or == does not work because sub grammars use their own
		// Modules with custom
		// grammarAccess instance and .equals is not overwritten.
		if (grammarElement instanceof Keyword && keyword.getValue().equals(((grammarElement as Keyword)).getValue())) {
			if (index.get(0) !== INSIGNIFICANT_INDEX) {
				{
					var _postIndx_index = 0
					var _postVal_index = {
						val _rdIndx_index = _postIndx_index
						index.get(_rdIndx_index)
					}
					{
						val _wrVal_index = index
						val _wrIndx_index = _postIndx_index
						_wrVal_index.set(_wrIndx_index, _postVal_index - 1)
					}
				}
			}
			if (index.get(0) === 0 || index.get(0) === INSIGNIFICANT_INDEX) {
				return root
			}
		}
		if (root instanceof ICompositeNode) {
			var ICompositeNode node = root
			for (INode child : node.getChildren()) {
				var INode result = findNode(source, sourceFound, child, keyword, index)
				if (result !== null) {
					return result
				}
			}
		}
		return null
	}

}
