/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Jonathan Thoene - itemis AG
 * 
 */
package com.yakindu.sct.generator.java.files

import com.google.inject.Inject
import com.yakindu.sct.generator.core.extensions.StringHelper
import com.yakindu.sct.generator.core.filesystem.OutputConfigProvider
import com.yakindu.sct.generator.java.GenmodelEntries
import com.yakindu.sct.generator.java.Naming
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sgen.GeneratorEntry
import org.eclipse.xtext.generator.IFileSystemAccess
import static com.yakindu.sct.generator.core.filesystem.ISCTFileSystemAccess.*
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations

/**
 * 
 * @author jonathan thoene - Initial contribution and API
 * 
 */
class VirtualTimer {

	@Inject extension GenmodelEntries
	@Inject extension Naming
	@Inject extension StatechartAnnotations
	@Inject extension StringHelper
	@Inject extension OutputConfigProvider

	GeneratorEntry entry;
	Statechart statechart;

	def generateVirtualTimerService(ExecutionFlow flow, GeneratorEntry entry, IFileSystemAccess fsa) {
		this.entry = entry;
		this.statechart = entry.statechart

		var fileName = entry.fileName
		var content = entry.toCode

		fsa.generateFile(fileName, entry.libraryOutputConfig, content)
	}
	
	// for SCTUnit java generator use 
	def generateWithStatechart(Statechart st, GeneratorEntry entry, IFileSystemAccess fsa) {
		this.entry = entry;
		this.statechart = st;
		
		var fileName = entry.fileName
		var content = entry.toCode
			
		fsa.generateFile(fileName, LIBRARY_TARGET_FOLDER_OUTPUT, content)
	}

	def protected getFileName(GeneratorEntry it) {
		return '''«it.libraryPackage.toPath»/VirtualTimer.java'''.toString
	}
	
	def protected toCode(GeneratorEntry it) {
		'''
			«entry.licenseText»
			package «libraryPackage»;
			
			import java.math.BigInteger;
			import java.util.PriorityQueue;
			import java.util.Queue;
			
			public class VirtualTimer implements «iTimerService» {
				
				private BigInteger stopTime = BigInteger.ZERO;
				protected BigInteger currentTime = BigInteger.ZERO;
				protected long cyclePeriod = 0;
				protected BigInteger scheduleCount = BigInteger.ZERO;
			
				private Queue<VirtualTimeTask> tasks;
			
				public abstract static class VirtualTimeTask implements Runnable, Comparable<VirtualTimeTask> {
			
					BigInteger nextExecutionTime = BigInteger.ZERO;
					long interval = 0;
					long period = -1;
					BigInteger scheduleOrder = BigInteger.ZERO;
					boolean isCanceled = false;
			
					public int compareTo(VirtualTimeTask o) {
						
						BigInteger diff = BigInteger.ZERO;
						
						if (!nextExecutionTime.equals(o.nextExecutionTime)) {
							diff = nextExecutionTime.subtract(o.nextExecutionTime);
						} else if (o instanceof CycleTimeEventTask && !(this instanceof CycleTimeEventTask)) {
							return -1;
						} else if (!(o instanceof CycleTimeEventTask) && this instanceof CycleTimeEventTask) {
							return 1;
						} else {
							diff = scheduleOrder.subtract(o.scheduleOrder);
						}		
						return diff.compareTo(BigInteger.ZERO);
					}
			
					public boolean isCanceled() {
						return isCanceled;
					}
			
					public void cancel() {
						isCanceled = true;
					}
				}
			
				public static class VirtualTimeEventTask extends VirtualTimeTask {
			
					private final int eventID;
					private «iTimed» callback;
			
					public VirtualTimeEventTask(«iTimed» callback, int eventID) {
						this.callback = callback;
						this.eventID = eventID;
					}
			
					public int getEventId() {
						return eventID;
					}
					
					public «iTimed» getCallback() {
						return callback;
					}
			
					public void run() {
						callback.raiseTimeEvent(eventID);
					}
			
				}
				
				public static class CycleTimeEventTask extends VirtualTimeTask {
			
					private «IF statechart.isCycleBased»«iCycleBased()»«ELSE»«iEventDriven»«ENDIF» statemachine;
			
					public CycleTimeEventTask(«IF statechart.isCycleBased»«iCycleBased()»«ELSE»«iEventDriven»«ENDIF» statemachine) {
						this.statemachine = statemachine;
					}
			
					public void run() {
						«IF statechart.isCycleBased»
							statemachine.runCycle();
						«ENDIF»
					}
			
				}
			
				public VirtualTimer() {
					tasks = new PriorityQueue<VirtualTimeTask>();
				}
				
				public VirtualTimer(long cyclePeriod) {
					tasks = new PriorityQueue<VirtualTimeTask>();
					this.cyclePeriod = cyclePeriod;
				}
			
				public void timeLeap(long ms) {
					stopTime = currentTime.add(BigInteger.valueOf(ms));
					processTasks();
				}
				
				public void cycleLeap(long cycles){
					int elapsedCycles = 0;
			
					while (elapsedCycles < cycles) {
			
						VirtualTimeTask cycleTask = getCycleTask();
						if (cycleTask == null)
							return;
			
						long timeToNextCycle = cycleTask.nextExecutionTime.subtract(currentTime).longValue();
						timeLeap(timeToNextCycle);
						elapsedCycles += 1;
					}
				}
			
				@Override
				public void setTimer(«iTimed» callback, int eventID, long duration, boolean isPeriodical) {
					if (duration <= 0)
						duration = 1;
					VirtualTimeEventTask timeEventTask = new VirtualTimeEventTask(callback, eventID);
					if (isPeriodical) {
						schedulePeriodicalTask(timeEventTask, duration, duration);
					} else {
						scheduleTask(timeEventTask, duration);
					}
				}
			
				@Override
				public void unsetTimer(«iTimed» callback, int eventID) {
					VirtualTimeTask timerTask = getTask(callback, eventID);
					if (timerTask != null)
						timerTask.cancel();
				}
			
				public void scheduleTask(VirtualTimeTask task, long interval) {
					task.interval = interval;
					scheduleInternal(task, currentTime.add(BigInteger.valueOf(interval)), -1);
				}
			
				public void schedulePeriodicalTask(VirtualTimeTask task, long interval, long period) {
					scheduleInternal(task, currentTime.add(BigInteger.valueOf(interval)), period);
				}
			
				private void scheduleInternal(VirtualTimeTask task, BigInteger time, long period) {
					task.nextExecutionTime = time;
					task.period = period;
					task.scheduleOrder = scheduleCount;
					scheduleCount = scheduleCount.add(BigInteger.ONE);
					tasks.add(task);
				}
			
				protected VirtualTimeTask getTask(«iTimed» callback, int eventName) {
					for (VirtualTimeTask virtualTimeTask : tasks) {
						if (!(virtualTimeTask instanceof VirtualTimeEventTask))
							continue;
						if (((VirtualTimeEventTask) virtualTimeTask).getEventId() == eventName
								&& ((VirtualTimeEventTask) virtualTimeTask).getCallback() == callback)
							return virtualTimeTask;
					}
					return null;
				}
				
				protected CycleTimeEventTask getCycleTask() {
					for (VirtualTimeTask task : tasks) {
						if (task instanceof CycleTimeEventTask && !task.isCanceled()) {
							return (CycleTimeEventTask) task;
						}
					}
					return null;
				}
				
				protected void processTasks() {
					boolean processTasks = !tasks.isEmpty();
					while (processTasks) {
						VirtualTimeTask task = tasks.peek();
						if (task == null)
							break;
						if (task.isCanceled) {
							tasks.remove();
							continue;
						}
						
						if (task.nextExecutionTime.compareTo(stopTime) <= 0) {
							currentTime = task.nextExecutionTime;
							task = tasks.poll();
							if (task.period > -1) {
								task.nextExecutionTime = currentTime.add(BigInteger.valueOf(task.period));
								tasks.add(task);
							}
							task.run();
						} else {
							currentTime = stopTime;
							processTasks = false;
						}
					}
				}
			
				public synchronized void stop() {
					for (VirtualTimeTask timerTask : tasks) {
						timerTask.cancel();
					}
					cancel();
				}
			
				public void cancel() {
					synchronized (tasks) {
						tasks.clear();
					}
				}
			
			}
		'''
	}
}
