/**
* Copyright (c) 2022 itemis AG - All rights Reserved
* Unauthorized copying of this file, via any medium is strictly prohibited
*
*/


package com.yakindu.sct.generator.c.typesystem

import com.google.inject.Inject
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.base.types.inferrer.ITypeSystemInferrer.InferenceResult
import com.yakindu.base.types.validation.IValidationIssueAcceptor
import com.yakindu.base.types.validation.IValidationIssueAcceptor.ValidationIssue
import com.yakindu.base.types.validation.IValidationIssueAcceptor.ValidationIssue.Severity
import com.yakindu.base.types.validation.TypeValidator
import com.yakindu.sct.model.sgraph.Statechart

import static com.yakindu.base.types.inferrer.AbstractTypeSystemInferrer.ASSERT_COMPATIBLE
import static com.yakindu.base.types.inferrer.ITypeSystemInferrer.NOT_COMPATIBLE_CODE

class CTypeValidator extends TypeValidator {
	
	@Inject protected extension OriginTracing

	override assertAssignable(InferenceResult varResult, InferenceResult valueResult, String msg,
		IValidationIssueAcceptor acceptor) {
		if (isNullOnPointer(varResult, valueResult)) {
			return;
		}
		if (isNullOnComplexType(varResult, valueResult)) {
			acceptor.accept(new ValidationIssue(
				Severity.ERROR,
				msg ?: String.format(ASSERT_COMPATIBLE, varResult, valueResult),
				NOT_COMPATIBLE_CODE
			));
			return;
		}
		if (valueResult.type.isAnyType) {
			return;
		}
		super.assertAssignable(varResult, valueResult, msg, acceptor);
	}
	
	override assertCompatible(InferenceResult result1, InferenceResult result2, String msg, IValidationIssueAcceptor acceptor) {
		if (result1?.type.isAnyType) {
			return;
		}
		//This case is added due to the fact that we handle Statechart types as pointers
		if (result1?.type.origin instanceof Statechart && CTypeSystem.POINTER_TYPES.exists[pt | result2.type.name.equals(pt)]) {
			return;
		}
		//This case is added due to the fact that the statechart declared statechart type uses the SCT2Type declared CT,
		//while the CToModel trafo creates a different CT for that type
		//TODO: Refactor CToModel trafo to use SCT2Type declared type
		if (result1?.type.origin instanceof Statechart && result2?.type.name == result1?.type.name) {
			return;
		}	
 		super.assertCompatible(result1, result2, msg, acceptor)
	}
	
	def isNullOnPointer(InferenceResult varResult, InferenceResult valueResult) {
		if (varResult.isPointer && valueResult.isNull) {
			return true;
		}
		return false
	}
	
	override assertTypeBindingsSame(InferenceResult result1, InferenceResult result2, String msg,
			IValidationIssueAcceptor acceptor) {
	
		if (result1.isAnyPointer || result2.isAnyPointer || result1.isVoidPointer || result2.isVoidPointer) {
			return
		}
		
		//For now we check whether the CToModel trafo type name is equivalent to the Sct2Type produced CT name
		//TODO: Use Sct2Type provided type also in CToModel trafo
		if(!result1?.bindings.nullOrEmpty && result2?.bindings.nullOrEmpty){
			val result1Binding = result1?.bindings.head?.type
			val result2Binding = result2?.bindings.head?.type
			if(result1Binding.origin instanceof Statechart && result1Binding?.name == result2Binding?.name ||
				result2Binding.origin instanceof Statechart && result1Binding?.name == result2Binding?.name)
			{
				return
			}
		}		
		super.assertTypeBindingsSame(result1, result2, msg, acceptor);
	}
	
	def boolean isAnyPointer(InferenceResult result) {
		return (result.isPointer && !result.bindings.isEmpty && (result.bindings.head.type.isAnyType || result.bindings.head.isAnyPointer))
	}
	
	def boolean isVoidPointer(InferenceResult result) {
		return (result.isPointer && !result.bindings.isEmpty && (registry.isVoid(result.bindings.head.type) || result.bindings.head.isVoidPointer))
	}
	
	def boolean isPointer(InferenceResult result) {
		if(result !== null) result.type instanceof ComplexType && CTypeSystem.POINTER_TYPES.exists[pt | result.type.name.equals(pt)]
		else false
	}
	
}