/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Thomas Kutz - itemis AG
 * 
 */
package com.yakindu.sct.domain.c.runtime.scoping

import com.google.common.base.Predicate
import com.google.inject.Inject
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Package
import com.yakindu.sct.domain.c.runtime.resource.transform.c.CTypeUtils
import com.yakindu.sct.model.stext.scoping.STextScopeProvider
import org.eclipse.emf.ecore.EReference
import org.eclipse.xtext.naming.QualifiedName
import org.eclipse.xtext.resource.IEObjectDescription
import org.eclipse.xtext.scoping.IScope
import org.eclipse.xtext.scoping.Scopes
import org.eclipse.xtext.scoping.impl.FilteringScope

import static extension org.eclipse.xtext.EcoreUtil2.*

/**
 * Enhance the scope of a package to be namespace aware, i.e. if the same package is multiple times in scope, 
 * as allowed by {@link CDomainImportScope}, collect all packages' members. This reflects the fact that C++ 
 * namespaces can be defined in multiple header files.
 * 
 * @author Thomas Kutz
 */
class CDomainScopeProvider extends STextScopeProvider {
	
	@Inject protected extension CTypeUtils
	
	override protected IScope addScopeForPackage(Package pkg, IScope parentScope,
		Predicate<IEObjectDescription> predicate, Expression context, EReference reference) {

		val names = pkg.allContainers.filter(Package).filter[eContainer !== null].map[name].toList.reverse
		names.add(pkg.name)
		val namespaceFQN = QualifiedName.create(names)
		val delegateScope = delegate.getScope(context, reference)
		val namespacePkgs = delegateScope.getElements(namespaceFQN)
		val namespaceMembers = namespacePkgs.map[EObjectOrProxy].filter(Package).map[member].flatten

		var scope = parentScope
		scope = Scopes.scopeFor(namespaceMembers, scope)
		scope = new FilteringScope(scope, predicate)
		return scope
	}
}
