/**
 * Copyright (c) 2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 */
package com.yakindu.base.types.libraries

import com.google.common.collect.Sets
import com.google.inject.Binder
import com.google.inject.Inject
import com.google.inject.Singleton
import com.google.inject.multibindings.Multibinder
import com.google.inject.name.Named
import com.google.inject.name.Names
import java.util.HashSet
import java.util.Set
import org.eclipse.emf.common.util.URI

/**
 * The type library registry is a composite type library provider which handles a set of type library providers.
 * Individual type library providers can by injected using a named guice multi-binding.
 * 
 * Example for setting up the binding 
 * <pre>
 * public void configure(Binder binder) {
 * 	Multibinder<ITypeLibraryProvider> typeLibraryBinder = Multibinder.newSetBinder(binder,
 * 	ITypeLibraryProvider.class, Names.named(TypeLibraryRegistry.LIBRARY_PROVIDERS));
 * 	typeLibraryBinder.addBinding().to(MyTypeLibrary.Provider.class);
 * }
 * </pre>
 * 
 * or simply using the bind utility method:
 * 
 * <pre>
 * public void configure(Binder binder) {
 *     TypeLibraryRegistry.bind(binder, MyTypeLibrary.Provider.class);
 * }
 * </pre>
 
 */
@Singleton
class TypeLibraryRegistry implements ITypeLibraryProvider {
	
	/** 
	 * Identifier of the named guice multi-binding.*/
	public static final String LIBRARY_PROVIDERS = "LibraryProviders";
	
	static def void bind(Binder binder, Class<? extends ITypeLibraryProvider> providerClass) {
		val typeLibraryBinder = Multibinder.newSetBinder(
			binder,
			ITypeLibraryProvider, 
			Names.named(LIBRARY_PROVIDERS));
		typeLibraryBinder.addBinding().to(providerClass);
		
	}
	
	@Inject(optional = true)
	@Named(LIBRARY_PROVIDERS)
	Set<ITypeLibraryProvider> providers = Sets.newHashSet();
	
	override provide() {
		val libraries = new HashSet<ITypeLibrary>
		
		providers.forEach[ libraries += provide ]
		
		libraries
	}
	
	def provide(URI libUri) {
		provide.filter[it.libraryURI == libUri].head
	}
}