/**
 * Copyright (c) 2022-24 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */

package com.yakindu.base.expressions.interpreter.types

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.interpreter.base.IInstanceFactory
import com.yakindu.base.expressions.interpreter.base.IInterpreter
import com.yakindu.base.expressions.interpreter.base.InterpreterException
import com.yakindu.base.expressions.interpreter.base.ResolvingRuntimeInstance
import com.yakindu.base.expressions.interpreter.base.ValueSemantics
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Direction
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.Part
import com.yakindu.base.types.Property
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.base.types.typesystem.ITypeValueProvider
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionEvent
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ReferenceSlot
import org.eclipse.xtext.EcoreUtil2

/**
 * Base implementation which provides basis type system conventions for execution instances.
 *  
 * @author Axel Terfloth
 */
abstract class RuntimeInstance extends ResolvingRuntimeInstance {
	
	public static final String INIT = "init";
	public static final String ON_START = "onStart";
	
	
	protected boolean isInitialized = false
	protected IInstanceFactory factory

	@Inject protected extension ValueSemantics
	@Inject protected extension ITypeSystem
	@Inject protected extension ITypeValueProvider


	abstract def Package declarationPackage()
	abstract def Iterable<Declaration> declarations()
		

	def void setUp(CompositeSlot instance, IInterpreter.Context context, IInstanceFactory factory) {
		super.setUp(instance, context)
		
		this.factory = factory
		this.isInitialized = false
	}
	
	
	//=========================================================================================
	//
	// Instance behavior implementation
	//
	
	/**
	 * The instance members have no default values before this method is called. This methods transformes 
	 * the instance from allocated to initialized state.  
	 */
	def void init() {
		
		if (! isInitialized) {
			
			instanceSlot._executeOperation( '''«instanceSlot.fqName».«INIT»''', #[], [
				
				declarations.forEach[
					it.initMember
				]
				_execute[isInitialized = true]		
			])				
		}
				
		
	}
	
	
	def protected dispatch void initMember(Declaration it){
		// do nothing by default
	}
	
	def protected dispatch void initMember(Property p) {
		
		if ( p.initialValue !== null) {
	
			val slot = instanceSlot.resolve(p.name) as ExecutionSlot
	
			p.initialValue._delegate
			_value
			
			_execute('''initialize «slot.fqName»''', [
				var value = popValue
				_valueSemantics.setValue(slot, value)
			])
		}
	}
	
	
	def protected dispatch void initMember(Part p){
		
		val slot = instanceSlot.resolve(p.name) as ExecutionSlot
		
		if ( slot instanceof ReferenceSlot ) {
		
			_execute('''new part «slot.fqName»''', [
				val part = factory.newInstance(p.type) => [
					if (it instanceof ExecutionSlot) it.fqName = slot.fqName
				]
				slot.reference = part as ExecutionSlot
			])
			
		}
	}
	
	
	def protected dispatch isPart(Part it) {
		true
	}
	
	def protected dispatch isPart(Property it) {
		it.isConst && it.type instanceof ComplexType
	}
	
	
	def protected void onStart() {
		ON_START.lookupOperation?.doInvoke
	}
	
	
	//=========================================================================================
	//
	// Handling invocations
	//
		
	def dispatch void invoke(Object slot, Object operation) {
		operation.doInvoke
	}
	
	def dispatch void invoke(ReferenceSlot slot, Object operation) {
		slot.reference.instance.invoke(slot.reference, operation)
	}


	def dispatch void doInvoke(Object operation) {
		throw new InterpreterException("Cannot invoke operation: " + operation)
	}
	
	def dispatch void doInvoke(Void void) {
		throw new InterpreterException("Cannot invoke null operation")
	}
	
	def dispatch void doInvoke(Operation it) {

		if (implementation !== null)  {
			executeImplementation
		}
		else {
			executeMock
		}
	}
	
	
	def protected void executeImplementation(Operation it) {
		if (implementation !== null)  {
			_executeOperation('''«(instanceSlot as ExecutionSlot).name».«name»''', instanceSlot, name, parameters.map[name], [
				implementation._delegate
			])
		}
	}
	
	
	def protected void executeMock(Operation it) {
		_executeOperation('''mock «(instanceSlot as ExecutionSlot).name».«name»''', instanceSlot, name, parameters.map[name], [
			val returnType = it.typeSpecifier.type
			if (! returnType.isVoid) {
				_return[
					if (returnType instanceof ComplexType && ! (returnType instanceof EnumerationType)) {
						factory.newInstance(it.typeSpecifier)
					}
					else {
						type.defaultValue
					}
				]
			}
		])
	}
	
	
	def dispatch void doInvoke(ArgumentExpression expr) {
		val feature = expr.featureOrReference
		
		switch (feature) {

			Operation case 'init'==feature.name : init
				
			Operation: {
				val method = resolveOperation(feature)
				if (method !== null) method.doInvoke
				else expr.doInvoke(feature)		
			}
			default:
				throw new InterpreterException("Cannot invoke :"+ feature)			
		}
	}

	
	def dispatch void doInvoke(Object caller, Object operation) {
		throw new UnsupportedOperationException('''Don't know how «caller» calls «operation»''')				
	}


	def dispatch void doInvoke(ArgumentExpression caller, Operation operation) {
		operation.doInvoke
	}



	//=========================================================================================
	//
	// Handling event raising
	//
	
	override raise(Object slot, Object value) {
		if (slot instanceof ExecutionEvent) {
			_execute('''raise «slot.name»''', [
				if (slot.direction == Direction::LOCAL /* && internalEventQueue !== null */ ) {
					_execute('''internalEventQueue.add(«slot.name»)''', [
//						internalEventQueue.add(new EventInstance(slot, value));
					])
				} else if (slot.direction == Direction::IN /* && inEventQueue !== null */ ) {
					_execute('''inEventQueue.add(«slot.name»)''', [
//						inEventQueue.add(new EventInstance(slot, value));
					])
					"runCycle"._exec
				} else {
					_valueSemantics.setValue(slot, value)
					slot.raised = true
					
					if ( slot.direction == Direction.OUT) {
						slot.connections.forEach[ c |
							_launch('''raise(«c.name»''', [
								c.instance.raise(c, value)
							])
//							c.instance.raise(c, value)
						]	
					} 
//					if (type.statechart.isEventDriven && slot.direction == Direction.IN) {
//						"runCycle"._exec
//					}
				}				
			])	
		}	
	}
	


	//=========================================================================================
	//
	// Handling property modification
	//

	override set(Object slot, Object value) {
		_valueSemantics.setValue(slot, value)
	}
	
		
	

	//=========================================================================================
	//
	// Handling executions
	//

	override provideExecution(Object program) {
		switch (program) {
									 
			Operation case declarations.filter(Operation).contains(program): program.doInvoke 
						
			case 'init': init

			case 'onStart' : onStart
			
			String case program.hasMethod : program.lookupOperation.doInvoke 
			
			Invokation : program.caller.doInvoke(program.operation)
			
			default : 
				throw new IllegalArgumentException("Cannot execute '" + program +"'")
		}
	}
	
	
	
	//=========================================================================================
	//
	// resolving slots
	//
	
	protected CompositeSlot heap
	def protected getHeap() {
		if (this.heap !== null ) return this.heap
		EcoreUtil2.getAllContainers(instanceSlot).filter(CompositeSlot).findFirst[name == "heap"]
	}
	
	
	/**
	 * TODO: base class is bound to layout of statechart execution context so we have to overwrite this  Refactor base class based on this implementation
	 */
	override dispatch ExecutionSlot resolveSlot(CompositeSlot slot, String symbol) {
		var ExecutionSlot s = slot.slotByName(symbol)
		 
		if ( s === null ) {
			val pkgSlot = declarationPackage.name.resolvePackageSlotById
			if (pkgSlot !== slot) s = context.resolve(pkgSlot, symbol) as ExecutionSlot
		}
		if ( s === null ) {
			for (import : imported) {
				if (import instanceof CompositeSlot) {
					s = import.slotByName(symbol)	
				}
				if (s !== null) return s					
			}	
		}
		
		if ( s === null ) { 
			_throw(new InterpreterException('''Can not resolve <«symbol»> for <«slot.name»>'''))	
		}
		
		return s
	}	
	
	
	def protected imported() {
		declarationPackage
			.imports
			.map[it.resolvePackageSlotById]
	}


	def protected resolvePackageSlotById(String id) {
		id
			.split("\\.")
			.fold( getHeap, 
				[ s, c | 
					s
						.slots
						.filter(CompositeSlot)
						.findFirst[name == c]
				]
			)
	}
	
	
	//=========================================================================================
	//
	// utility extension methods
	//
	
	def protected boolean hasMethod(String name) {
		lookupOperation(name) !== null
	}	
	
	def lookupOperation(String name) {
		declarations.filter(Operation).filter[it.name == name].head
	}
	
	def protected allOperations(Package it) {
		return member.filter(Operation).toList
	}
	
	def protected dispatch featureOrReference(FeatureCall it) {feature}
	def protected dispatch featureOrReference(ElementReferenceExpression it) {reference}
	
	
	def protected Operation resolveOperation(Declaration decl) {
		declarations.filter(Operation).findFirst[name == decl.name]
	} 
	
}
