/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.stext.validation

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.EventSpec
import com.yakindu.base.expressions.expressions.ExpressionsPackage
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.PostFixUnaryExpression
import com.yakindu.base.expressions.expressions.ReactionEffect
import com.yakindu.base.expressions.expressions.ReactionTrigger
import com.yakindu.base.expressions.expressions.RegularEventSpec
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Direction
import com.yakindu.base.types.Event
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Property
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.model.sgraph.util.StatechartUtil
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.extensions.STextExtensions
import com.yakindu.sct.model.stext.stext.AlwaysEvent
import com.yakindu.sct.model.stext.stext.EntryEvent
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.ExitEvent
import com.yakindu.sct.model.stext.stext.LocalReaction
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.naming.IQualifiedNameProvider
import org.eclipse.xtext.nodemodel.util.NodeModelUtils
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.CheckType

class ReactionValidator extends STextBaseValidator {

	@Inject
	extension ExpressionExtensions
	@Inject
	extension STextExtensions
	@Inject
	extension IQualifiedNameProvider nameProvider
	@Inject
	extension StatechartUtil
	@Inject
	extension StatechartAnnotations

	static final String ENTRY_EXIT_TRIGGER_NOT_ALLOWED_MSG = "Entry and exit events are allowed as local reactions only.";
	public static final String ENTRY_EXIT_TRIGGER_NOT_ALLOWED_CODE = "EntryExitNotAllowed";

	@Check(CheckType.FAST)
	def void checkReactionTrigger(ReactionTrigger reactionTrigger) {
		for (EventSpec eventSpec : reactionTrigger.getTriggers()) {
			if (!(reactionTrigger.eContainer() instanceof LocalReaction) &&
				(eventSpec instanceof EntryEvent || eventSpec instanceof ExitEvent)) {
				error(ENTRY_EXIT_TRIGGER_NOT_ALLOWED_MSG, ExpressionsPackage.Literals.REACTION_TRIGGER__TRIGGERS,
					INSIGNIFICANT_INDEX, ENTRY_EXIT_TRIGGER_NOT_ALLOWED_CODE)
			}
		}
	}

	public static final String TRIGGER_IS_NO_EVENT = "Trigger is no event.";

	@Check(CheckType.FAST)
	def void checkReactionTriggerRegularEvent(ReactionTrigger reactionTrigger) {
		for (var int i = 0; i < reactionTrigger.getTriggers().size(); i++) {
			var EventSpec eventSpec = reactionTrigger.getTriggers().get(i)
			if (eventSpec instanceof RegularEventSpec) {
				var EObject element = eventSpec.getEvent().featureOrReference
				if (element !== null && (!(element instanceof Event))) {
					val triggerNode = NodeModelUtils.getNode(eventSpec);
					val elementName = NodeModelUtils.getTokenText(triggerNode);

					error(
						'''Trigger '«elementName»' is no event.''',
						reactionTrigger,
						ExpressionsPackage.Literals.REACTION_TRIGGER__TRIGGERS,
						i,
						TRIGGER_IS_NO_EVENT,
						#[contextElementURI.toPlatformString(true) + "#" + contextElementURI.fragment, elementName]
					)
				}
			}
		}
	}

	public static final String FEATURE_CALL_HAS_NO_EFFECT = "FeatureCall has no effect.";

	@Check(CheckType.FAST)
	def void checkReactionEffectActions(ReactionEffect effect) {
		for (Expression exp : effect.getActions()) {
			if (!(exp instanceof AssignmentExpression) && !(exp instanceof EventRaisingExpression) &&
				!(exp instanceof PostFixUnaryExpression)) {
				if (exp instanceof FeatureCall) {
					checkFeatureCallEffect(exp)
				} else if (exp instanceof ElementReferenceExpression) {
					checkElementReferenceEffect(exp)
				} else {
					error("Action has no effect.", ExpressionsPackage.Literals.REACTION_EFFECT__ACTIONS,
						effect.getActions().indexOf(exp), FEATURE_CALL_HAS_NO_EFFECT)
				}
			}
		}
	}

	def protected void checkElementReferenceEffect(ElementReferenceExpression refExp) {
		if (!(refExp.getReference() instanceof Operation)) {
			if (refExp.getReference() instanceof Property) {
				error('''Access to property '«»«nameProvider.getFullyQualifiedName(refExp.getReference())»' has no effect.''',
					refExp, ExpressionsPackage.Literals.ELEMENT_REFERENCE_EXPRESSION__REFERENCE, INSIGNIFICANT_INDEX,
					FEATURE_CALL_HAS_NO_EFFECT)
			} else if (refExp.getReference() instanceof Event) {
				error('''Access to event '«»«nameProvider.getFullyQualifiedName(refExp.getReference())»' has no effect.''',
					refExp, ExpressionsPackage.Literals.ELEMENT_REFERENCE_EXPRESSION__REFERENCE, INSIGNIFICANT_INDEX,
					FEATURE_CALL_HAS_NO_EFFECT)
			} else {
				error('''Access to feature '«»«nameProvider.getFullyQualifiedName(refExp.getReference())»' has no effect.''',
					refExp, ExpressionsPackage.Literals.ELEMENT_REFERENCE_EXPRESSION__REFERENCE, INSIGNIFICANT_INDEX,
					FEATURE_CALL_HAS_NO_EFFECT)
			}
		}
	}

	def protected void checkFeatureCallEffect(FeatureCall call) {
		if (call.getFeature() !== null && !(call.getFeature() instanceof Operation)) {
			if (call.getFeature() instanceof Property) {
				error('''Access to property '«»«nameProvider.getFullyQualifiedName(call.getFeature())»' has no effect.''',
					call, ExpressionsPackage.Literals.FEATURE_CALL__FEATURE, INSIGNIFICANT_INDEX,
					FEATURE_CALL_HAS_NO_EFFECT)
			} else if (call.getFeature() instanceof Event) {
				error('''Access to event '«»«nameProvider.getFullyQualifiedName(call.getFeature())»' has no effect.''',
					call, ExpressionsPackage.Literals.FEATURE_CALL__FEATURE, INSIGNIFICANT_INDEX,
					FEATURE_CALL_HAS_NO_EFFECT)
			} else {
				error('''Access to feature '«»«nameProvider.getFullyQualifiedName(call.getFeature())»' has no effect.''',
					call, ExpressionsPackage.Literals.FEATURE_CALL__FEATURE, INSIGNIFICANT_INDEX,
					FEATURE_CALL_HAS_NO_EFFECT)
			}
		}
	}

	public static final String REACTING_ON_SUBMACHINES_IN_EVENT = "Can not react on submachine's in event. Use an out event instead of: "
	public static final String REACTING_ON_SUBMACHINES_IN_EVENT_CODE = "ReactingOnSubmachinesInEventCode"

	@Check(CheckType.FAST)
	def void checkNotReactingOnsubmachinesInEvent(ReactionTrigger reaction) {
		reaction.triggers.filter(RegularEventSpec).forEach [ trigger |
			val event = trigger.event.featureOrReference
			if (event instanceof EventDefinition) {
				if (event.direction == Direction.IN && event.eContainer.isMultiSM) {
					error(REACTING_ON_SUBMACHINES_IN_EVENT + event.name, trigger, null, REACTING_ON_SUBMACHINES_IN_EVENT_CODE)
				}
			}
		]
	}
	
	
	public static final String SELF_TRANSITION_INFINITE_LOOP = "Using %s trigger on a self-transition with enabled super steps may cause an infinite loop."
	public static final String SELF_TRANSITION_INFINITE_LOOP_CODE = "SelfTransitionInfiniteLoop"
	
	@Check(CheckType.FAST)
	def void checkSelfTransitionInfiniteLoop(ReactionTrigger reaction) {
		
		if (!reaction.statechart.isSuperStep) return;
		if (!reaction.eContainer.isSelfTransition) return;
		if (reaction.guard !== null) return;
		
		reaction.triggers.filter(RegularEventSpec).forEach [ trigger |
			warning(String.format(SELF_TRANSITION_INFINITE_LOOP, "event"), trigger, null, SELF_TRANSITION_INFINITE_LOOP_CODE)
		]
		reaction.triggers.filter(AlwaysEvent).forEach [ trigger |
			warning(String.format(SELF_TRANSITION_INFINITE_LOOP, "always"), trigger, null, SELF_TRANSITION_INFINITE_LOOP_CODE)
		]
	}
	
	def protected dispatch isSelfTransition(EObject it) {
		false
	}
	def protected dispatch isSelfTransition(Transition t) {
		t.source === t.target
	}

	def protected getContextElementURI() {
		val fake = super.getCurrentObject()
		EcoreUtil.getURI(fake.contextElement)
	}

}
