/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Jonathan Thoene - itemis AG
 * 
 */
package com.yakindu.sctunit.generator.python.extensions

import com.google.inject.Inject
import com.yakindu.sct.model.sgen.GeneratorEntry
import com.yakindu.sctunit.generator.base.extensions.BaseNavigationExtensions
import com.yakindu.sctunit.generator.python.features.PythonSCTUnitGenmodelEntries
import com.yakindu.sctunit.sCTUnit.SCTUnitElement
import org.eclipse.xtext.generator.IFileSystemAccess

import static com.yakindu.sct.generator.core.filesystem.ISCTFileSystemAccess.*

/**
 * 
 * @author jonathan thoene - Initial contribution and API
 * 
 */
class PythonVirtualTimerExtensions {

	@Inject extension PythonSCTUnitGenmodelEntries
	@Inject extension BaseNavigationExtensions
	
	GeneratorEntry entry;

	def generate(GeneratorEntry entry, IFileSystemAccess fsa, String targetPath) {
		this.entry = entry;
		
		var testPackage = entry.testPackage
		var content = testPackage.SCTUnitElement.toCode
			
		fsa.generateFile(targetPath, LIBRARY_TARGET_FOLDER_OUTPUT, content)
	}

	def protected toCode(SCTUnitElement it) {
		'''
			«entry.licenseText»
			
			import queue
			import functools
			
			class VirtualTimer:
			
				"""A `virtual`timer that actually doesn't use real time.
				Instead all the processing is done via PriorityQueue.
				"""
			
				def __init__(self, cycle_period=0):
					self.stop_time = 0
					self.current_time = 0
					self.cycle_period = cycle_period
					self.schedule_count = 0
					self.tasks = queue.PriorityQueue()
				
				def time_leap(self, ms):
					self.stop_time = self.current_time + ms
					self.process_tasks()
				
				def cycle_leap(self, cycles):
					for _ in range(cycles):
						cycle_task = self.get_cycle_task()
						if cycle_task is not None:
							self.time_leap(cycle_task.next_execution_time - self.current_time)
				
				def set_timer(self, callback, event_id, duration, is_periodical):
					if duration <= 0:
						duration = 1
					time_event_task = VirtualTimeEventTask(callback, event_id)
					if is_periodical:
						self.schedule_periodical_task(time_event_task, duration, duration)
					else:
						self.schedule_task(time_event_task, duration)
				
				def unset_timer(self, callback, event_id):
					task_found = self.get_task(callback, event_id)
					if task_found is not None:
						task_found.cancel()
						self.tasks.put(task_found)
						return True
					return False
				
				def schedule_task(self, task, interval):
					task.interval = interval
					self.schedule_internal(task, self.current_time + interval, -1)
				
				def schedule_periodical_task(self, task, interval, period):
					self.schedule_internal(task, self.current_time + interval, period)
				
				def schedule_internal(self, task, time, period):
					task.next_execution_time = time
					task.period = period
					task.schedule_order = self.schedule_count
					self.schedule_count += 1
					self.tasks.put(task)
				
				def process_tasks(self):
					process = not self.tasks.empty()
					while not self.tasks.empty() and process:
						task = self.tasks.get()
						if task is None:
							self.tasks.task_done()
							break
						if task.is_canceled:
							self.tasks.task_done()
							continue
						if task.next_execution_time <= self.stop_time:
							self.current_time = task.next_execution_time
							if task.period > -1:
								self.schedule_periodical_task(task, task.period, task.period)
							task.run()
							self.tasks.task_done()
						else:
							self.tasks.task_done()
							self.tasks.put(task)
							self.current_time = self.stop_time
							process = False
				
				def stop(self):
					temp_task_list = []
					for _ in range(self.tasks.qsize()):
						temp = self.tasks.get()
						temp.cancel()
						temp_task_list.append(temp)
						self.tasks.task_done()
					for task in temp_task_list:
						self.tasks.put(task)
				
				def cancel(self):
					for _ in range(self.tasks.qsize()):
						self.tasks.get()
						self.tasks.task_done()
						
				def get_task(self, callback, event_id):
					"""This method could be optimized in terms of performance.
					"""
					# find task
					task_found = None
					temp_task_list = []
					for _ in range(self.tasks.qsize()):
						temp = self.tasks.get()
						temp_task_list.append(temp)
						self.tasks.task_done()
						if not isinstance(temp, VirtualTimeEventTask):
							continue
						if not callback == temp.callback:
							continue
						if temp.event_id == event_id:
							task_found = temp
							break
					# add all removed tasks to queue
					for task in temp_task_list:
						self.tasks.put(task)
					
					return task_found
				
				def get_cycle_task(self):
					"""This method could be optimized in terms of performance.
					"""
					# find task
					task_found = None
					temp_task_list = []
					for _ in range(self.tasks.qsize()):
						temp = self.tasks.get()
						temp_task_list.append(temp)
						self.tasks.task_done()
						if isinstance(temp, CycleTimeEventTask):
							task_found = temp
							break
					# add all removed tasks to queue
					for task in temp_task_list:
						self.tasks.put(task)
					
					return task_found
				
			@functools.total_ordering
			class VirtualTimeTask:
			
				def __init__(self):
					self.next_execution_time = 0
					self.interval = 0
					self.period = -1
					self.schedule_order = 0
					self.is_canceled = False
				
				def __eq__(self, other):
					return (self.next_execution_time == other.next_execution_time)\
							and (isinstance(other, CycleTimeEventTask) and isinstance(self, CycleTimeEventTask))\
							and (self.period == other.period)\
							and (self.schedule_order == other.schedule_order)
				
				def __lt__(self, other):
					if self.next_execution_time != other.next_execution_time:
						return (self.next_execution_time - other.next_execution_time) < 0
					if isinstance(other, CycleTimeEventTask) and not isinstance(self, CycleTimeEventTask):
						return True
					if not isinstance(other, CycleTimeEventTask) and isinstance(self, CycleTimeEventTask):
						return False
					if self.period != other.period:
						return (self.period - other.period) < 0
					return (self.schedule_order - other.schedule_order) < 0
				
				def cancel(self):
					self.is_canceled = True
			
			class VirtualTimeEventTask(VirtualTimeTask):
			
				def __init__(self, callback, event_id):
					self.callback = callback
					self.event_id = event_id
					super().__init__()
				
				def run(self):
					self.callback.time_elapsed(self.event_id)
					
			class CycleTimeEventTask(VirtualTimeTask):
			
				def __init__(self, statemachine):
					self.statemachine = statemachine
					super().__init__()
				
				def run(self):
					self.statemachine.run_cycle()
		'''
	}
}