/**
 * Copyright (c) 2012-2025 committers of itemis CREATE and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     committers of itemis CREATE - initial API and implementation
 */
package com.yakindu.sct.model.sgraph.validation;

import com.yakindu.sct.model.sgraph.Choice
import com.yakindu.sct.model.sgraph.Exit
import com.yakindu.sct.model.sgraph.Synchronization
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.model.sgraph.Vertex
import java.util.List
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.CheckType

import static com.yakindu.sct.model.sgraph.util.SGgraphUtil.areOrthogonal
import java.util.Set
import com.yakindu.sct.model.sgraph.RegularState

/**
 * 
 * All validation constraints for the meta model element {@link Synchronization}
 * 
 */
class SynchronizationValidator extends AbstractSGraphValidator {

	def dispatch private List<Transition> syncChain(Synchronization sync, List<Transition> transitionList, Set<Vertex> traversed) {
		transitionList += sync.outgoingTransitions.filter(t|!(t.target instanceof Synchronization) && !(t.target instanceof Choice))
		sync.outgoingTransitions.filter(t|(t.target instanceof Synchronization || t.target instanceof Choice) && t.target !== sync && !traversed.contains(sync)).forEach [
			traversed += sync
			target.syncChain(transitionList,traversed)
		]
		transitionList
	}
	
	def dispatch private List<Transition> syncChain(Choice choice, List<Transition> transitionList, Set<Vertex> traversed) {
		choice.outgoingTransitions.filter(t|t.target instanceof Synchronization).forEach [
			traversed += choice
			target.syncChain(transitionList,traversed)
		]
		transitionList
	}
	
	def dispatch private List<Transition> syncChain(Vertex vertex, List<Transition> transitionList, Set<Vertex> traversed) {
		transitionList
	}

	def private syncChain(Synchronization sync) {
		sync.syncChain(newArrayList,newHashSet)
	}

	public static val String SYNCHRONIZATION_MULTIPLE_TARGET_SAME_REGION_MSG = "Synchronization node(s) connected to this one result(s) in multiple states to be active in the same region at the same time. This would result in an inconsistent state.";
	public static val String SYNCHRONIZATION_MULTIPLE_TARGET_SAME_REGION_CODE = "synchronization.transitions.MultipleTargetSameRegion";

	@Check(CheckType.FAST)
	def checkSynchronizationHasTargetToSameRegion(Synchronization it) {
		if (!areOrthogonal(syncChain.map[target]) && areOrthogonal(outgoingTransitions.map[target]))
			error(SYNCHRONIZATION_MULTIPLE_TARGET_SAME_REGION_MSG, it, null, -1,
				SYNCHRONIZATION_MULTIPLE_TARGET_SAME_REGION_CODE);
	}	 
	
	
	public static val String SYNCHRONIZATION_TRANSITIONS_REQUIRE_N_OUT_MSG = "A synchronization must have at least one outgoing transition.";
	public static val String SYNCHRONIZATION_TRANSITIONS_REQUIRE_N_OUT_CODE = "synchronization.transitions.RequireNOut";

	@Check(CheckType.FAST)
	def checkSynchronizationTransitionsRequireNOut(Synchronization it) {
		if (outgoingTransitions.size == 0) {
			error(SYNCHRONIZATION_TRANSITIONS_REQUIRE_N_OUT_MSG, it, null, -1,
				SYNCHRONIZATION_TRANSITIONS_REQUIRE_N_OUT_CODE);
		}
	}

	public static val String SYNCHRONIZATION_TRANSITIONS_REQUIRE_MULTIPLE_IN_OR_MULTIPLE_OUT_MSG = "A synchronization must have either multiple incoming or multiple outgoing transitions.";
	public static val String SYNCHRONIZATION_TRANSITIONS_REQUIRE_MULTIPLE_IN_OR_MULTIPLE_OUT_CODE = "synchronization.transitions.RequireMultipleInOrMultipleOut";

	@Check(CheckType.FAST)
	def checkSynchronizationTransitionsRequireMultipleInOrMultipleOut(Synchronization it) {
		if (incomingTransitions.size < 2 && outgoingTransitions.size < 2) {
			error(SYNCHRONIZATION_TRANSITIONS_REQUIRE_MULTIPLE_IN_OR_MULTIPLE_OUT_MSG, it, null, -1,
				SYNCHRONIZATION_TRANSITIONS_REQUIRE_MULTIPLE_IN_OR_MULTIPLE_OUT_CODE);
		}
	}

	public static val String SYNCHRONIZATION_REQUIRES_ORTHOGONAL_SOURCE_STATES_MSG = "The source states of a synchronization must be orthogonal.";
	public static val String SYNCHRONIZATION_REQUIRES_ORTHOGONAL_SOURCE_STATES_CODE = "synchronization.RequiresOrthogonalSourceStates";

	@Check
	def checkSynchronizationRequiresOrthogonalSourceStates(Synchronization it) {
		if (!areOrthogonal(incomingTransitions.map[source])) {
			error(SYNCHRONIZATION_REQUIRES_ORTHOGONAL_SOURCE_STATES_MSG, it, null, -1,
				SYNCHRONIZATION_REQUIRES_ORTHOGONAL_SOURCE_STATES_CODE);
		}
	}

	public static val String SYNCHRONIZATION_REQUIRES_ORTHOGONAL_TARGET_STATES_MSG = "The target states of a synchronization must be orthogonal.";
	public static val String SYNCHRONIZATION_REQUIRES_ORTHOGONAL_TARGET_STATES_CODE = "synchronization.RequiresOrthogonalTargetStates";

	@Check
	def checkSynchronizationRequiresOrthogonalTargetStates(Synchronization it) {
		if (!areOrthogonal(outgoingTransitions.map[target])) {
			error(SYNCHRONIZATION_REQUIRES_ORTHOGONAL_TARGET_STATES_MSG, it, null, -1,
				SYNCHRONIZATION_REQUIRES_ORTHOGONAL_TARGET_STATES_CODE);
		}
	}

	public static val String SYNCHRONIZATION_OUTGOING_TRIGGER_MSG = "Triggers and guards on a synchronization's outgoing transitions will be ignored.";
	public static val String SYNCHRONIZATION_OUTGOING_TRIGGER_CODE = "synchronization.NoOutgoingTrigger";

	@Check(CheckType.FAST)
	def checkSyncNoTriggersOnOutgoingTransition(Synchronization sync) {
		sync.outgoingTransitions.filter(t|t.trigger !== null).forEach [
			warning(SYNCHRONIZATION_OUTGOING_TRIGGER_MSG, it, null, -1, SYNCHRONIZATION_OUTGOING_TRIGGER_CODE)
		]
	}
																
	public static final String SYNCHRONIZATION_TARGET_EXIT_MSG = "An exit node can not be the target of a synchronization node. Please target the exit node's destination directly.";
	public static final String SYNCHRONIZATION_TARGET_EXIT_CODE = "synchronization.InvalidTargetExitNode";

	@Check(CheckType.FAST)
	def checkSyncNoExitNodeAsOutgoingTransitionTarget(Synchronization sync) {
		sync.outgoingTransitions.filter(t|t.target instanceof Exit).forEach [
			error(SYNCHRONIZATION_TARGET_EXIT_MSG, it, null, -1, SYNCHRONIZATION_TARGET_EXIT_CODE);
		]
	}

	//Join: A synchronization's source states must be orthogonal to it's target states. (Common ancestor must be a region.)
	//Fork: A synchronization's target states must be exclusive to it's source states. (Common ancestor must be a region.)
	
	public static val String SYNCHRONIZATION_REQUIRES_TARGET_STATES_EXCLUSIVE_TO_SOURCE_STATES_MSG = "A synchronization's target states must be exclusive to it's source states. (Common ancestor must be a region.)";
	public static val String SYNCHRONIZATION_REQUIRES_TARGET_STATES_EXCLUSIVE_TO_SOURCE_STATES_CODE = "synchronization.RequiresTargetStatesExclusiveToSourceStates";
	
	 
	public static val String SYNCHRONIZATION_REQUIRES_SOURCE_STATES_ORTHOGONAL_TO_TARGET_STATES_MSG = "A synchronization's source states must be orthogonal to it's target states. (Common ancestor must be a region.)";
	public static val String SYNCHRONIZATION_REQUIRES_SOURCE_STATES_ORTHOGONAL_TO_TARGET_STATES_CODE = "synchronization.RequiresSourceStatesOrthogonalToTargetStates";

	@Check
	def checkSynchronizationRequiresSourceStatesOrthogonalToTargetStates(Synchronization it) {
		if(incomingTransitions.size === 1 && (incomingTransitions.head.source instanceof Synchronization)){
		} else {
			incomingTransitions.forEach(i | outgoingTransitions.forEach(j | 
			if(areOrthogonal(#[i.source,j.target])){
				if(it.incomingTransitions.size > 1){
					error(SYNCHRONIZATION_REQUIRES_SOURCE_STATES_ORTHOGONAL_TO_TARGET_STATES_MSG, i, null, -1,
						SYNCHRONIZATION_REQUIRES_SOURCE_STATES_ORTHOGONAL_TO_TARGET_STATES_CODE)
					error(SYNCHRONIZATION_REQUIRES_SOURCE_STATES_ORTHOGONAL_TO_TARGET_STATES_MSG, j, null, -1,
						SYNCHRONIZATION_REQUIRES_SOURCE_STATES_ORTHOGONAL_TO_TARGET_STATES_CODE);
				} else{
					error(SYNCHRONIZATION_REQUIRES_TARGET_STATES_EXCLUSIVE_TO_SOURCE_STATES_MSG, i, null, -1,
						SYNCHRONIZATION_REQUIRES_TARGET_STATES_EXCLUSIVE_TO_SOURCE_STATES_CODE)
					error(SYNCHRONIZATION_REQUIRES_TARGET_STATES_EXCLUSIVE_TO_SOURCE_STATES_MSG, j, null, -1,
						SYNCHRONIZATION_REQUIRES_TARGET_STATES_EXCLUSIVE_TO_SOURCE_STATES_CODE);
				}
				
			}
		))
		}
	}
	
	
	public static final String SYNCHRONIZATION_JOIN_INVALID_SOURCE_MSG = "All incoming transitions sources of a join (synchronization) must be regular states."
	public static final String SYNCHRONIZATION_JOIN_INVALID_SOURCE_CODE = "synchronization.JoinInvalidSourceNode"

	@Check(CheckType.FAST)
	def checkSyncJoinRegularStates(Synchronization sync) {
		if (sync.incomingTransitions.size > 1) {
			sync.incomingTransitions.filter(t| ! (t.source instanceof RegularState)).forEach [
			error(SYNCHRONIZATION_JOIN_INVALID_SOURCE_MSG, it, null, -1, SYNCHRONIZATION_JOIN_INVALID_SOURCE_CODE);
			]
		}
	}
	
}
