/**
 * Copyright (c) 2020-2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.domain.c.runtime.resource.transform.c

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.types.Parameter
import com.yakindu.base.types.Type
import com.yakindu.base.types.typesystem.GenericTypeSystem
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.domain.c.runtime.resource.ISimpleTypeMapper
import com.yakindu.sct.generator.c.typesystem.CTypeSystem
import java.util.Collection
import org.eclipse.cdt.core.dom.ast.IASTArrayDeclarator
import org.eclipse.cdt.core.dom.ast.IASTCompositeTypeSpecifier
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier
import org.eclipse.cdt.core.dom.ast.IASTDeclaration
import org.eclipse.cdt.core.dom.ast.IASTDeclarator
import org.eclipse.cdt.core.dom.ast.IASTEnumerationSpecifier
import org.eclipse.cdt.core.dom.ast.IASTNamedTypeSpecifier
import org.eclipse.cdt.core.dom.ast.IASTNode
import org.eclipse.cdt.core.dom.ast.IASTSimpleDeclSpecifier
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.EcoreUtil2

import static com.yakindu.sct.domain.c.runtime.resource.transform.c.CTypeNaming.ANONYMOUS_TYPE_SUFFIX
import static org.eclipse.cdt.core.dom.ast.IASTSimpleDeclSpecifier.*
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.base.types.typesystem.ITypeSemantics

/**
 * @author rbeckmann
 * 
 */
class CTypeUtils {
	@Inject
	protected ITypeSystem ts
	
	@Inject 
	protected extension ITypeSemantics
	
	@Inject
	protected extension ITypeSystemInferrer
	
	@Inject
	protected extension SExecExtensions

	@Inject
	protected ISimpleTypeMapper mapper
	
	//As The AST does not interprete the smart pointers correctly as pointers we match those by names
	public static val String SHARED_POINTER = "shared_ptr"
	public static val String WEAK_POINTER = "weak_ptr"
	public static val String UNIQUE_POINTER = "unique_ptr"
	public static val SMART_POINTER_TYPES = #[SHARED_POINTER,WEAK_POINTER,UNIQUE_POINTER]


	def isValueOnPointer(FeatureCall call) {
		return CTypeSystem.POINTER_TYPES.exists[ pt |
			ts.isSame(call.owner.infer?.type, ts.getType(pt)) &&
			call.definition.name == CTypeSystem.POINTER_VALUE_PROPERTY
		]
	}

	def isBuiltInType(IASTDeclSpecifier specifier) {
		specifier.getBuiltInType() !== null
	}

	def dispatch getBuiltInType(IASTSimpleDeclSpecifier specifier) {
		return mapper.map(specifier)
	}

	def dispatch getBuiltInType(IASTNamedTypeSpecifier specifier) {
		var type = ts.getType(specifier.name.toString)
		if (type !== null)
			return type

		return mapper.map(specifier)
	}

	def dispatch getBuiltInType(IASTDeclSpecifier declSpecifier) {
		null
	}

	def dispatch isChar(IASTSimpleDeclSpecifier specifier) {
		return specifier.getType() == t_char
	}

	def dispatch isChar(IASTDeclSpecifier specifier) {
		return false
	}

	def isPointer(IASTDeclarator declarator) {
		return !declarator.getPointerOperators.isNullOrEmpty
	}
	
	def isSmartPointer(IASTNamedTypeSpecifier declarator) {
		return false
	}
	
	def dispatch isArray(IASTDeclarator declarator) {
		false
	}
	
	def dispatch isArray(IASTArrayDeclarator declarator) {
		true
	}
	
	def isTypedef(IASTDeclSpecifier it) {
		storageClass == IASTDeclSpecifier.sc_typedef
	}
	
	def isStruct(IASTCompositeTypeSpecifier it) {
		key == IASTCompositeTypeSpecifier.k_struct
	}
	
	def isUnion(IASTCompositeTypeSpecifier it) {
		key == IASTCompositeTypeSpecifier.k_union
	}

	def dispatch boolean isVisible(IASTCompositeTypeSpecifier declSpecifier) {
		!declSpecifier.name.toString.empty
	}

	def dispatch boolean isVisible(IASTEnumerationSpecifier declSpecifier) {
		!declSpecifier.name.toString.empty
	}

	def dispatch boolean isVisible(IASTDeclaration declaration) {
		true
	}

	def getArrayType() {
		return ts.getType(ITypeSystem.ARRAY);
	}

	def getPointerType() {
		return ts.getType(CTypeSystem.POINTER);
	}

	def getUnsupportedType() {
		// TODO: add warning for unsupported type as diagnostic
		ts.getType(CTypeSystem.UNSUPPORTED_TYPE)
	}

	def boolean isVoid(Parameter parameter) {
		return ts.isSame(ts.getType(GenericTypeSystem.VOID), parameter.type)
	}
	
	def boolean isAnonymous(Type type) {
		!type.eIsProxy && type.name.endsWith(ANONYMOUS_TYPE_SUFFIX)
	}
	
	def protected isUsed(EObject elem, Collection<?> roots) {
		val references = EcoreUtil.UsageCrossReferencer.find(elem, roots)
			.map[EObject]
			.filter[e | !EcoreUtil2.getAllContainers(e).exists[it === elem]]
		references.size > 0
	}

	/**
	 * Finds the next parent of type {@link IASTDeclSpecifier}.
	 */
	def IASTDeclSpecifier findParentDeclSpecifier(IASTNode node) {
		if (node.parent === null) {
			return null
		}
		if (node.parent instanceof IASTDeclSpecifier) {
			return node.parent as IASTDeclSpecifier
		} else {
			return node.parent.findParentDeclSpecifier
		}
	}
	
	def isArrayType(Type valueType) {
		return isBuiltInType(valueType) && CTypeSystem.ARRAY.equals(valueType.name)
	}
	
	def isPointerType(Type valueType) {
		return isBuiltInType(valueType) && CTypeSystem.POINTER_TYPES.contains(valueType.name)
	}
	
	def isSmartPointerType(Type valueType) {
		return isBuiltInType(valueType) && CTypeUtils.SMART_POINTER_TYPES.contains(valueType.name)
	}
}
