/**
 * Copyright (c) 2022-2023 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 *
 * Contributors:
 * 	andreas muelder - itemis AG
 *  axel terfloth - itemis AG
 *
 */
package com.yakindu.sct.domain.java.inferrer

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.BoolLiteral
import com.yakindu.base.expressions.expressions.DoubleLiteral
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.FloatLiteral
import com.yakindu.base.expressions.expressions.HexLiteral
import com.yakindu.base.expressions.expressions.IntLiteral
import com.yakindu.base.expressions.expressions.StringLiteral
import com.yakindu.base.types.Argument
import com.yakindu.base.types.Event
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Property
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.domain.java.typesystem.JavaTypeSystem

import static com.yakindu.base.types.typesystem.ITypeSystem.*
import com.yakindu.sct.model.stext.inferrer.STextTypeInferrer

class JavaTypeSystemInferrer extends STextTypeInferrer {
	
	@Inject extension ITypeSystem
	
	override doInfer(IntLiteral literal) {
		getResultFor(ITypeSystem.INTEGER)
	}
	
	override doInfer(StringLiteral literal) {
		getResultFor(JavaTypeSystem.STRING)
	}
	
	override doInfer(HexLiteral literal) {
		getResultFor(JavaTypeSystem.INTEGER)
	}
	
	override doInfer(DoubleLiteral literal) {
		getResultFor(JavaTypeSystem.DOUBLE)
	}
	
	override doInfer(FloatLiteral literal) {
		getResultFor(JavaTypeSystem.FLOAT)
	}
	
	override doInfer(BoolLiteral literal) {
		getResultFor(JavaTypeSystem.BOOLEAN)
	}
	
	def dispatch inferByContext(IntLiteral lit, Expression exp) {
		getResultFor(ANY)	
	}
	
	def dispatch InferenceResult inferByContext(IntLiteral e, Argument argument) {
		argument.inferByContext(argument.eContainer).assertedIntegerType("Argument")
	}

	def dispatch InferenceResult inferByContext(IntLiteral e, EventRaisingExpression context) {
		val event = context.event.featureOrReference
		if (event instanceof Event) {
			return InferenceResult.from(event.type).assertedIntegerType("Event payload")
		}
		getResultFor(ANY)
	}

	def dispatch InferenceResult inferByContext(IntLiteral e, AssignmentExpression context) {
		val variable = context.varRef.featureOrReference
		if (variable !== null && variable instanceof Property) {
			return variable.inferTypeDispatch.assertedIntegerType("Assignment target")
		}
		getResultFor(ANY)
	}

	def dispatch InferenceResult inferByContext(IntLiteral e, Property context) {
		if (context.typeSpecifier !== null) inferTypeDispatch(context)
		else getResultFor(INTEGER)
	}
	
	def protected assertedIntegerType(InferenceResult it, String subject) {
		if(it !== null){
			if (! (it.type.isInteger)) {
				assertNotType(it, subject + " is no integer type.")
				return getResultFor(ANY)
			}
			return it	
		}
		return getResultFor(ANY)
	}
	
	
	override assertAssignable(InferenceResult varResult, InferenceResult valueResult, String msg) {
		val integerType = ITypeSystem.INTEGER.type
		
		if (valueResult.type.isSame(integerType)) {
			// either we assign to a sub type of integer
			if (varResult.type.isSuperType(integerType))  {
				return
			}
			// or we can covert to a subtype of integer, e.g. Byte, Short etc.
			if (varResult.type.isConvertableTo(integerType)) {
				return
			}
		}
		super.assertAssignable(varResult, valueResult, msg)
	}
	
	
}