/**
* Copyright (c) 2022 itemis AG - All rights Reserved
* Unauthorized copying of this file, via any medium is strictly prohibited
* 
* Contributors:
* 	axel terfloth - itemis AG
*
*/
package com.yakindu.sct.simulation.core.sexec.interpreter

import com.google.common.collect.Maps
import com.google.inject.Inject
import com.yakindu.base.expressions.interpreter.base.InterpreterException
import com.yakindu.base.types.validation.IValidationIssueAcceptor.ListBasedValidationIssueAcceptor
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.transformation.IModelSequencer
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.util.SubchartDFS
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ReferenceSlot
import com.yakindu.sct.model.stext.stext.VariableDefinition
import java.util.Map
import java.util.Stack
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.naming.IQualifiedNameProvider
import com.yakindu.base.expressions.interpreter.base.IInterpreter

/**
 * Injects new statechart instances for the statechart references specified by a statechart type.
 * 
 * @author axel terfloth - Initial contribution and API
 * 
 */
class MultiStatechartInjector {
	
	@Inject protected IInterpreter interpreter
	@Inject protected extension StatechartRuntimeExtension 
	@Inject protected extension IModelSequencer sequencer
	@Inject protected IQualifiedNameProvider fqnProvider;
	
	
	def dispatch void inject(Object it) {
		throw new InterpreterException("Cannot inject :" + it)
	}
	
	def dispatch void inject(ExecutionContext it) {
		it.injectByType(it.type) 
	}
	
	def protected dispatch void injectByType(ExecutionContext it, Object type) {
		throw new InterpreterException("Cannot inject " + it + " using type " + type)		
	}

	def protected dispatch void injectByType(ExecutionContext it, ExecutionFlow type) {
		injectByType(type.sourceElement)
	}
	
	def protected dispatch void injectByType(ExecutionContext it, Statechart type) {
		val dfs = new SubchartDFS() {
			
			Map<VariableDefinition, ExecutionContext> contextMap = Maps.newHashMap();
			Stack<String> fqnStack = new Stack<String>()
			ExecutionContext root = it
			
			override void beginVisit(Object element, Object parent, int depth) {
				if (element instanceof VariableDefinition) {
					fqnStack.add(element.name)
					val Statechart subchart = getStatechart(element);
					
					if (!EcoreUtil.equals(subchart, type)) {
						try {
							val acceptor = new ListBasedValidationIssueAcceptor()
							val flow = sequencer.transform(subchart, acceptor)

							// TODO: handle validation result : issue #159
//							if (acceptor.getTraces(Severity.ERROR).size() > 0) {
//								Status errorStatus = new Status(Status.ERROR, SimulationCoreActivator.PLUGIN_ID,
//										ERROR_DURING_SIMULATION, acceptor.getTraces(Severity.ERROR).iterator().next().toString(), null);
//								IStatusHandler statusHandler = DebugPlugin.getDefault().getStatusHandler(errorStatus);
//								try {
//									statusHandler.handleStatus(errorStatus, getDebugTarget());
//								} catch (CoreException e) {
//									e.printStackTrace();
//								}
//							}

							val instance = flow.newInstance => [ init ]
							
							var ExecutionContext parentContext = null;
							if (parent instanceof VariableDefinition) {
								parentContext = contextMap.get(parent);
							} else {
								parentContext = root;
							}
							var ExecutionSlot machineSlot = parentContext.getSlot(fqnProvider.getFullyQualifiedName(element).toString())
							
							if ( machineSlot instanceof ReferenceSlot ) {
																
								instance.name = machineSlot.name
								instance.fqName = machineSlot.fqName
								instance.contextFqn = fqn()

								machineSlot.set(instance)
															
								contextMap.put(element, instance)
							}
						} catch (RuntimeException e) {
							e.printStackTrace();
						}
					}
				}
				
				
			}
						
			override void endVisit(Object element, Object parent, int depth, int minDepth) {
				if (element instanceof VariableDefinition) {
					fqnStack.pop();
				}
			}

			def String fqn() {
				return String.join(".", fqnStack);
			}
			
		}
		dfs.perform(type)
	}
	
	
}