/**
 * Copyright (c) 2020-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.domain.c.runtime.resource.transform.c

import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.interpreter.context.PropertyInitialValueAdapter
import com.yakindu.base.expressions.interpreter.context.PropertyInitialValueAdapter.LazyInitialValue
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.GenericElement
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeParameter
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.TypesFactory
import com.yakindu.base.types.typesystem.GenericTypeSystem
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.commons.PathHelper
import com.yakindu.sct.domain.c.runtime.resource.CFileLocationAdapter
import com.yakindu.sct.domain.c.runtime.resource.CHeaderResource
import com.yakindu.sct.domain.c.runtime.resource.CachingPathResolver
import com.yakindu.sct.domain.c.runtime.resource.IMacroDefinitionParser
import com.yakindu.sct.domain.c.runtime.resource.ISimpleTypeMapper
import com.yakindu.sct.domain.c.runtime.resource.transform.ICToModelTrafo
import com.yakindu.sct.domain.c.runtime.resource.transform.PackageHierarchy
import com.yakindu.sct.domain.c.runtime.resource.transform.ProxyBuilder
import com.yakindu.sct.domain.c.runtime.resource.transform.TypeScope
import com.yakindu.sct.domain.c.runtime.resource.transform.types.CTypeHandlerChain
import com.yakindu.sct.generator.c.types.CTypeAnnotations
import com.yakindu.sct.generator.c.typesystem.CTypeSystem
import java.util.ArrayList
import java.util.HashSet
import java.util.List
import java.util.Map
import java.util.Optional
import org.eclipse.cdt.core.dom.ast.ASTVisitor
import org.eclipse.cdt.core.dom.ast.IASTArrayDeclarator
import org.eclipse.cdt.core.dom.ast.IASTArrayModifier
import org.eclipse.cdt.core.dom.ast.IASTBinaryExpression
import org.eclipse.cdt.core.dom.ast.IASTCompositeTypeSpecifier
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier
import org.eclipse.cdt.core.dom.ast.IASTDeclaration
import org.eclipse.cdt.core.dom.ast.IASTDeclarator
import org.eclipse.cdt.core.dom.ast.IASTElaboratedTypeSpecifier
import org.eclipse.cdt.core.dom.ast.IASTEnumerationSpecifier
import org.eclipse.cdt.core.dom.ast.IASTExpression
import org.eclipse.cdt.core.dom.ast.IASTLiteralExpression
import org.eclipse.cdt.core.dom.ast.IASTNamedTypeSpecifier
import org.eclipse.cdt.core.dom.ast.IASTNode
import org.eclipse.cdt.core.dom.ast.IASTParameterDeclaration
import org.eclipse.cdt.core.dom.ast.IASTPreprocessorFunctionStyleMacroDefinition
import org.eclipse.cdt.core.dom.ast.IASTPreprocessorMacroDefinition
import org.eclipse.cdt.core.dom.ast.IASTPreprocessorStatement
import org.eclipse.cdt.core.dom.ast.IASTSimpleDeclaration
import org.eclipse.cdt.core.dom.ast.IASTStandardFunctionDeclarator
import org.eclipse.cdt.core.dom.ast.IASTTranslationUnit
import org.eclipse.cdt.core.dom.ast.IASTUnaryExpression
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTLinkageSpecification
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateId
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTypeId
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil

import static com.yakindu.sct.domain.c.runtime.resource.transform.c.CTypeNaming.FQN_SEP
import static com.yakindu.sct.generator.c.typesystem.CTypeSystem.*

/**
 * @author rbeckmann
 * @author axel terfloth
 * 
 */
class CToModelTrafo implements ICToModelTrafo {
	
	protected extension TypesFactory factory = TypesFactory.eINSTANCE

	@Inject protected ITypeSystem ts

	@Inject protected extension CTypeNaming

	@Inject protected extension CTypeUtils

	@Inject protected IMacroDefinitionParser parser

	@Inject protected extension ProxyBuilder

	@Inject protected extension CTypeHandlerChain typeHandler
	
	@Inject protected extension PathHelper
	
	@Inject protected extension CachingPathResolver
	
	@Inject protected extension ISimpleTypeMapper
	
	@Inject protected extension CTypeAnnotations
	
	@Inject protected extension ExpressionBuilder

	protected TypeScope typeScope
	
	protected extension CHeaderResource resource
	
	protected IASTTranslationUnit ast
		
	HashSet<IASTSimpleDeclaration> declarationsWithUnresolvedTypes = newHashSet

	@Inject
	new() {
		this.typeScope = new TypeScope();
	}

	/**
	 * The ASTVisitor visits all declarations in the AST.
	 * Declarations introduce (or re-introduce) names into the C/C++ program.
	 */
	override transformToModel(CHeaderResource resource, Map<?, ?> options) {
		typeScope.clear()
		typeHandler.setup(typeScope, resource)
		
		this.resource = resource
		var visitor = createASTVisitor()
		ast = resource.ast
		ast.accept(visitor)

		ast.allPreprocessorStatements.filter[isFromThisResource].forEach[
			addToPackage(transformPreprocessorStatement({if (options !== null) options else newHashMap}))
		]
		declarationsWithUnresolvedTypes.complete()
		
		this.resource.unusedAnonymousTypes.forEach[t | EcoreUtil.delete(t)]
	}
		
	def void complete(HashSet<IASTSimpleDeclaration> declarations) {
		// Try to complete types of forward declared typedefs
		declarations.forEach[transform]
	}
		
	def protected createASTVisitor() {
		val visitor = new ASTVisitor(false) {

			override visit(IASTDeclaration declaration) {

				/* when included headers are not yet indexed, their content gets inserted into the same AST, 
				 * but we do not want to transform included types multiple times
				 */
				if (declaration.isFromThisResource) {
					declaration.transform
				}

				/*
				 * represents a linkage specification. e.g. extern "C" { ... }, hence we need to look into the 
				 * contained declarations
				 */
				if (declaration instanceof ICPPASTLinkageSpecification) {
					return PROCESS_CONTINUE
				}
				return PROCESS_SKIP
			}
			
		}
		visitor.shouldVisitDeclarations = true
		return visitor
	}
	
	def protected isFromThisResource(IASTNode node) {
		return pathsEqual(node.fileLocation.fileName.resolve, resourceLocation)
	}
	
	override void setup() {
		typeHandler.setup(typeScope, resource)
		typeScope.clear()
	}
	
	def dispatch protected void transform(IASTNode node) {
	}

	/**
	 * A simple declaration is a statement that introduces, creates, 
	 * and optionally initializes one or several identifiers, typically variables.
	 * 
	 * Each declarator introduces exactly one object, reference, function, 
	 * or (for typedef declarations) type alias, whose type is provided by decl-specifier-seq 
	 * and optionally modified by operators such as 
	 * & (reference to) or 
	 * [] (array of) or 
	 * () (function returning) 
	 * in the declarator. These operators can be applied recursively, as shown below.
	 */
	def dispatch protected void transform(IASTSimpleDeclaration declaration) {
		val declSpecifier = declaration.declSpecifier
		if (!declSpecifier.typeIsKnown || declaration.declarators.isEmpty) {
			declSpecifier.findOrCreateType
			if (declSpecifier instanceof IASTCompositeTypeSpecifier) {
				completeComplexType(declSpecifier)
			}
		}

		declaration.declarators.forEach [ declarator |
			transformStep(declaration.declSpecifier, declarator)?.addToModel(declaration.declSpecifier)
		]
	}

	def protected dispatch EObject transformStep(IASTDeclSpecifier declSpecifier,
		IASTStandardFunctionDeclarator declarator) {
		// "static is used with global variables and functions to set their scope to the containing file"
		// => has no effect on our transformation
		val isStatic = false
		toOperation(declSpecifier, declarator, isStatic)
	}

	def protected dispatch EObject transformStep(IASTDeclSpecifier declSpecifier, IASTDeclarator declarator) {
		if (declSpecifier.storageClass == IASTDeclSpecifier.sc_typedef) {
			val type = declSpecifier.findType
			if (type.isAnonymous && !declarator.isPointer && !declarator.isArray) {
				type.createNamedCopy(declSpecifier, declarator)
			} else {
				toTypeAlias(declSpecifier, declarator)
			}
		} else {
			toProperty(declSpecifier, declarator, false)
		}
	}
	
	def protected EObject createNamedCopy(Type type, IASTDeclSpecifier declSpecifier, IASTDeclarator declarator) {
		val copy = EcoreUtil.copy(type)
		copy.name = declarator.name.toString
		copy.visible = true
		copy.eAllContents.filter(NamedElement).forEach[
			name = name.replace(type.name + FQN_SEP, copy.name + FQN_SEP)
		]
		copy.eAdapters.add(new CFileLocationAdapter(declSpecifier.fileLocation))
		typeScope.put(declarator.qualifiedName, copy)
		copy
	}
	
	def protected dispatch void addToModel(TypeParameter result, IASTDeclSpecifier declSpecifier) {
		val type = declSpecifier.findOrCreateType
		if(type instanceof GenericElement) {
			type.typeParameters += result
		}
	}
	
	def protected dispatch void addToModel(EObject result, IASTDeclSpecifier declSpecifier) {
		if(result instanceof Declaration) {
			if(result.isInPackage) {
				return
			}
		}
		if(declSpecifier.parent instanceof IASTDeclaration) {
			val declaration = declSpecifier.parent as IASTDeclaration
			if (declaration.isToplevelDeclaration && result instanceof Declaration) {
					val packagePath = new PackageHierarchy(declSpecifier)
					addToPackage(result as Declaration, packagePath)
					return
			}
			val parentDeclSpecifier = findParentDeclSpecifier(declaration)
			if (parentDeclSpecifier !== null && declaration.isVisible) {
				val type = findOrCreateType(parentDeclSpecifier)
				if (type instanceof ComplexType && result instanceof Declaration) {
					(type as ComplexType).features += result as Declaration
					return
				}
			}
		}
	}
	
	def protected boolean isInPackage(Declaration it) {
		val packageMembers = resource.packageMembers
		val packageMemberContents = packageMembers.map[eAllContents.toList].flatten.filter(Declaration)
		(packageMembers + packageMemberContents).toList.contains(it)
	}
	
	def protected isToplevelDeclaration(IASTDeclaration declaration) {
		if(declaration.parent instanceof IASTTranslationUnit) {
			return true
		}
		if(declaration.findParentDeclSpecifier === null) {
			return true
		}
		return false
	}

	def Type findOrCreateType(IASTDeclSpecifier declSpecifier) {
		val t = findType(declSpecifier)
		if(t instanceof ComplexType) {
			addToModel(t, declSpecifier)
		}
		
		if (t === unsupportedType) {
			declarationsWithUnresolvedTypes.add(declSpecifier.parent)
			errors += new UnresolvableTypeProblem(declSpecifier.name)
		}
		t
	}
	
	def dispatch private add(HashSet<IASTSimpleDeclaration> declarations, IASTNode node) {
		return false
	}
	
	def dispatch private add(HashSet<IASTSimpleDeclaration> declarations, IASTSimpleDeclaration node) {
		return declarations.add(node)
	}

	def protected boolean isCompositeMember(IASTDeclaration declaration) {
		return declaration.parent instanceof IASTDeclSpecifier || declaration.parent.parent instanceof IASTDeclSpecifier
	}

	def protected boolean typeIsKnown(IASTDeclSpecifier declSpecifier) {
		if (declSpecifier.isBuiltInType) {
			return true
		}
		var type = declSpecifier.typeLookup
		return (type !== null && type != unsupportedType)
	}

	def protected dispatch Type typeLookup(IASTDeclSpecifier declSpecifier) {
		System.err.println("Unknown type specifier for lookup " + declSpecifier + " " + declSpecifier.class)
		unsupportedType
	}

	/** Always returns null because Composite can't be in typeScope */
	def protected dispatch Type typeLookup(IASTCompositeTypeSpecifier declSpecifier) {
		fromTypeScope(declSpecifier.qualifiedName)
	}

	/** Always returns null because Enumeration can't be in typeScope */
	def protected dispatch Type typeLookup(IASTEnumerationSpecifier declSpecifier) {
		null
	}

	/**
	 * Represents the use of a typedef name in a decl specifier in C.
	 */
	def protected dispatch Type typeLookup(IASTNamedTypeSpecifier declSpecifier) {
		fromTypeScope(declSpecifier.qualifiedName)
	}

	/**
	 * An elaborated type specifier is a type name preceded by either the class, struct, enum, or union keyword.
	 */
	def protected dispatch Type typeLookup(IASTElaboratedTypeSpecifier declSpecifier) {
		fromTypeScope(declSpecifier.qualifiedName)
	}

	def protected fromTypeScope(String typeName) {
		val type = typeScope.get(typeName)
		if (type !== null)
			return type
		return unsupportedType
	}

	def protected void transformCompositeMembers(IASTCompositeTypeSpecifier declSpecifier) {
		declSpecifier.members.forEach[transform]
	}
	
	def protected void completeComplexType(IASTCompositeTypeSpecifier declSpecifier) {
		declSpecifier.transformCompositeMembers
	}
	/**
	 * Computes a type based on IASTDeclSpecifier and IASTDeclarator which defines whether it is an array or pointer type. 
	 * Currently, an UnsupportedType is created for pointers.
	 * 
	 * @param declSpecifier
	 * 			used to decide the underlying type
	 * @param declarator
	 * 			used to decide whether it is an array or a pointer.
	 */
	def protected computeType(IASTDeclSpecifier declSpecifier, IASTDeclarator declarator) {
		if (declSpecifier.isChar && (declarator.isPointer || (declarator.isArray && (declarator as IASTArrayDeclarator).requiresToStringConversion))) {
			return ts.getType(GenericTypeSystem.STRING)
		}
		//if((declSpecifier instanceof IASTNamedTypeSpecifier) && (declSpecifier as IASTNamedTypeSpecifier).isSmartPointer){
		//	return (declSpecifier as IASTNamedTypeSpecifier).map
		//}
		// create new type or use already created type in this resource
		return declSpecifier.findOrCreateType
	}
	
	def protected createArraySizeExpression(IASTArrayDeclarator decl){
		decl.arrayModifiers.head.createExpression
	}
	
	def protected requiresToStringConversion(IASTArrayDeclarator decl){
		val arraySize = decl.arrayModifiers.head.constantExpression.toArraySize
		
		return arraySize.present && arraySize.get > 0
	}
	
	
	
	def protected createExpression(IASTArrayModifier mod){
		val value = mod.constantExpression.toArraySize
		if(value.empty){
			return 0._integer
		} else
		return value.get._integer
	}

	def protected create createEnumerator enumLiteral(IASTEnumerationSpecifier.IASTEnumerator enumerator,
		Type owningEnumType) {
		name = enumerator.name.toString
		typeSpecifier = createTypeSpecifier => [
			type = owningEnumType
		]
		it.eAdapters.add(new CFileLocationAdapter(enumerator.fileLocation))
	}

	def protected arraySizes(IASTArrayDeclarator declarator) {
		val sizes = new ArrayList<Integer>()
		declarator.arrayModifiers.forEach [
			sizes += constantExpression.toArraySize.orElse(0)
		]
		return sizes
	}

	def protected dispatch Optional<Integer> toArraySize(Void expression) {
		Optional.empty
	}

	def protected dispatch Optional<Integer> toArraySize(IASTExpression expression) {
		Optional.empty
	}

	def protected dispatch Optional<Integer> toArraySize(IASTLiteralExpression expression) {
		try {
			return Optional.of(Integer.valueOf(String.valueOf(expression.value)))
		} catch (NumberFormatException e) {
			System.err.println("Could not derive array size from expression '" + expression.value.toString +
				"'.")
		}
		return Optional.empty
	}

	def protected dispatch Optional<Integer> toArraySize(IASTUnaryExpression it) {
		val operandValue = operand.toArraySize
		
		if (operandValue.empty) return operandValue
		
		switch (operator) {
			case IASTUnaryExpression.op_plus : operandValue
			case IASTUnaryExpression.op_minus : Optional.of(-1 * operandValue.get)
			default : Optional.empty
		}
	}

	def protected dispatch Optional<Integer> toArraySize(IASTBinaryExpression it) {
		val operand1Value = operand1.toArraySize
		val operand2Value = operand2.toArraySize
		
		if (operand1Value.empty || operand2Value.empty) return Optional.empty
		
		switch (operator) {
			case IASTBinaryExpression.op_plus : Optional.of(operand1Value.get + operand2Value.get)
			case IASTBinaryExpression.op_minus : Optional.of(operand1Value.get - operand2Value.get)
			case IASTBinaryExpression.op_multiply : Optional.of(operand1Value.get * operand2Value.get)
			case IASTBinaryExpression.op_divide : Optional.of(operand1Value.get / operand2Value.get)
			default : {
				Optional.empty	
			}
		}		
	}


	def protected Type toTypeAlias(IASTDeclSpecifier declSpecifier, IASTDeclarator declarator) {
		val typeAlias = createTypeAlias
		val typeSpecifier = computeTypeSpecifier(declSpecifier, declarator)
		if (typeSpecifier.type == unsupportedType) {
			// use any type to enable usage of forward declarations
			typeAlias.typeSpecifier = createTypeSpecifier => [type = ts.getType(CTypeSystem.ANY)]
		} else {
			typeAlias.typeSpecifier = typeSpecifier
		}
		typeAlias.name = declarator.name.toString
		typeAlias.eAdapters += new CFileLocationAdapter(declarator.fileLocation)
		typeScope.put(declarator.qualifiedName, typeAlias)
		return typeAlias
	}
	
	def protected removeFromTypeScope(Iterable<? extends Type> types) {
		types.forEach[typeScope.remove(it.name)]
	}

	def protected dispatch TypeSpecifier computeTypeSpecifier(IASTDeclSpecifier declSpecifier,
		IASTDeclarator declarator) {
		return createPointerOrSimpleTypeSpecifier(declSpecifier, declarator)
	}

	def protected dispatch TypeSpecifier computeTypeSpecifier(IASTDeclSpecifier declSpecifier,
		IASTArrayDeclarator declarator) {
		val sizes = declarator.arraySizes
		val elementTypeSpecifier = createPointerOrSimpleTypeSpecifier(declSpecifier, declarator)
		if(declSpecifier.isChar && !declarator.isPointer) return elementTypeSpecifier
		return createArrayTypeSpecifier(elementTypeSpecifier, sizes)
	}

	protected def TypeSpecifier createPointerOrSimpleTypeSpecifier(IASTDeclSpecifier declSpecifier,
		IASTDeclarator declarator) {
		// do not create pointer for *char but a pointer<string> for **char
		if (declarator.isPointer && (!declSpecifier.isChar || declarator.pointerOperators.size > 1)) {
			val ptrType = computeType(declSpecifier, declarator)
			val numberOfPtrs = if (declSpecifier.isChar)
					declarator.pointerOperators.size - 1
				else
					declarator.pointerOperators.size

			return createPointerTypeSpecifier(ptrType, numberOfPtrs)
		}
		if(declSpecifier instanceof IASTNamedTypeSpecifier && (declSpecifier as IASTNamedTypeSpecifier).isSmartPointer){
			val namedDeclSpecifier = declSpecifier as IASTNamedTypeSpecifier
			if (namedDeclSpecifier.name.lastName instanceof ICPPASTTemplateId) {
				return createPointerTypeSpecifier(namedDeclSpecifier.name.lastName as ICPPASTTemplateId)
			}
		}

		return buildTypeSpecifier(declSpecifier, declarator)
	}

	def protected buildTypeSpecifier(IASTDeclSpecifier declSpecifier, IASTDeclarator declarator) {
		return createTypeSpecifier => [
			it.type = computeType(declSpecifier, declarator)
		]
	}

	def protected TypeSpecifier createArrayTypeSpecifier(TypeSpecifier elementTypeSpecifier, List<Integer> sizes) {
		val result = createArrayTypeSpecifier => [
			type = getArrayType
			size = sizes.get(0)
			if (sizes.size == 1) {
				typeArguments += elementTypeSpecifier
			} else {
				typeArguments += createArrayTypeSpecifier(elementTypeSpecifier, sizes.subList(1, sizes.size))
			}
		]
		return result
	}

	def protected Operation create createOperation toOperation(IASTDeclSpecifier specifier,
		IASTStandardFunctionDeclarator declarator, boolean isStatic) {
		name = declarator.name.toString
		if(name.nullOrEmpty && declarator.nestedDeclarator !== null) {
			name = declarator.nestedDeclarator.name.toString
		}
		it.eAdapters.add(new CFileLocationAdapter(declarator.fileLocation))
		declarator.parameters.forEach[param|if(!param.parameter.isVoid) parameters += param.parameter]
		if (declarator.takesVarArgs) {
			parameters += variadicParameter()
		}
		val typeSpecifier = computeTypeSpecifier(specifier, declarator)
		it.typeSpecifier = typeSpecifier
		it.static = isStatic
	}

	def protected toProperty(IASTDeclSpecifier declSpecifier, IASTDeclarator declarator, boolean isStatic) {
		val property = createProperty
		property.const = declSpecifier.isConst
		property.static = isStatic
		property.name = declarator.name.toString
		property.eAdapters += new CFileLocationAdapter(declarator.fileLocation)
		property.typeSpecifier = computeTypeSpecifier(declSpecifier, declarator)
		if(declarator.isArray){
			val arrayDecl = declarator as IASTArrayDeclarator
			if(declSpecifier.isChar && arrayDecl.requiresToStringConversion && !declarator.isPointer) property.annotations += createCharArrayAnnotationType(arrayDecl.createArraySizeExpression)
		}
		property
	}

	def protected variadicParameter() {
		var it = createParameter
		typeSpecifier = createTypeSpecifier
		typeSpecifier.type = ts.getType(CTypeSystem.ANY)
		varArgs = true
		name = "..."
		it
	}

	def protected create createParameter parameter(IASTParameterDeclaration declaration) {
		typeSpecifier = computeTypeSpecifier(declaration.declSpecifier, declaration.declarator)
		name = declaration.declarator.name.toString
		it.eAdapters.add(new CFileLocationAdapter(declaration.fileLocation))
	}

	def protected TypeSpecifier createPointerTypeSpecifier(Type ptrType, int level) {
		return createArrayTypeSpecifier => [
			type = getPointerType
			size = 1
			if (level == 1) {
				typeArguments += createTypeSpecifier => [
					type = ptrType
				]
			} else {
				typeArguments += createPointerTypeSpecifier(ptrType, level - 1)
			}
		]
	}
	
	def protected TypeSpecifier createPointerTypeSpecifier(ICPPASTTemplateId template) {
		return createArrayTypeSpecifier => [
			switch (template) {
				case template.templateName.toString.equals("unique_ptr"): type = ts.getType(UNIQUE_POINTER)
				case template.templateName.toString.equals("weak_ptr"): type = ts.getType(WEAK_POINTER)
				case template.templateName.toString.equals("shared_ptr"): type = ts.getType(SHARED_POINTER)
			}
			size = 1
			template.templateArguments.filter(ICPPASTTypeId).forEach[ ta | 
				it.typeArguments += createTypeSpecifier => [
					type = ta.declSpecifier.findOrCreateType
			]]
			
		]
	}

	def protected dispatch Declaration transformPreprocessorStatement(IASTPreprocessorStatement statement,
		Map<?, ?> options) {
		// fall back
	}

	def protected dispatch Declaration transformPreprocessorStatement(IASTPreprocessorMacroDefinition macroDef,
		Map<?, ?> options) {
		val macroResult = if (options !== null && options.containsKey(PARSE_DEFINE_EXPRESSIONS) &&
				options.get(PARSE_DEFINE_EXPRESSIONS) == true)
				parser.parseMacroDefinition(macroDef.expansion)
			else {
				if (macroDef.expansion.isNullOrEmpty) {
					Pair.of(ts.getType(CTypeSystem.ANY), null)
				} else {
					Pair.of(ts.getType(CTypeSystem.ANY), LazyInitialValue.of(macroDef.expansion))
				}
			}
		val property = createProperty => [
			name = macroDef.name.toString
			typeSpecifier = createTypeSpecifier => [type = macroResult.key]
			const = !ts.isSame(macroResult.key, ts.getType(CTypeSystem.ANY))
			it.eAdapters += new CFileLocationAdapter(macroDef.fileLocation)
		]
		if (macroResult.value !== null) {
			property.eAdapters += new PropertyInitialValueAdapter(macroResult.value)
		}
		property
	}

	/** 
	 * The literal which specifies the variable argument list of a macro.
	 */ 
	public static final String C_MACRO_VA_ARGS_NAME = "..." // this is what the user specifies in the argument list of a macro
	
	/** 
	 * The name of the variable argument in the statechart model. 
	 * Here the name which is defined by the C preprocessor to refer to the variable argument list is used.
	 */
	public static final String C_MODEL_VA_ARGS_NAME = "__VA_ARGS__" // this is the name used by the C preprocessor to refer to the variable argument part
	
	def protected dispatch Declaration transformPreprocessorStatement(
		IASTPreprocessorFunctionStyleMacroDefinition macroDef, Map<?, ?> options) {
		val operation = createOperation => [
			name = macroDef.name.toString
			typeSpecifier = createTypeSpecifier => [type = ts.getType(CTypeSystem.ANY)]
			parameters += macroDef.parameters.map [ param |
				createParameter => [
					name = param.parameter
					if (name == C_MACRO_VA_ARGS_NAME) {
						name = C_MODEL_VA_ARGS_NAME
						varArgs = true
					}
					typeSpecifier = createTypeSpecifier => [type = ts.getType(CTypeSystem.ANY)]
				]
			]
			it.eAdapters += new CFileLocationAdapter(macroDef.fileLocation)
		]
		operation
	}
	
	def protected getUnusedAnonymousTypes(CHeaderResource res) {
		res.packageMembers.map[#[it] + eAllContents.toList].flatten
			.filter(Type)
			.filter[isAnonymous]
			.filter[!isUsed(it, res.packageMembers)]
			.toList
	}
	
}
