/**
 * Copyright (c) 2023 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.itemis.create.statechart.generator.csharp

import com.google.inject.Inject
import com.google.inject.Provider
import com.itemis.create.base.generator.core.artifacts.IContent
import com.itemis.create.base.generator.csharp.artifacts.CSharpFile
import com.itemis.create.base.generator.csharp.artifacts.CsharpITimed
import com.itemis.create.base.generator.csharp.artifacts.CsharpITimerService
import com.itemis.create.base.generator.csharp.artifacts.CsharpVirtualTimer
import com.itemis.create.base.generator.csharp.codemodel.CsharpCompilationUnit
import com.itemis.create.base.generator.csharp.codemodel.CsharpNamespace
import com.itemis.create.base.generator.csharp.codemodel.CsharpStatemachineLibrary
import com.itemis.create.statechart.generator.csharp.codemodel.CsharpStatemachineCodeModel
import com.itemis.create.statechart.generator.csharp.codemodel.CsharpStatemachineEvents
import com.itemis.create.statechart.generator.csharp.codemodel.CsharpStatemachineNaming
import com.itemis.create.statechart.generator.csharp.codemodel.CsharpTraceEvent
import com.itemis.create.statechart.generator.csharp.features.CsharpStatechartGenmodelEntries
import com.itemis.create.statechart.generator.csharp.transformation.ExecFlow2stmClassTransformation
import com.itemis.create.statechart.generator.csharp.transformation.SctTypes2CsharpTypesTransformation
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Package
import com.yakindu.sct.generator.core.IExecutionFlowGenerator
import com.yakindu.sct.generator.core.artifacts.IGenArtifactConfigurations
import com.yakindu.sct.generator.core.artifacts.IGenArtifactConfigurations.GenArtifactConfiguration
import com.yakindu.sct.generator.core.codemodel.StatemachineClass
import com.yakindu.sct.generator.core.extensions.CodeModelGeneratorExtension
import com.yakindu.sct.generator.core.filesystem.ISCTFileSystemAccess
import com.yakindu.sct.generator.core.library.ICoreLibraryHelper
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sexec.transformation.config.IFlowConfiguration
import com.yakindu.sct.model.sgen.GeneratorEntry
import org.eclipse.xtext.generator.IFileSystemAccess

import static com.yakindu.sct.generator.core.filesystem.ISCTFileSystemAccess.*

/**
 * @author laszlo kovacs - Initial contribution.
 */
class CsharpGenerator implements IExecutionFlowGenerator {

	@Inject protected extension StatemachineClass
	@Inject protected extension SExecExtensions
	@Inject protected extension CodeModelGeneratorExtension
	@Inject protected extension CsharpStatemachineNaming
	@Inject protected extension IFlowConfiguration
	@Inject IGenArtifactConfigurations locations
	@Inject CsharpITimed timedInterface
	@Inject CsharpITimerService timerServiceInterface
	@Inject CsharpVirtualTimer virtualTimer
	@Inject protected Provider<CSharpFile> fileProvider
	@Inject protected extension CsharpStatemachineLibrary
	@Inject protected extension CsharpStatechartGenmodelEntries
	@Inject protected Provider<ExecFlow2stmClassTransformation> exec2StmProvider
	@Inject protected Provider<SctTypes2CsharpTypesTransformation> sctTypes2CsharpTypesProvider
	@Inject protected extension CsharpNamespace
	@Inject protected extension CsharpTraceEvent
	@Inject protected extension CsharpCompilationUnit
	@Inject protected extension CsharpStatemachineEvents
	@Inject protected extension ICoreLibraryHelper
	

	@Inject protected extension CsharpStatemachineCodeModel

	override generate(ExecutionFlow flow, GeneratorEntry entry, ISCTFileSystemAccess fsa) {
		initGenerationArtifacts(flow, entry, locations)
		generateArtifacts(flow, entry, fsa, locations)
	}

	def initGenerationArtifacts(ExecutionFlow flow, GeneratorEntry entry, IGenArtifactConfigurations locations) {
		sctTypes2CsharpTypesProvider.get.transform(flow)
		flow.defineClass(entry)
		exec2StmProvider.get.transform(flow)
		flow.addNamespace(entry)
		if(entry.tracingGeneric){
			configureCompilationUnit(traceEventUnit, entry.getLibraryOutput)
			configureCompilationUnit(traceEventTypeUnit, entry.getLibraryOutput)
			flow.stateMachineClass.features += flow.traceEventFeature
			val traceEventForStm = flow.stateMachineClass.traceEventForStm
			flow.stateMachineClass.features += traceEventForStm
			flow.stateMachineClass.features += traceEventForStm.defineOutEventAccessors
		}
		configureCompilationUnit(flow.implementationClassCompilationUnit, TARGET_FOLDER_OUTPUT)
		configureCompilationUnit(statemachineLibrary.member.filter(Package).filter [
			eAllContents.filter(ComplexType).exists[it === statemachineInterface]
		].head as Package, entry.getLibraryOutput)

		if (isCycleBased || flow.timed || entry.createTimerService) {
			configureCompilationUnit(
				statemachineLibrary.member.filter(Package).filter [
					eAllContents.filter(ComplexType).exists[it === cycleBasedStatemachineInterface]
				].head as Package, entry.getLibraryOutput
			)
			val suffix = entry.getgeneratedFileExtension ? ".g.cs" : ".cs"
			locations.configure("ITimed" + suffix, entry.getLibraryOutput, timedInterface)
			locations.configure("ITimerService" + suffix, entry.getLibraryOutput, timerServiceInterface)
			locations.configure("VirtualTimer" + suffix, entry.getLibraryOutput, virtualTimer)
		}
		if (isEventDriven) {
			configureCompilationUnit(
				statemachineLibrary.member.filter(Package).filter [
					eAllContents.filter(ComplexType).exists[it === eventDrivenStatemachineInterface]
				].head as Package , entry.getLibraryOutput
			)
		}
	}
	
	def private addNamespace(ExecutionFlow flow, GeneratorEntry entry){
		if(entry.baseNamespace !== null){
			_csNamespace => [
				member += flow.implementationClass
				name = entry.baseNamespace
			]
		}
	}

	def void configureCompilationUnit(Package compilationUnit, String path) {
		fileProvider.get => [ file |
			file.compilationUnit = compilationUnit
			locations.configure(file.fileName, path, file)
		]
	}

	def generateArtifacts(ExecutionFlow flow, GeneratorEntry entry, IFileSystemAccess fsa,
		IGenArtifactConfigurations locations) {
		for (GenArtifactConfiguration a : locations.configurations) {
			if (!a.skip) {
				fsa.generateFile(a.getName, a.getOutputName, a.getArtifact.fileContent())
			}
		}
	}
	
	def protected getLibraryOutput(GeneratorEntry entry) {
		if (entry.libraryTargetFolderValue !== null) {
			LIBRARY_TARGET_FOLDER_OUTPUT
		} else {
			TARGET_FOLDER_OUTPUT
		}
	}

	def dispatch String fileContent(IContent it) {
		it.content()
	}

	def dispatch String fileContent(Object it) {
		""
	}

}
