/**
 * Copyright (c) 2022-2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */

package com.yakindu.yet.sct.debugger.interpreter

import com.google.inject.Inject
import com.yakindu.base.expressions.interpreter.base.IInterpreter
import com.yakindu.base.expressions.interpreter.base.ResolvingRuntimeInstance
import com.yakindu.base.expressions.interpreter.base.ValueSemantics
import com.yakindu.sct.model.sgraph.RegularState
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Vertex
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionEvent
import com.yakindu.sct.model.sruntime.ExecutionVariable
import com.yakindu.yet.core.YETEvent
import com.yakindu.yet.core.YETEvent.YETInitEvent
import com.yakindu.yet.core.YETEvent.YETUpdateEvent
import com.yakindu.yet.core.YETScope
import com.yakindu.yet.core.protocol.IYETMessageSender
import java.util.Optional
import org.eclipse.xtext.naming.IQualifiedNameProvider

import static com.yakindu.yet.core.format.YETConstants.*

/**
 * @author Axel Terfloth
 */
class YETStatechartInstance extends ResolvingRuntimeInstance {
	
	protected Statechart statechart
	
	@Inject protected IQualifiedNameProvider fqnProvider
	@Inject(optional=true) protected IYETMessageSender sender
	@Inject protected extension ValueSemantics
	
	
	protected YETScope traceScope;
	
	public static final char META_SEPARATOR = '@'
	
	new() {
		traceScope = new YETScope() {
						
			override executed(YETEvent<?> event) {
//				System.out.println(fullyQualifiedName + " executed: " + event);
				accept(event)
			}

		}	
	}
	
	def void setUp(CompositeSlot instance, IInterpreter.Context context, Statechart statechart) {
		super.setUp(instance, context)
		this.statechart = statechart
	}
	
	
	override invoke(Object slot, Object operation) {
		throw new UnsupportedOperationException("TODO: auto-generated method stub")
	}
	
	//=========================================================================================
	//
	// Setting properties
	//
	
	
	def dispatch set(ExecutionVariable slot, Object value) {
//		_execute('''set «slot.name»''', [
			val event = new YETUpdateEvent(currentTime, slot.fqName, if( value !== null ) Optional.of(value.toString) else Optional.empty)
			traceScope.stimulated(event)
			sender?.send(event)	
//		])	
	}
	
	def dispatch set(Object slot, Object value) {
	}
	


	//=========================================================================================
	//
	// Raising events
	//
	
	
	override raise(Object slot, Object value) {
		if (slot instanceof ExecutionEvent) {
			_execute('''raise «slot.name»''', [
				val event = new YETUpdateEvent(currentTime, slot.fqName, if( value !== null ) Optional.of(value.toString) else Optional.empty)
				traceScope.stimulated(event)
				sender?.send(event)	
			])	
		}	
	}

	
	override provideExecution(Object program) {
//		throw new UnsupportedOperationException("TODO: auto-generated method stub")
	}
		
	
	def dispatch void accept(YETInitEvent message) {
		val key = message.key.withoutScope
		val value = message.value
		if ( key.isMetaFeature ) {
			switch (key.feature) {
			case START_KEY: {
				onExecutionStart
			}

			}
		} else if ( key.isRegularFeature ) {
			onValueChanged(0, key, value)
		}
	}


	def dispatch void accept(YETUpdateEvent message) {
		val key = message.key.withoutScope
		val value = message.value
		val timestamp = message.timestamp
			

		
		if (isMetaFeature(key)) {
			switch (getFeature(key)) {
				case RUNCYCLE_START: {
					onRunCycleStarted(timestamp)
				}
				case RUNCYCLE_END: {
					onRunCycleEnded(timestamp)
				}
				case TIME_EVENT_FEATURE: {
					handleTimeEvent(timestamp, key)
				}
				case STATE_ENTER: {
					if (value.isPresent())
						onStateEntered(timestamp, value.get())
				}
				case STATE_EXIT: {
					if (value.isPresent())
						onStateExited(timestamp, value.get())
				}
			}
		} else if (isRegularFeature(key)) {
			val eventOrVariable = key.resolveYETFeature // instanceSlot.resolve(getFeature(key));
			if (eventOrVariable instanceof ExecutionVariable && value.isPresent()) {
				onValueChanged(timestamp, getFeature(key), value.get())
			} else if (eventOrVariable instanceof ExecutionEvent) {
				onEvent(timestamp, key, value)
			}
		}
	}
	
	def protected resolveYETFeature(String key) {
		key.split("\\.").filter[
			!empty
		].fold(instanceSlot as Object, [
			slot, f | slot.resolveSlot(f)
		])
	}
	
	
	def protected String withoutScope(String it) {
		val fqScopeName = traceScope.fullyQualifiedName
		var key = it
		if (! fqScopeName.isNullOrEmpty ) {
			if (it.startsWith(fqScopeName)) {
				key = it.substring(fqScopeName.length)
			}
			if (key.startsWith(".")) 
				key = key.substring(1)
		}
		return key
	}
	
	
	def protected boolean isMetaFeature(String key) {
		return key.charAt(0) === META_SEPARATOR
	}

	def protected boolean isRegularFeature(String key) {
		return !isMetaFeature(key);
	}

	def protected String getFeature(String s) {
		if (isMetaFeature(s)) {
			return s.substring(1);
		}
		return s;
	}
	
	
	
	def void onExecutionStart() {}

	
	def void onExecutionEnd(long timestamp) {}


	def void onEvent(long timestamp, String name, Optional<String> value) {
		val slot = name.resolveYETFeature // instanceSlot.resolve(name)
		if (slot instanceof ExecutionEvent) {
			val parsed = parseString(value, slot.getValue())
			internalOnEvent(slot, parsed)
		}
	}

	def protected void internalOnEvent(ExecutionEvent event, Optional<Object> value) {
		event.setRaised(true);
		if (value.isPresent()) {
			event.setValue(value.get());
		}
	}

	def void onStateEntered(long timestamp, String name) {
		val state = name.findState
		if (state !== null) {
			internalOnStateEntered(state)
		}
	}

	def protected void internalOnStateEntered(Vertex state) {
		executionContextSlot.getActiveStates().add(state);
	}

	def void onStateExited(long timestamp, String name) {
		val state = name.findState
		if (state !== null) {
			internalOnStateExited(state)
		}
	}

	def protected void internalOnStateExited(Vertex state) {
		executionContextSlot.getActiveStates().remove(state);
	}

	def void onRunCycleStarted(long timestamp) {
	}

	def void onRunCycleEnded(long timestamp) {
	}

	def void onValueChanged(long timestamp, String name, String newValue) {
//		val slot = resolver.findEventOrVariable(name, context);
		val slot = name.resolveYETFeature //instanceSlot.resolve(name)
		if (slot instanceof ExecutionVariable) {
			val value = parseString(newValue, slot.getValue())
			internalOnValueChanged(slot, value)
		}
	}

	def protected void internalOnValueChanged(ExecutionVariable it, Object value) {
		it.setValue(value);
	}

	def Vertex findState(String name) {		
		statechart
			.eAllContents
			.filter(RegularState)
			.filter[fqnProvider.getFullyQualifiedName(it).skipFirst(1).toString == name].head
	}

	def protected void handleTimeEvent(long timestamp, String key) {
		val String feature = getFeature(key);
		val parts = feature.split("\\.");
		val int eventId = Integer.valueOf(parts.get(1));
		val String event = parts.get(2);
		
		switch (event) {
			case TIME_EVENT_SET:
				onSetTimeEvent(timestamp, eventId)
			case TIME_EVENT_UNSET:
				onUnsetTimeEvent(timestamp, eventId)
			case TIME_EVENT_EXPIRE:
				onTimeEvent(timestamp, eventId)
		}

	}

	def  Object parseString(String value, Object targetType) {
		if (targetType instanceof String) {
			return value;
		}
		if (targetType instanceof Long) {
			return Long.valueOf(value);
		}
		if (targetType instanceof Integer) {
			return Integer.valueOf(value);
		}
		if (targetType instanceof Boolean) {
			return value.equals("t");
		}
		if (targetType instanceof Float) {
			return Float.valueOf(value);
		}
		if (targetType instanceof Double) {
			return Double.valueOf(value);
		}
		if (targetType instanceof Character) {
			return Character.valueOf(value.charAt(0));
		}
		return value;
	}

	def Optional<Object> parseString(Optional<String> value, Object targetType) {
		if (value.isPresent()) {
			return Optional.of(parseString(value.get(), targetType));
		} else {
			return Optional.empty();
		}
	}

	def void onClearInEvents(long timestamp) {
		internalOnClearInEvents
	}

	def protected void internalOnClearInEvents() {
//		contextExtensions.clearLocalAndInEvents(context);
	}

	def void onClearOutEvents(long timestamp) {
		internalOnClearOutEvents
	}

	def protected void internalOnClearOutEvents() {
//		contextExtensions.clearOutEvents(context);
	}

	def void onSetTimeEvent(long timestamp, int id) {
	}

	def void onUnsetTimeEvent(long timestamp, int id) {
	}

	def void onTimeEvent(long timestamp, int id) {
//		val timeEvent = resolver.findTimeEvent(id, context);
		val timeEvent = instanceSlot.resolve(timeEventName(id)) as ExecutionEvent
		internalOnTimeEvent(timeEvent)
	}

	def protected String timeEventName(int id) {
		return "" + id
	}

	def protected void internalOnTimeEvent(ExecutionEvent findTimeEvent) {
		findTimeEvent.setRaised(true);
	}

	
	
	def protected executionContextSlot(){
		return instanceSlot as ExecutionContext
	}
	
}