/**
 * Copyright (c) 2020-2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter.context

import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.TypeCastExpression
import com.yakindu.base.expressions.interpreter.IExpressionInterpreter
import com.yakindu.base.types.Event
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.TypedElement
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ReferenceSlot
import java.util.List
import java.util.Optional
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.naming.IQualifiedNameProvider
import org.eclipse.xtext.util.SimpleAttributeResolver

/**
 * Default implementation for resolving execution slots based on expressions.
 * 
 * @author Thomas Kutz
 * @author axel terfloth
 */
class DefaultExecutionSlotResolver implements IExecutionSlotResolver {

	@Inject protected extension IQualifiedNameProvider
	@Inject protected extension IExpressionInterpreter
//	@Inject protected extension BuiltInTypeContextInitializer
	
	@Inject protected ITypeSystem ts
	@Inject protected IExecutionContextInitializer slotInitializer
	

	def dispatch Optional<ExecutionSlot> resolve(ExecutionContext context, FeatureCall e) {
		return Optional.ofNullable(resolveByFeature(context, e, e.feature))
	}

	def dispatch Optional<ExecutionSlot> resolve(ExecutionContext context, ElementReferenceExpression e) {
		var slot = packageNamespaceAwareResolve(context, e.reference)
		if (e.arrayAccess) {
			slot = resolveArrayElement(context, slot, e.arraySelector, (e.reference as TypedElement).typeSpecifier)
		}
		return Optional.ofNullable(slot);
	}
	
	

	def dispatch Optional<ExecutionSlot> resolve(ExecutionContext context, AssignmentExpression e) {
		return context.resolve(e.varRef)
	}

	def dispatch Optional<ExecutionSlot> resolve(ExecutionContext context, TypeCastExpression e) {
		context.resolve(e.operand)
	}


	def protected dispatch ExecutionSlot resolveByFeature(ExecutionContext context, FeatureCall e, EObject feature) {
		return context.getVariable(e.feature.fullyQualifiedName.toString)
	}

	def protected dispatch ExecutionSlot resolveByFeature(ExecutionContext context, FeatureCall e, Operation feature) {
		return resolveCompositeSlot(context, e)
	}

	def protected dispatch ExecutionSlot resolveByFeature(ExecutionContext context, FeatureCall e, Event feature) {
		return resolveCompositeSlot(context, e)
	}

	def protected dispatch ExecutionSlot resolveByFeature(ExecutionContext context, FeatureCall e, Property feature) {
		return resolveCompositeSlot(context, e)
	}

	def protected ExecutionSlot resolveCompositeSlot(ExecutionContext context, FeatureCall e) {
		var featureSlot = resolve(context, e.owner)
		if (!featureSlot.isPresent) {
			return null
		}
		return resolveFromSlot(featureSlot.get, e)
	}

	def protected dispatch ExecutionSlot resolveFromSlot(ExecutionSlot slot, FeatureCall call) {
		slot // fallback
	}

	def protected dispatch ExecutionSlot resolveFromSlot(CompositeSlot slot, Operation call) {
		resolveByFeature(slot, call)
	}
	
	def protected dispatch ExecutionSlot resolveFromSlot(CompositeSlot slot, FeatureCall call) {
		var resolvedSlot = resolveByFeature(slot, call.feature)
		if (call.arrayAccess) {
			resolvedSlot = resolveArrayElement(EcoreUtil2.getContainerOfType(resolvedSlot, ExecutionContext),
				resolvedSlot, call.arraySelector, (call.feature as TypedElement).typeSpecifier)
		}
		return resolvedSlot
	}

	
	
	def protected dispatch ExecutionSlot resolveFromSlot(ReferenceSlot slot, Operation call) {
		if (slot.reference instanceof CompositeSlot) {
			resolveByFeature(slot.reference as CompositeSlot, call)
		} else {
			resolveByFeature(slot, call)
		}
	}
	
	def protected dispatch ExecutionSlot resolveFromSlot(ReferenceSlot slot, FeatureCall call) {
		if (slot.reference instanceof CompositeSlot) {
			resolveByFeature(slot.reference as CompositeSlot, call.feature)
		} else {
			resolveByFeature(slot, call.feature)
		}
	}

	def protected dispatch ExecutionSlot resolveByFeature(CompositeSlot slot, EObject feature) {
		slot // fallback
	}

	def protected dispatch ExecutionSlot resolveByFeature(CompositeSlot slot, Property feature) {
		if (ts.isExtensionProperty(feature)) {
			// this property comes from outside, so it is not reflected in execution context
			return slot
		}
		
		resolveByName(slot, feature)
	}

	def protected dispatch ExecutionSlot resolveByFeature(CompositeSlot slot, Operation feature) {
		resolveByName(slot, feature)
	}

	def protected dispatch ExecutionSlot resolveByFeature(CompositeSlot slot, Event feature) {
		resolveByName(slot, feature)
	}

	def protected dispatch ExecutionSlot resolveByName(ExecutionContext slot, NamedElement element) {
		val defaultSlot = slot.slots.filter(CompositeSlot).findFirst[it.name == "default"]
		slot.slots.findFirst[name == element.name] ?: if (defaultSlot !== null) resolveByName(defaultSlot, element)
	}

	def protected dispatch ExecutionSlot resolveByName(CompositeSlot slot, NamedElement element) {
		slot.slots.findFirst[name == element.name]
	}

	def protected ExecutionSlot packageNamespaceAwareResolve(ExecutionContext context, EObject element) {
		context.getSlot(element.fullyQualifiedName.toString)
	}

	def protected name(EObject e) {
		return SimpleAttributeResolver::NAME_RESOLVER.apply(e)
	}
	
	
	// #### HANDLING ARRAYS 	
	
	def protected dispatch ExecutionSlot resolveArrayElement(ExecutionContext context, ExecutionSlot slot,
		List<Expression> arraySelectors, TypeSpecifier specifier) {
		var arrayElementSlot = slot
		var sizes = arraySelectors.map[toIndex(context) + 1]
		createDynamicArraySlots(arrayElementSlot as CompositeSlot, specifier.typeArguments.head, sizes)

		for (int arrayIndex : arraySelectors.map[toIndex(context)]) {
			arrayElementSlot = (arrayElementSlot as CompositeSlot).slots.get(arrayIndex)
		}
		arrayElementSlot
	}

	def protected dispatch ExecutionSlot resolveArrayElement(ExecutionContext context, ReferenceSlot slot,
		List<Expression> arraySelectors, TypeSpecifier specifier) {
		if (slot.reference !== null) {
			return context.resolveArrayElement(slot.reference, arraySelectors, specifier)
		}
		return null
	}

	def protected void createDynamicArraySlots(CompositeSlot arrayElementSlot, TypeSpecifier elementTypeSpecifer,
		Iterable<Integer> sizes) {
		val currentSize = (arrayElementSlot as CompositeSlot).slots.size
		if (currentSize < sizes.head) {
			for (i : currentSize .. sizes.head - 1) {
				val newSlot = slotInitializer.newInstance(elementTypeSpecifer) 
				=> [
					name = '''[«i»]'''
					type = elementTypeSpecifer.type
				]
				(arrayElementSlot as CompositeSlot).slots += newSlot
				if (!sizes.drop(1).empty)
					createDynamicArraySlots(newSlot as CompositeSlot, elementTypeSpecifer.typeArguments.head,
						sizes.drop(1))
			}
		} else {
			val slot = arrayElementSlot.slots.get(sizes.head - 1)
			if (!sizes.drop(1).empty)
				createDynamicArraySlots(slot as CompositeSlot, elementTypeSpecifer.typeArguments.head, sizes.drop(1))
		}

	}

	def protected toIndex(Expression arraySelector, ExecutionContext context) {
		(arraySelector.evaluate(context) as Long).intValue
	}
	
}
