/**
 * Copyright (c)2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 */
package com.yakindu.sct.simulation.core.sexec.launch

import com.google.inject.Guice
import com.google.inject.Injector
import com.google.inject.Module
import com.yakindu.base.base.NamedElement
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.resource.ResourceUtil
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.simulation.core.debugmodel.SCTDebugTarget
import com.yakindu.sct.simulation.core.engine.ISimulationEngine
import org.eclipse.core.resources.IProject
import org.eclipse.core.resources.IResource
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.core.runtime.Assert
import org.eclipse.core.runtime.CoreException
import org.eclipse.core.runtime.IProgressMonitor
import org.eclipse.debug.core.ILaunch
import org.eclipse.debug.core.ILaunchConfiguration
import org.eclipse.debug.core.model.LaunchConfigurationDelegate
import org.eclipse.emf.ecore.resource.Resource

/**
 * Provides a basic abstract implementation of a launch configuration delegate that cares about launching multiple instances and statecharts. 
 * For each instanciated statechart a debug target will be created.
 * 
 * Concrete implementations must implement a set of abstract methods defined by this class in order to contribute application specifics.
 * 
 * @author axel terfloth
 */
abstract class BaseLaunchConfigurationDelegate extends LaunchConfigurationDelegate {

	public static final String FILE_NAME = "filename"
	public static final String DEFAULT_FILE_NAME = ""
	
	protected Injector injector
	
	/**
	 * The execution root element is the element that will be instantiated and 'started'. Concrete implementations must
	 * provide this element which must be a 'NamedElement' from the loaded resource.
	 */
	abstract def protected NamedElement loadExecutionRoot(Resource it)
	
	/** Concrete implementations must provide a valid simulation guice module. */
	abstract def protected Module provideSimulationModule()
	
	/** Concrete implementations must provide a simulation engine for the root element. */
	abstract def protected ISimulationEngine provideRootSimulationEngine(NamedElement execRoot)
	
	/** Concrete implementations must provide a simulation engine for each statechart. */
	abstract def protected ISimulationEngine provideStatechartSimulationEngine(Statechart statechart, ILaunch launch)
	
	
	override void launch(ILaunchConfiguration configuration, String mode, ILaunch launch,
		IProgressMonitor monitor) throws CoreException {
		
		var Module module = provideSimulationModule
		injector = Guice.createInjector(module)
		injector.injectMembers(this) 
		
		var String filename = configuration.getAttribute(FILE_NAME, DEFAULT_FILE_NAME)
		val resource = loadResource(filename)
		var executionRoot = resource.loadExecutionRoot
		var simulationEngine = createExecutionContainer(launch, executionRoot, monitor)
		val SCTDebugTarget target = new SCTDebugTarget(launch, executionRoot,simulationEngine)
		target.setPrimary(true)
		target.init()
		launch.addDebugTarget(target)
		addStatemachineDebugTargets(launch, target)
		target.start()
	}
	
	def protected Resource loadResource(String filename){
		return ResourceUtil.loadResource(filename)
	}
	
	def protected ISimulationEngine createExecutionContainer(ILaunch launch, NamedElement execRoot,
		IProgressMonitor monitor) {
		
		var ISimulationEngine engine = provideRootSimulationEngine(execRoot)
		injector.injectMembers(engine)
		return engine
	}
	
	def protected void addStatemachineDebugTargets(ILaunch launch, SCTDebugTarget mainTarget) {
		val context = (mainTarget.getAdapter(ExecutionContext) as ExecutionContext) 
		val statemachineInstances = context.eAllContents.filter(ExecutionContext).filter[type instanceof ExecutionFlow].toList
		
		statemachineInstances
			.forEach[
				try {
					val machineTarget = createSubmachineDebugTarget(launch, statechart, fqName);
					val engine = machineTarget.getAdapter(ISimulationEngine) as ISimulationEngine
					engine.executionContext = it
					machineTarget.init
					launch.addDebugTarget(machineTarget)					
				} catch (CoreException e) {
					e.printStackTrace();
				}
			]
	}
	
	def protected SCTDebugTarget createSubmachineDebugTarget(ILaunch launch, Statechart statechart, String context)
			throws CoreException {
		Assert.isNotNull(statechart)
		val engine = provideStatechartSimulationEngine(statechart, launch)
		return new SCTDebugTarget(launch, statechart, context, engine)
	}
	
	def protected Statechart getStatechart(ExecutionContext context) {
		val type = context.type
		if (type instanceof ExecutionFlow) {
			if (type.sourceElement instanceof Statechart) {
				return type.sourceElement as Statechart
			}
		}
		return null
	}
	
	override protected IProject[] getProjectsForProblemSearch(ILaunchConfiguration configuration,
		String mode) throws CoreException {
		var String filename = configuration.getAttribute(FILE_NAME, "")
		var IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(filename)
		return #[resource.getProject()]
	}
	
	override boolean buildForLaunch(ILaunchConfiguration configuration, String mode,
		IProgressMonitor monitor) throws CoreException {
		// Never build the workspace before simulation
		return false
	}
	
}
