/**
 * Copyright (c) 2016 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Thomas Kutz - itemis AG
 * 
 */
package com.yakindu.sct.generator.c.typesystem

import com.google.inject.Inject
import com.google.inject.Singleton
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.BoolLiteral
import com.yakindu.base.expressions.expressions.DoubleLiteral
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.FloatLiteral
import com.yakindu.base.expressions.expressions.IntLiteral
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Property
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.base.types.annotations.TypeAnnotations
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.stext.inferrer.STextTypeInferrer
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.model.stext.stext.VariableDefinition

import static com.yakindu.base.types.typesystem.ITypeSystem.INTEGER
import static com.yakindu.sct.generator.c.typesystem.CTypeSystem.BOOL
import static com.yakindu.sct.generator.c.typesystem.CTypeSystem.DOUBLE
import static com.yakindu.sct.generator.c.typesystem.CTypeSystem.FLOAT

@Singleton
class CTypeInferrer extends STextTypeInferrer {

	@Inject protected extension OriginTracing
	@Inject protected extension TypeAnnotations
	@Inject protected extension CTypeSystem ts

	override doInfer(IntLiteral literal) {
		return InferenceResult.from(ts.getType(INTEGER))
	}

	override doInfer(BoolLiteral literal) {
		return InferenceResult.from(ts.getType(BOOL))
	}

	override doInfer(FloatLiteral literal) {
		getResultFor(FLOAT)
	}

	override doInfer(DoubleLiteral literal) {
		getResultFor(DOUBLE)
	}
	
	override InferenceResult doInfer(VariableDefinition e) {
		if(e.type instanceof ComplexType && !e.type.hasBuiltInTypeAnnotation){
			val inferred = inferTypeDispatch(e.getTypeSpecifier(false))
			if(inferred !== null)
				return inferred
		}
		super.doInfer(e)
	}

	override assertAssignable(InferenceResult varResult, InferenceResult valueResult, String msg) {
		if (ts.isSame(valueResult?.type, ts.getType(ITypeSystem.INTEGER)) &&
			ts.isSuperType(varResult.type, ts.getType(ITypeSystem.INTEGER))) {
			return;
		}
		if (ts.isSame(valueResult?.type, ts.getType(ITypeSystem.BOOLEAN)) &&
			ts.isSuperType(varResult.type, ts.getType(ITypeSystem.BOOLEAN))) {
			return;
		}
		if (ts.isSame(varResult?.type, ts.getType(ITypeSystem.ANY))) {
			// allow assignments to any-typed parameters or variables
			return;
		}
		if (valueResult?.type.origin instanceof Statechart &&
			ts.isSame(InferenceResult.from(getResultFor(CTypeSystem.POINTER).type, newArrayList(doInfer(valueResult?.type)))?.type, varResult?.type)
		) {
			super.assertAssignable(varResult, InferenceResult.from(getResultFor(CTypeSystem.POINTER).type, newArrayList(doInfer(valueResult?.type))), msg);
		} else {
			super.assertAssignable(varResult, valueResult, msg);
		}
	}

	override doInfer(ElementReferenceExpression e) {
		val inferred = e.reference.inferTypeDispatch
		if (inferred === null)
			return super.doInfer(e)
		if (e.isArrayAccess && CTypeSystem.POINTER_TYPES.exists[pt | pt == inferred?.type.name]) {
			return getResult(inferred, 1)
		}
		return super.doInfer(e)
	}

	override doInfer(FeatureCall e) {
		val inferred = e.feature.inferTypeDispatch
		if (inferred === null)
			return super.doInfer(e)
		else if (e.isArrayAccess && CTypeSystem.POINTER_TYPES.exists[pt | pt == inferred?.type.name]) {
			return getResult(inferred, 1)
		} // TODO: For all extension operations && properties
		else if (e.feature instanceof Property && CTypeSystem.POINTER_PROPERTY.exists[p | p == (e.feature as NamedElement).name]) {
			return InferenceResult.from(inferTypeDispatch(e.feature).type, newArrayList(inferTypeDispatch(e.owner)))
		}

		return super.doInfer(e)
	}

	def doInfer(ComplexType e) {
		if (!(e.origin instanceof Statechart) &&
				!(e.origin instanceof InterfaceScope)) {
			for (element : e.features.filter(Property)) {
				var inferred = element.initialValue.inferTypeDispatch
				if (inferred === null)
					return super.doInfer(e)
				return getResult(inferred, 1)
			}
		}		
		return super.doInfer(e)
	}

}
