/**
 * Copyright (c) 2023 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.itemis.create.sunit.generator.csharp.nunit.codemodel

import com.google.inject.Inject
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.Part
import com.yakindu.base.types.TypeBuilder
import org.eclipse.emf.ecore.EObject
import com.itemis.create.base.generator.core.concepts.InternallyDefinedTypeAnnotation

/**
 * This class provides all those NUnit declarations which are used by the C# code generator.
 * These declarations can be used to build up a proper code model for code generation.
 *  
 * @author axel terfloth
 */
class NUnitLib {
	
	public static final String NUNIT_PACKAGE_NAME = "NUnit"
	
	public static final String ASSERT_CLASS_NAME = "Assert"
	public static final String IS_TRUE_METHOD_NAME = "That"
	
	public static final String MOCK_CLASS_NAME = "Mock"
	public static final String SETUP_METHOD_NAME = "Setup"
	public static final String VERIFY_METHOD_NAME = "Verify"
	public static final String RETURNS_METHOD_NAME = "Returns"
	public static final String OBJECT_PART_NAME = "Object"
	
	public static final String TIMES_CLASS_NAME = "Times"
	public static final String ONCE_METHOD_NAME = "Once"
	public static final String AT_LEAST_METHOD_NAME = "AtLeast"
	public static final String EXACTLY_METHOD_NAME = "Exactly"
	public static final String NEVER_METHOD_NAME = "Never"
	
	public static final String IT_CLASS_NAME = "It"
	public static final String IS_ANY_METHOD_NAME = "IsAny"
	
	@Inject protected extension TypeBuilder
	@Inject protected extension InternallyDefinedTypeAnnotation
	
	def nunitPackage(EObject context) {
		context.eAllContents.filter(Package).filter[NUNIT_PACKAGE_NAME == name].head	
	}
	
	def nunitAssertClass(EObject context) {
		context.nunitPackage.member.filter(ComplexType).filter[ASSERT_CLASS_NAME == name].head
	}
	
	def nunitAssertIsTrue(EObject context) {
		context.nunitAssertClass.features.filter(Operation).filter[IS_TRUE_METHOD_NAME == name].head
	}
	
	def nunitMockObject(EObject context) {
		context.nunitMockClass.features.filter(Part).filter[OBJECT_PART_NAME == name].head
	}
	
	def nunitMockClass(EObject context) {
		context.nunitPackage.member.filter(ComplexType).filter[MOCK_CLASS_NAME == name].head
	}
	
	def nunitTimesClass(EObject context) {
		context.nunitMockClass.features.filter(ComplexType).filter[TIMES_CLASS_NAME == name].head
	}
	
	def nunitExactly(EObject context) {
		context.nunitTimesClass.features.filter(Operation).filter[EXACTLY_METHOD_NAME == name].head
	}
	
	def nunitNever(EObject context) {
		context.nunitTimesClass.features.filter(Operation).filter[NEVER_METHOD_NAME == name].head
	}
	
	def nunitIsAny(EObject context) {
		context.nunitItClass.features.filter(Operation).filter[IS_ANY_METHOD_NAME == name].head
	}
	
	def nunitItClass(EObject context) {
		context.nunitMockClass.features.filter(ComplexType).filter[IT_CLASS_NAME == name].head
	}
	
	def nunitMockVerify(EObject context) {
		context.nunitMockClass.features.filter(Operation).filter[VERIFY_METHOD_NAME == name].head
	}
	
	def nunitMockSetup(EObject context) {
		context.nunitMockClass.features.filter(Operation).filter[SETUP_METHOD_NAME == name].head
	}
	
	def nunitMockReturns(EObject context) {
		context.nunitMockClass.features.filter(Operation).filter[RETURNS_METHOD_NAME == name].head
	}
	
	def defineNunitPackage() {
		_package(NUNIT_PACKAGE_NAME) => [
			_no_namespace
			it.member += defineAssertClass
			it.member += defineMockClass
		]	
	}
	
	def protected defineAssertClass() {
		_complexType(ASSERT_CLASS_NAME) => [
			features += defineIsTrueMethod
		]
	}
	
	def protected defineMockClass() {
		_complexType(MOCK_CLASS_NAME) => [
			features += defineObjectPart
			features += defineSetupMethod
			features += defineReturnsMethod
			features += defineVerifyMethod
			features += defineTimesClass
			features += defineItClass
		]
	}
	
	def protected defineObjectPart() {
		_part(OBJECT_PART_NAME, _any) => [
			_internallyDefinedType
		]
	}
	
	def protected defineItClass() {
		_complexType(IT_CLASS_NAME) => [
			features += defineIsAnyMethod
		]
	}
	
	def protected defineIsAnyMethod() {
		_op(IS_ANY_METHOD_NAME) => [
		]
	}
	
	def protected defineTimesClass() {
		_complexType(TIMES_CLASS_NAME) => [
			features += defineExactlyMethod
			features += defineNeverMethod
			features += defineAtLeastMethod
		]
	}
	
	def protected defineSetupMethod() {
		_op(SETUP_METHOD_NAME) => [
			_param("ocb", _typeSpecifier(_any))
		]
	}
	
	def protected defineReturnsMethod() {
		_op(RETURNS_METHOD_NAME) => [
			_param("returns", _typeSpecifier(_any))
		]
	}
	
	def protected defineVerifyMethod() {
		_op(VERIFY_METHOD_NAME) => [
			_param("op", _typeSpecifier(_any))
			_param("op", _typeSpecifier(_any))
			_param("condition", _typeSpecifier(_any))
		]
	}
	
	def protected defineExactlyMethod() {
		_op(EXACTLY_METHOD_NAME) => [	
			_param("times", _typeSpecifier(_integer))
		]
	}
	
	def protected defineNeverMethod() {
		_op(NEVER_METHOD_NAME) => [
		]
	}
	
	def protected defineAtLeastMethod() {
		_op(AT_LEAST_METHOD_NAME) => [
			_param("times", _typeSpecifier(_boolean))
		]
	}
	
	def protected defineIsTrueMethod() {
		_op(IS_TRUE_METHOD_NAME) => [
			_param("condition", _typeSpecifier(_boolean))
		]
	}
	
}