/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.domain.c.runtime.resource.transform.cpp

import com.google.inject.Inject
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.GenericElement
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeParameter
import com.yakindu.sct.domain.c.runtime.resource.CFileLocationAdapter
import com.yakindu.sct.domain.c.runtime.resource.ISimpleTypeMapper
import com.yakindu.sct.domain.c.runtime.resource.transform.c.CToModelTrafo
import java.util.HashSet
import org.eclipse.cdt.core.dom.ast.IASTCompositeTypeSpecifier
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier
import org.eclipse.cdt.core.dom.ast.IASTDeclarator
import org.eclipse.cdt.core.dom.ast.IASTFunctionDefinition
import org.eclipse.cdt.core.dom.ast.IASTNamedTypeSpecifier
import org.eclipse.cdt.core.dom.ast.IASTNode
import org.eclipse.cdt.core.dom.ast.IASTParameterDeclaration
import org.eclipse.cdt.core.dom.ast.IASTSimpleDeclaration
import org.eclipse.cdt.core.dom.ast.IASTStandardFunctionDeclarator
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTCompositeTypeSpecifier
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTCompositeTypeSpecifier.ICPPASTBaseSpecifier
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTParameterDeclaration
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTSimpleTypeTemplateParameter
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateDeclaration
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateId
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateParameter
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTypeId
import org.eclipse.cdt.core.dom.ast.cpp.ICPPConstructor
import org.eclipse.cdt.core.dom.ast.cpp.ICPPMethod
import org.eclipse.emf.ecore.EObject

/**
 * @author rbeckmann
 * 
 */
class CPPToModelTrafo extends CToModelTrafo {

	@Inject protected extension CPPTypeUtils
	@Inject protected extension ISimpleTypeMapper

	override void complete(HashSet<IASTSimpleDeclaration> declarations) {
		// Try to complete types of forward declared typedefs, skip templates
		declarations.filter[!declarators.exists[isTemplateOperation]].forEach[transform]
	}

	def protected dispatch void transform(ICPPASTTemplateDeclaration template) {
		template.declaration.transform
	}

	def protected dispatch void transform(IASTFunctionDefinition definition) {
		val declarator = definition.declarator
		if (declarator instanceof IASTStandardFunctionDeclarator) {
			val isStatic = definition.declSpecifier.storageClass === IASTDeclSpecifier.sc_static
			val operation = toOperation(definition.declSpecifier, declarator as IASTStandardFunctionDeclarator,
				isStatic)
			operation?.addToModel(definition.declSpecifier)
		}
	}

	def transformTemplateDeclaration(ICPPASTTemplateDeclaration it, IASTSimpleDeclaration declaration) {
		typeScope.push()

		var Type type

		if (!typeIsKnown(declaration.declSpecifier) || declaration.declarators.empty) {
			type = declaration.declSpecifier.parameterizedType(typeParameters)
		} else {
			declaration.declarators.forEach [ declarator |
				transformStep(declaration.declSpecifier, declarator)
			]
		}

		typeScope.pop()
		if (type !== null)
			typeScope.put(declaration.declSpecifier.qualifiedName, type)
	}

	override protected dispatch EObject transformStep(IASTDeclSpecifier declSpecifier, IASTDeclarator declarator) {
		if (declSpecifier.storageClass != IASTDeclSpecifier.sc_typedef) {
			val isStatic = declSpecifier.storageClass === IASTDeclSpecifier.sc_static
			return toProperty(declSpecifier, declarator, isStatic)
		}
		super._transformStep(declSpecifier, declarator)
	}

	override protected buildTypeSpecifier(IASTDeclSpecifier declSpecifier, IASTDeclarator declarator) {
		return createTypeSpecifier => [
			it.type = computeType(declSpecifier, declarator)
			if (declSpecifier instanceof IASTNamedTypeSpecifier) {
				val name = declSpecifier.getName();
				if (name instanceof ICPPASTTemplateId) {
					it.typeArguments += name.templateArguments.toTypeArguments
				}
			}
		]
	}

	override protected create createParameter parameter(IASTParameterDeclaration declaration) {
		typeSpecifier = computeTypeSpecifier(declaration.declSpecifier, declaration.declarator)
		val declarator = declaration.declarator
		if (declaration instanceof ICPPASTParameterDeclaration) {
			if (declarator.initializer !== null) {
				optional = true
			}
		}
		name = declarator.name.toString
		it.eAdapters.add(new CFileLocationAdapter(declaration.fileLocation))
	}

	override protected toOperation(IASTDeclSpecifier specifier, IASTStandardFunctionDeclarator declarator,
		boolean isStatic) {
		if (declarator.isConstructor || declarator.isDestructor) {
			return null
		}
		if (!isTemplateOperation(declarator)) {
			return super.toOperation(specifier, declarator, isStatic)
		} else {
			return createTemplateOperation(specifier, declarator)
		}
	}

	def isConstructor(IASTStandardFunctionDeclarator it) {
		val binding = ASTName.resolveBinding
		return binding instanceof ICPPConstructor
	}

	def isDestructor(IASTStandardFunctionDeclarator it) {
		val binding = ASTName.resolveBinding
		if (binding instanceof ICPPMethod) {
			return binding.isDestructor
		}
		return false
	}

	def createTemplateOperation(IASTDeclSpecifier declSpecifier, IASTStandardFunctionDeclarator declarator) {
		val template = findParentTemplate(declarator)
		typeScope.push()
		val typeParameters = template.templateParameters.map[toTypeParameter].filterNull
		val result = createOperation => [ op |
			op.typeParameters += typeParameters
			op.name = declarator.name.toString
			op.parameters += declarator.parameters.map[parameter]
			if (declarator.takesVarArgs) {
				op.parameters += variadicParameter()
			}
			op.typeSpecifier = computeTypeSpecifier(declSpecifier, declarator)
			op.eAdapters.add(new CFileLocationAdapter(declarator.fileLocation))
		]
		typeScope.pop()
		result
	}

	def protected getSuperTypes(IASTCompositeTypeSpecifier declSpecifier) {
		if (declSpecifier instanceof ICPPASTCompositeTypeSpecifier) {
			declSpecifier.baseSpecifiers.map[computeSuperType].filter(ComplexType).map [ t |
				createTypeSpecifier => [type = t]
			]
		} else {
			#[]
		}
	}

	def protected computeSuperType(ICPPASTBaseSpecifier baseSpecifier) {
		val declFileLocation = baseSpecifier.pathToDeclaringHeader
		if (!resourceLocation.equals(declFileLocation)) {
			return createComplexTypeProxy(resource, declFileLocation.toString, baseSpecifier) as ComplexType
		} else {
			return fromTypeScope(baseSpecifier.qualifiedName)
		}
	}

	override protected computeTypeSpecifier(IASTDeclSpecifier specifier, IASTDeclarator declarator) {
		val typeSpecifier = //if(specifier instanceof IASTNamedTypeSpecifier && (specifier as IASTNamedTypeSpecifier).isSmartPointer) (specifier as IASTNamedTypeSpecifier).mapTypeSpecifier else
		 super.computeTypeSpecifier(specifier, declarator)
		if (typeSpecifier.type instanceof GenericElement) {
			val template = findParentTemplate(specifier)
			if (template !== null)
				(typeSpecifier.type as GenericElement).typeParameters += findParentTemplate(specifier).typeParameters
		}
		typeSpecifier
	}

	def protected dispatch isTemplateOperation(IASTStandardFunctionDeclarator it) {
		val template = findParentTemplate
		return template !== null && template === parent.parent
	}

	def protected dispatch isTemplateOperation(IASTNode it) {
		false
	}

	def protected toTypeArguments(IASTNode[] nodes) {
		nodes.filter(ICPPASTTypeId).map[toTypeArgument]
	}

	def protected toTypeArgument(ICPPASTTypeId typeId) {
		createTypeSpecifier => [type = computeType(typeId.declSpecifier, typeId.abstractDeclarator)]
	}

	def protected typeParameters(ICPPASTTemplateDeclaration it) {
		templateParameters.map[toTypeParameter].filterNull
	}

	def protected Type parameterizedType(IASTDeclSpecifier specifier, Iterable<TypeParameter> typeParameters) {
		val type = specifier.findOrCreateType
		if (type instanceof GenericElement) {
			type.typeParameters += typeParameters
		}
		type
	}

	def protected dispatch toTypeParameter(ICPPASTTemplateParameter templateParameter) {
		// fallback
		null
	}

	def protected dispatch create param : createTypeParameter toTypeParameter(
		ICPPASTSimpleTypeTemplateParameter templateParameter) {
		param.name = templateParameter.getName().toString
		typeScope.put(templateParameter.qualifiedName, param)
	}

	def addSuperTypes(IASTCompositeTypeSpecifier declSpecifier) {
		val type = declSpecifier.findOrCreateType
		if (type instanceof ComplexType) {
			type.superTypes.addAll(getSuperTypes(declSpecifier))
		}
	}

	override protected void completeComplexType(IASTCompositeTypeSpecifier declSpecifier) {
		typeScope.push()
		val template = findParentTemplate(declSpecifier)
		if (template !== null) {
			val typeParameters = template.typeParameters
			typeParameters.forEach[addToModel(declSpecifier)]
		}
		addSuperTypes(declSpecifier)
		super.completeComplexType(declSpecifier)
		typeScope.pop()

	}

	override Type findOrCreateType(IASTDeclSpecifier declSpecifier) {
		//if ((declSpecifier instanceof IASTNamedTypeSpecifier) &&
		//	(declSpecifier as IASTNamedTypeSpecifier).isSmartPointer) {
		//	return (declSpecifier as IASTNamedTypeSpecifier).map
		//} else {
			super.findOrCreateType(declSpecifier)
		//}

	}
}
