/**
* Copyright (c) 2022 itemis AG - All rights Reserved
* Unauthorized copying of this file, via any medium is strictly prohibited
* 
* Contributors:
* 	Thomas Kutz - itemis AG
*   Axel Terfloth - itemis AG
*
*/
package com.yakindu.sct.domain.c.runtime.validator;

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.types.ArrayTypeSpecifier
import com.yakindu.base.types.Expression
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.TypedElement
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.domain.c.runtime.resource.transform.c.CTypeUtils
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtext.validation.AbstractDeclarativeValidator
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.EValidatorRegistrar

/**
 * 
 * @author kutz - initial contribution
 * @author terfloth - extensions
 *
 */
class CBuiltInTypesValidator extends AbstractDeclarativeValidator {

	public static final String ARRAY_DIMENSIONS_OUT_OF_BOUNDS = "Array type has %s dimensions, not %s."
	public static final String ARRAY_ACCESS_NOT_SUPPORTED_WITH_SMART_PTR = "Array access is not supported on smart pointers."
	
	@Inject
	protected ITypeSystem ts
	
	@Inject
	protected extension CTypeUtils
	
	def dispatch void checkArrayAccesForSmartPointer(Expression e){}
	
	def dispatch isArrayAcces(EObject it){
		false
	}
	
	def dispatch isArrayAcces(ElementReferenceExpression it){
		isArrayAccess
	}
	
	def dispatch isArrayAcces(FeatureCall it){
		isArrayAccess
	}
		
	@Check
	def dispatch void checkArrayAccesForSmartPointer(ElementReferenceExpression e) {
		if ((e.isArrayAccess() || e.eContainer?.isArrayAcces) && e.getReference() instanceof TypedElement) {
			val typeSpec = getArrayTypeSpecifier(e.getReference() as TypedElement)
			if (isSmartPointerType(typeSpec.getType())) {
				error(String.format(ARRAY_ACCESS_NOT_SUPPORTED_WITH_SMART_PTR), e, null)
			}
		}
	}
	
	@Check
	def dispatch void checkArrayAccesForSmartPointer(FeatureCall e) {
		if (e.isArrayAccess() && e.feature instanceof TypedElement ){
			if(e.isValueOnPointer) {
				checkArrayAccesForSmartPointer(e.owner)
			}
			val typeSpec = getArrayTypeSpecifier(e.feature as TypedElement)
			if (isSmartPointerType(typeSpec.type)) {
				error(String.format(ARRAY_ACCESS_NOT_SUPPORTED_WITH_SMART_PTR), e, null)
			}
		}
	}
	
	@Check
	def checkArrayDimension(ElementReferenceExpression e) {
		if (e.isArrayAccess() && e.reference instanceof TypedElement) {
			val typeSpec = getArrayTypeSpecifier(e.reference as TypedElement)
			val arrayDims = typeSpec !== null ? getArrayDimensions(typeSpec) : 0
			if (e.arraySelector.size > arrayDims) {
				error(String.format(ARRAY_DIMENSIONS_OUT_OF_BOUNDS, arrayDims, e.arraySelector.size), e, null)
			}
		}
	}
	
	@Check
	def checkArrayDimension(FeatureCall e) {
		if (e.isArrayAccess() && e.getFeature() instanceof TypedElement) {
			val typeSpec = getArrayTypeSpecifier(e.feature as TypedElement)
			val arrayDims = typeSpec !== null ? getArrayDimensions(typeSpec) : 0
			if (e.arraySelector.size > arrayDims) {
				error(String.format(ARRAY_DIMENSIONS_OUT_OF_BOUNDS, arrayDims, e.arraySelector.size), e, null)
			}
		}
	}
	
	def protected int getArrayDimensions(TypeSpecifier typeSpecifier) {
		if (isPointerType(typeSpecifier.type)) {
			return 1;
		}
		else if (isArrayType(typeSpecifier.type)) {
			return 1 + getArrayDimensions(typeSpecifier.typeArguments.get(0))
		}
		return 0;
	}
	
	/**
	 * TODO: consolidate with CBuiltInTypeContextInitializer.getArrayTypeSpecifier(TypedElement)
	 * 
	 */
	def protected TypeSpecifier getArrayTypeSpecifier(TypedElement element) {
		if (element.typeSpecifier instanceof ArrayTypeSpecifier) {
			return element.typeSpecifier as ArrayTypeSpecifier 
		}
		// iterate recursively for type aliases
		val typeSpec = element.typeSpecifier
		if (typeSpec.type instanceof TypedElement) {
			return getArrayTypeSpecifier(typeSpec.type as TypedElement);
		}
		return typeSpec
	}
	
	@Inject
	override void register(EValidatorRegistrar registrar) {
		// Do not register because this validator is only a composite #398987
	}
	
}
