/**
 * Copyright (c) 2022-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	andreas muelder - itemis AG
 * 
 */
package com.yakindu.sct.types.resource

import com.google.inject.Inject
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.types.AnnotatableElement
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.TypedDeclaration
import com.yakindu.base.types.TypesFactory
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.base.types.annotations.DeprecatedAnnotations
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.model.sgraph.RegularState
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.naming.SGraphShortNameProvider
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.types.lib.StatechartLibrary
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil.Copier
import org.eclipse.xtext.EcoreUtil2

import static com.yakindu.base.types.annotations.DeprecatedAnnotations.DEPRECATED_ANNOTATION

import static extension com.yakindu.sct.model.sgraph.naming.IdentifierConverter.*

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * @author axel terfloth - extensions
 * 
 */
class Statechart2TypeTransformation {

	extension TypesFactory factory = TypesFactory.eINSTANCE
	public static String NAMED_IFACE_CLASS = "__named_iface_class__"
	public static final String STATECHART_TYPE_ANNOTATION = "__statechart_type__";
	public static final String STATECHART_STATE_ENUM_ANNOTATION = "__statechart_state_enum__";
	public static final String CHILD_STATECHART_ANNOTATION = "__child_statechart__";

	@Inject extension ITypeSystem ts
	@Inject protected extension TypeBuilder
	@Inject protected extension ExpressionBuilder
	@Inject protected extension DeprecatedAnnotations
	@Inject extension OriginTracing
	@Inject extension SGraphShortNameProvider
	@Inject extension StatechartAnnotations
	@Inject StatechartLibrary lib

	def typeDescription(Statechart sc) {
		_createCache_createTypeDescription.get(#[sc])
	}

	def create p : createPackage createTypeDescription(Statechart sc) {

		p => [ package |
			package.name = sc.namespace

			package.member += createStateEnumType(sc)

			package.member += createStatechartType(sc)
		]
	}

	protected def <T extends EObject> T copyWithoutResolvingProxies(T eObject) {
		val copier = new Copier(false)
		val result = copier.copy(eObject)
		copier.copyReferences()
		result.traceOrigin(eObject)
		return result as T
	}

	def statechartType(Statechart it) {
		_createCache_createStatechartType.get(#[it])
	}

	def protected create result : createComplexType createStatechartType(Statechart sc) {
		result => [ scType |
			scType._annotate(STATECHART_TYPE_ANNOTATION)
			scType.name = sc.name
			scType.superTypes += createTypeSpecifier => [
				type = sc.isCycleBased ? lib.cycleBased : lib.getEventDriven
			]
			// Named Interfaces
			sc.scopes.filter(InterfaceScope).filter[!name.nullOrEmpty].forEach [ iface |
				scType.features += iface.createInterfaceType
				scType.features += createProperty => [ prop |
					prop.traceOrigin(iface)
					prop.name = iface.name
					prop.typeSpecifier = createTypeSpecifier => [
						type = iface.createInterfaceType
					]
				]
			]
			// Unnamed interfaces
			sc.scopes.filter(InterfaceScope).filter[name === null].forEach [ iface |
				iface.declarations.forEach[decl|scType.features += decl.copyWithoutResolvingProxies]
			]
			// isStateActive Operation
			scType.features += createOperation => [
				name = "isStateActive"
				parameters += createParameter => [
					name = "state"
					typeSpecifier = createTypeSpecifier => [
						type = sc.stateEnumType

					]
				]
				typeSpecifier = createTypeSpecifier => [
					type = ts.getType(ITypeSystem.BOOLEAN)
				]
			]
			scType.traceOrigin(sc)
		]
	}

	def stateEnumType(Statechart it) {
		_createCache_createStateEnumType.get(#[it])
	}

	protected def create enumeration : createEnumerationType createStateEnumType(Statechart sc) {
		enumeration => [
			name = '''«sc.name»States'''
			_annotate(STATECHART_STATE_ENUM_ANNOTATION)
			
			enumerator += createEnumerator => [name = noStateName]

			// create enumerators with qualified names
			sc.eAllContents.filter(RegularState).filter[true || !name.isNullOrEmpty].forEach [ state |
				val fqname = getFullyQualifiedName(state).toString.replaceAll(".*" + sc.name + "\\.", "").
					replaceAll("\\.", "_")
				enumerator += createEnumerator => [
					name = fqname
					traceOrigin(state)
				]
			]

			// create short named enumerator for backwards compatibility
			sc.eAllContents.filter(RegularState).filter[true || !name.isNullOrEmpty].forEach [ state |
				if (!state.name.isNullOrEmpty) {
					val shortStateEnumeratorName = state.name.toIdentifier

					if (! enumerator.exists[name.equals(shortStateEnumeratorName)]) {
						val intendedEnumerator = enumerator.findFirst[origin === state]

						enumerator += createEnumerator => [
							name = shortStateEnumeratorName
							it._annotate(DEPRECATED_ANNOTATION,
								("Use '" + intendedEnumerator.name + "' instead.")._string)
							traceOrigin(state)
						]
					}
				}
			]

			traceOrigin(sc)
		]
	}

	def noStateName() {
		"__NoState__"
	}

	protected def create createComplexType createInterfaceType(InterfaceScope iface) {
		it => [
			_annotate(NAMED_IFACE_CLASS)
			traceOrigin(iface)
			name = iface.name + "Type"
			static = true
			iface.declarations.forEach[decl|features += decl.copyWithoutResolvingProxies]
		]
	}

	def dispatch isNamedIface(AnnotatableElement it) {
		if (annotations.exists[type.name === NAMED_IFACE_CLASS])
			true
		else
			false
	}

	def dispatch isNamedIface(EObject it) {
		false
	}
	
	def dispatch isNamedIface(Void it) {
		false
	}

	def isInNamedIface(EObject it) {
		if (it !== null && it.eContainer !== null && eContainer instanceof AnnotatableElement &&
			(eContainer as AnnotatableElement).annotations.exists[type.name === NAMED_IFACE_CLASS])
			true
		else
			false
	}

	def getPropertyForType(ComplexType type) {
		(type.eContainer as ComplexType).features.filter(TypedDeclaration).filter[f|f.type === type].head
	}

	def dispatch isStatechartType(Type it) {
		if(it === null) return false
		val statechartType = it.getAnnotationOfType(STATECHART_TYPE_ANNOTATION);
		if (statechartType !== null)
			return true
		else
			return false
	}
	
	def ComplexType getContainingStatechartType(EObject it){
		if(it === null) return null
		if(it instanceof Statechart) return getContainingStatechartType
		val foundContainer = EcoreUtil2.getContainerOfType(it, ComplexType)
		if(foundContainer !== null && foundContainer.isStatechartType)
			foundContainer
		else
			it.eContainer.getContainingStatechartType
	}
	
	def dispatch isStatechartType(EObject it) {
		return false
	}
	
	def dispatch isStatechartType(Void it) {
		return false
	}

	def dispatch isChildStatechart(Declaration it) {
		val statechartType = it.getAnnotationOfType(CHILD_STATECHART_ANNOTATION);
		if (statechartType !== null)
			return true
		else
			return false
	}
	
	def dispatch isChildStatechart(EObject it) {
		return false
	}

	def dispatch isStatechartStateEnum(EnumerationType it) {
		if(it === null) return false
		val stateEnum = it.getAnnotationOfType(STATECHART_STATE_ENUM_ANNOTATION);
		if (stateEnum !== null)
			return true
		else
			return false
	}
	
	def dispatch isStatechartStateEnum(EObject it) {
		return false
	}
	
	def dispatch isStatechartStateEnum(Void it) {
		return false
	}
	
}
