/** 
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Contributors:
 * Andreas Muelder - itemis AG
 */
package com.yakindu.sctunit.simulation.core.interpreter

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.interpreter.IOperationExecutor
import com.yakindu.base.expressions.interpreter.scheduling.ITimeTaskScheduler
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.sct.simulation.core.sexec.interpreter.StextExpressionInterpreter
import com.yakindu.sctunit.sCTUnit.AssertionStatement
import com.yakindu.sctunit.sCTUnit.CodeBlock
import com.yakindu.sctunit.sCTUnit.ExpressionStatement
import com.yakindu.sctunit.sCTUnit.IfStatement
import com.yakindu.sctunit.sCTUnit.LoopStatement
import com.yakindu.sctunit.sCTUnit.MockReturnStatement
import com.yakindu.sctunit.sCTUnit.ProceedExpression
import com.yakindu.sctunit.sCTUnit.ProceedUnit
import com.yakindu.sctunit.sCTUnit.ReturnStatement
import com.yakindu.sctunit.sCTUnit.SCTUnitClass
import com.yakindu.sctunit.sCTUnit.SCTUnitOperation
import com.yakindu.sctunit.sCTUnit.VariableDefinitionStatement
import com.yakindu.sctunit.sCTUnit.VerifyCalledStatement
import com.yakindu.sctunit.simulation.core.junit.Failure
import com.yakindu.sctunit.simulation.core.junit.TestCase
import java.util.Collections
import java.util.List
import org.eclipse.core.runtime.IProgressMonitor
import org.eclipse.core.runtime.NullProgressMonitor
import org.eclipse.core.runtime.OperationCanceledException
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.EcoreUtil2

class BaseSCTUnitTestCaseInterpreter extends StextExpressionInterpreter implements ISCTUnitTestCaseInterpreter {

	@Inject protected ISCTUnitExecutionContextInitializer sctUnitContextInitializer
	@Inject protected IFailureTracer failureTracer;
	@Inject protected ITimeTaskScheduler schedulingService;
	@Inject(optional=true) protected IProgressMonitor monitor = new NullProgressMonitor

	protected var TestCase testCase
	protected var cyclePeriod = 200L;

	override init(SCTUnitClass unitClass) {
		sctUnitContextInitializer.initialize(context, unitClass)
	}

	protected static class Return extends RuntimeException {
		Object value;

		new(Object value) {
			setValue(value)
		}

		def Object getValue() {
			return value
		}

		def Object setValue(Object value) {
			this.value = value
		}
	}

	override TestCase executeOperation(SCTUnitOperation operation) {

		sctUnitContextInitializer.initialize(context, operation, Collections.emptyList)
		testCase = new TestCase => [
			name = operation.name
			className = EcoreUtil.getURI(operation).toString
		]
		try {
			for (stm : operation.body.code) {
				if (monitor.isCanceled)
					throw new OperationCanceledException()
				if (testCase.failures.nullOrEmpty) {
					stm.executeStatement
				}
			}
		} catch (Return r) {
			// @Test operations never return a value
		} catch (OperationCanceledException exx) {
			val failure = new Failure
			failure.message = '''Test execution was canceled.'''
			testCase.failures.add(failure)
		} catch (Exception e) {
			val failure = new Failure
			failure.message = '''Test failed due to '«e.class.name»': "«e.message»"'''
			testCase.failures.add(failure)
		}
		sctUnitContextInitializer.dispose(context, operation)
		resetOperationMockups()
		return testCase

	}

	protected def void resetOperationMockups() {
		operationExecutors?.filter(SCTUnitOperationMockup).forEach[reset]
	}

	def Object executeStatement(SCTUnitOperation operation, List<Expression> args, TestCase testCase,
		SCTUnitOperation owner) {
		sctUnitContextInitializer.initialize(context, operation, args, owner)
		try {
			return operation.body.executeStatement
		} catch (Return r) {
			return r.value;
		}
	}

	def dispatch Object executeStatement(Expression expression) {
		evaluate(expression, context)
	}

	def dispatch Object executeStatement(ExpressionStatement stm) {
		val result = stm.expression.executeStatement?.asValue
		result
	}

	def dispatch Object executeStatement(ReturnStatement stm) {
		throw new Return(
			if (stm.returnValue !== null)
				evaluate(stm.returnValue, context).asValue
			else
				null
		)
	}

	def dispatch Object executeStatement(VariableDefinitionStatement stm) {
		val op = EcoreUtil2.getContainerOfType(stm, typeof(SCTUnitOperation))
		sctUnitContextInitializer.initialize(context, stm.definition, op)
		null
	}

	def protected dispatch Object executeStatement(CodeBlock it) {
		if (monitor.isCanceled)
			throw new OperationCanceledException()
		var Object lastResult = null
		for (statement : code) {
			if (monitor.isCanceled)
				throw new OperationCanceledException()
			lastResult = statement.executeStatement
		}
		lastResult
	}

	def dispatch Object execute(ProceedExpression stm) {
		val proceedValue = stm.value.value as Long
		if (stm.unit == ProceedUnit.CYCLES) {
			val cycles = proceedValue.intValue
			schedulingService.cycleLeap(cycles)
		} else {
			schedulingService.timeLeap(toMilliseconds(proceedValue, stm.unit))
		}
		return null
	}

	def toMilliseconds(long value, ProceedUnit unit) {
		switch (unit) {
			case MILLISECOND: value
			case MICROSECOND: value / 1_000
			case NANOSECOND: value / 1_000_000
			case SECOND: value * 1_000
			default: value
		}
	}

	def dispatch Object executeStatement(AssertionStatement stm) {
		var valid = evaluate(stm.expression, context).asValue as Boolean
		if (!valid) {
			testCase.failures.add(failureTracer.getFailureStack(stm, context))
		}
		null
	}

	override tearDown() {
		schedulingService.terminate
		context = null
	}

	override getExecutionContext() {
		return context
	}

	override dispatch Object execute(ElementReferenceExpression expression) {
		if (expression.expressionSCTUnitOperation !== null) {
			val owner = EcoreUtil2.getContainerOfType(expression, SCTUnitOperation)
			return expression.expressionSCTUnitOperation.executeStatement(expression.expressions, testCase, owner)
		}
		return executeElementReferenceExpression(expression)
	}

	def dispatch Object executeStatement(VerifyCalledStatement it) {
		val failure = failureTracer.getFailureStack(it, reference.params.map[executeStatement])
		if (failure !== null)
			testCase.failures.add(failure)
		null
	}

	def dispatch Object executeStatement(MockReturnStatement stm) {
		val args = if (!stm.reference.operationArgs.nullOrEmpty) {
				stm.reference.operationArgs.map[exp|exp.executeStatement]
			} else {
				Collections.EMPTY_LIST
			}
		operationExecutors?.filter(SCTUnitOperationMockup).forEach [
			mockReturnValue(stm.reference.getOperation, args, stm.value)
		]
		return null
	}

	override dispatch execute(FeatureCall call) {
		if (call.expressionSCTUnitOperation !== null) {
			val owner = EcoreUtil2.getContainerOfType(call, SCTUnitOperation)
			return call.expressionSCTUnitOperation.executeStatement(call.expressions, testCase, owner)
		} else {
			return executeFeatureCall(call)
		}
	}

	def dispatch SCTUnitOperation getExpressionSCTUnitOperation(ElementReferenceExpression it) {
		var reference = it.reference
		if (reference instanceof SCTUnitOperation) {
			return reference as SCTUnitOperation
		}
		return null
	}

	def dispatch SCTUnitOperation getExpressionSCTUnitOperation(FeatureCall it) {
		if (feature instanceof SCTUnitOperation) {
			return feature as SCTUnitOperation
		}
		return null
	}

	def dispatch List<Expression> getOperationArgs(ArgumentExpression it) {
		return expressions;
	}

	def dispatch List<Expression> getOperationArgs(Expression it) {
		return null;
	}

	def dispatch Operation getOperation(Expression it) {
	}

	def dispatch Operation getOperation(FeatureCall it) {
		return feature as Operation
	}

	def dispatch Operation getOperation(ElementReferenceExpression it) {
		return reference as Operation
	}

	def dispatch List<Expression> getParams(Expression it) {
		return Collections.emptyList
	}

	def dispatch List<Expression> getParams(ArgumentExpression it) {
		if (expressions.nullOrEmpty) {
			return Collections.emptyList
		} else {
			return expressions
		}
	}

	def dispatch Object executeStatement(LoopStatement stm) {
		var condition = evaluate(stm.guard, context) as Boolean
		var Object result = null
		while (condition) {
			result = stm.body.executeStatement
			condition = evaluate(stm.guard, context) as Boolean
		}
		return result
	}

	def dispatch Object executeStatement(IfStatement stm) {
		val condition = evaluate(stm.condition, context) as Boolean
		var Object result = null
		if (condition) {
			result = stm.then.executeStatement
		} else {
			if (stm.^else !== null)
				result = stm.^else.executeStatement
		}
		return result
	}

	override executeOperation(IOperationExecutor executor, ArgumentExpression expression) {
		val result = super.executeOperation(executor, expression)
		if (result instanceof Expression)
			return result.executeStatement
		return result
	}

}
