/**
 * Copyright (c) 2023-2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * @author finlay weegen - Initial contribution and API
 * 
 */
package com.yakindu.sctunit.coverage.io

import com.fasterxml.jackson.core.JsonParseException
import com.fasterxml.jackson.core.JsonParser
import com.fasterxml.jackson.core.JsonProcessingException
import com.fasterxml.jackson.databind.DeserializationContext
import com.fasterxml.jackson.databind.JsonDeserializer
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.module.SimpleModule
import com.google.common.collect.HashBasedTable
import com.google.common.collect.Table
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sctunit.sCTUnit.SCTUnitClass
import com.yakindu.sctunit.sCTUnit.SCTUnitSuite
import java.io.IOException
import java.util.ArrayList
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl
import org.eclipse.xtext.naming.DefaultDeclarativeQualifiedNameProvider
import com.yakindu.sct.simulation.core.coverage.MeasurementExtension
import com.yakindu.sctunit.coverage.builder.SCTUnitMeasurementBuilder
import com.yakindu.sct.simulation.core.coverage.Measurement.StateTransitionCoverage
import com.yakindu.sct.simulation.core.coverage.Measurement.TestCaseCount
import com.yakindu.sct.simulation.core.coverage.Measurement
import com.yakindu.sct.simulation.core.coverage.Measurement.Measure
import com.yakindu.base.base.NamedElement

/**
 * 
 * TODO: add constants for all string literals used for JSON keys.
 */
class JSONToMeasurement {

	protected extension DefaultDeclarativeQualifiedNameProvider provider
	protected extension MeasurementExtension mExtension = new MeasurementExtension

	var String jsonString = ""
	var ObjectMapper mapper
	var JsonNode jsonNode

	new(String inputJsonString) {
		this.jsonString = inputJsonString
		this.mapper = new ObjectMapper
		this.jsonNode = mapper.readTree(jsonString) as JsonNode
	}

	def String getURIString() {
		var uri = jsonNode.get("uri").asText
		uri
	}

	def getDate() {
		var date = jsonNode.get("date")
		date.asText
	}

	def int getNumberOfTestCases() {
		var numberOfTestCases = jsonNode.get("numberOfTestCases")
		numberOfTestCases.asInt
	}

	def Measurement generateMeasurement() {
		var testURI = URI.createURI(URIString, true)
		var measurementNode = jsonNode.get("measurement").toString

		// register type adapter
		var module = new SimpleModule()
		module.addDeserializer(Measurement, new MeasurementDeserializer())
		module.addDeserializer(Class, new ClassDeserializer)
		module.addDeserializer(Measure, new MeasuresDeserializer)
		mapper.registerModule(module)

		var measurement = (mapper.readValue(measurementNode, Measurement)).setMeasurementParents
		var populatedMeasurement = buildNewMeasurement(measurement, testURI)
		populatedMeasurement
	}

	protected def Measurement buildNewMeasurement(Measurement measurement, URI uri) {
		// creating a new resource set and getting the resource from the uri
		var Measurement builtMeasurement
		var measurementBuilder = new SCTUnitMeasurementBuilder()
		val resourceSet = new ResourceSetImpl().getResource(uri, true)

		val subject = resourceSet.allContents.filter(NamedElement)
		   				.filter[it instanceof SCTUnitClass || it instanceof SCTUnitSuite || it instanceof Statechart].head
   		builtMeasurement = measurementBuilder.buildMeasurement(subject)

		measurement.reconcileSubjects(builtMeasurement)
		measurement
	}

	def protected reconcileSubjects(Measurement target, Measurement source) {
		val Table<String, String, EObject> subjectTable = HashBasedTable.create
		
		source.findAll[subject !== null].forEach [
			if (! subjectTable.contains(type.simpleName, subjectID)) {
				subjectTable.put(type.simpleName, subjectID, subject)
			}
		]

		target.findAll[subject === null && subjectID !== null && type !== null].forEach [
			subject = subjectTable.get(type.simpleName, subjectID)
		]
	}

	protected def Measurement setMeasurementParents(Measurement measurement) {
		measurement.findAll[children !== null].forEach [
			children.forEach[child|child.parent = it]
		]
		measurement
	}

	static class MeasurementDeserializer extends JsonDeserializer<Measurement> {

		override Measurement deserialize(JsonParser parser, DeserializationContext context) throws IOException {

			var mapper = new ObjectMapper
			var module = new SimpleModule
			module.addDeserializer(Measurement, new MeasurementDeserializer())
			module.addDeserializer(Class, new ClassDeserializer)
			module.addDeserializer(Measure, new MeasuresDeserializer)
			mapper.registerModule(module)

			var node = (mapper.readTree(parser) as JsonNode)
			var subjectID = node.get("subjectID").asText
			var name = node.get("name").asText
			var timestamp = (node.get("timestamp") !== null) ? node.get("timestamp").asLong : 0L
			var type = mapper.readValue(node.get("type").traverse, Class)

			var children = new ArrayList
			if (node.has("children")) {
				var childrenNode = node.get("children")
				if (childrenNode.isArray) {
					for (childNode : childrenNode) {
						var child = mapper.readValue(childNode.traverse, Measurement)
						children.add(child)
					}
				}
			}

			var measures = new ArrayList
			if (node.has("measures")) {
				var measuresNode = node.get("measures")
				if (measuresNode.isArray) {
					for (mNode : measuresNode) {
						var measure = mapper.readValue(mNode.traverse, Measure)
						if(measure !== null) measures.add(measure)
					}
				}
			}

			var measurement = new Measurement
			measurement.setName(name)
			measurement.setSubjectID(subjectID)
			measurement.setType(type)
			measurement.timestamp = timestamp
			measurement.children.addAll(children)
			measurement.measures.addAll(measures)
			measurement
		}
	}
}

class ClassDeserializer extends JsonDeserializer<Class<?>> {

	override Class<?> deserialize(JsonParser parser,
		DeserializationContext context) throws IOException, JsonProcessingException {
		var className = parser.getText
		try {
			return Class.forName(className)
		} catch (ClassNotFoundException e) {
			throw new JsonParseException(parser, "Failed to deserialize class: " + className, e)
		}
	}
}

class MeasuresDeserializer extends JsonDeserializer<Measurement.Measure> {

	override Measure deserialize(JsonParser parser,
		DeserializationContext context) throws IOException, JsonProcessingException {

		val mapper = new ObjectMapper
		val node = mapper.readTree(parser) as JsonNode

		val count = node.get("count")
		if (count !== null) {
			val visits = new Measurement.Visits
			visits.count = count.asInt
			return visits
		}

		val testCases = node.get("testCases")
		if (testCases !== null) {
			val testCaseCount = new TestCaseCount(testCases.asInt)
			return testCaseCount
		}

		val coverage = node.get("coverage")
		if (coverage !== null) {
			val weight = node.get("weight")
			if (weight !== null) {
				val stCoverage = new StateTransitionCoverage(coverage.floatValue, weight.intValue)
				return stCoverage
			}
		}

		return null
	}

}
