/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.json.transformation.xmi2json

import com.yakindu.sct.json.transformation.model.JsonCell
import com.yakindu.sct.json.transformation.model.JsonChoice
import com.yakindu.sct.json.transformation.model.JsonEntry
import com.yakindu.sct.json.transformation.model.JsonExit
import com.yakindu.sct.json.transformation.model.JsonFinal
import com.yakindu.sct.json.transformation.model.JsonGraph
import com.yakindu.sct.json.transformation.model.JsonNodeLabel
import com.yakindu.sct.json.transformation.model.JsonPosition
import com.yakindu.sct.json.transformation.model.JsonRegion
import com.yakindu.sct.json.transformation.model.JsonRoot
import com.yakindu.sct.json.transformation.model.JsonState
import com.yakindu.sct.json.transformation.model.JsonStatechart
import com.yakindu.sct.json.transformation.model.JsonSynchronization
import com.yakindu.sct.json.transformation.model.JsonTransition
import com.yakindu.sct.json.transformation.model.JsonTransitionSource
import com.yakindu.sct.json.transformation.model.JsonTransitionTarget
import com.yakindu.sct.json.transformation.model.util.NodeModelFactory
import com.yakindu.sct.json.transformation.model.util.NotationUtil
import com.yakindu.sct.model.sgraph.Choice
import com.yakindu.sct.model.sgraph.Entry
import com.yakindu.sct.model.sgraph.Exit
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Synchronization
import com.yakindu.sct.model.sgraph.Transition
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.emf.ecore.xmi.XMIResource
import org.eclipse.gmf.runtime.notation.Diagram
import org.eclipse.gmf.runtime.notation.Edge

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
class XMIToJSONTransformation {

	extension NotationUtil = new NotationUtil
	extension NodeModelFactory = new NodeModelFactory

	val jsonGraph = new JsonGraph

	def JsonRoot transform(Resource resource) {
		val statechart = resource.contents.filter(Statechart).head
		val diagram = resource.contents.filter(Diagram).head

		val jsonStatechart = new JsonStatechart => [
			name = statechart.name
			specification = statechart.specification
		]
		jsonGraph.cells += jsonStatechart
		statechart.regions.forEach[toJSON(null, diagram)]

		new JsonRoot => [
			graph = jsonGraph
		]
	}

	def dispatch protected void toJSON(Region region, JsonCell parent, Diagram diagram) {
		val jsonRegion = new JsonRegion => [
			name = region.name
			id = region.ID
			parent = parent?.id ?: null
			z = if(parent !== null) parent.z + 1 else 1
		]
		jsonGraph.cells += jsonRegion
		if (parent instanceof JsonState)
			parent.embeds += jsonRegion.id

		val node = findNotationView(region, diagram, Region.simpleName)
		jsonRegion.setBounds(node, jsonGraph)
		region.vertices.forEach[toJSON(jsonRegion, diagram)]
	}

	def dispatch protected void toJSON(State state, JsonCell parent, Diagram diagram) {
		val jsonState = new JsonState => [
			id = state.ID
			parent = parent?.id ?: null
			name = state.name
			specification = state.specification
			z = parent.z + 1
		]
		jsonGraph.cells += jsonState
		if (parent instanceof JsonRegion)
			parent.embeds += jsonState.id

		var node = findNotationView(state, diagram, State.simpleName)
		jsonState.setBounds(node, jsonGraph)

		state.regions.forEach[toJSON(jsonState, diagram)]
		state.outgoingTransitions.forEach[toJSON(jsonState, diagram)]

	}

	def dispatch protected void toJSON(Entry entry, JsonCell parent, Diagram diagram) {

		val nodeLabel = new JsonNodeLabel => [
			!entry.name.nullOrEmpty ? name = entry.name
			parent = entry.ID
			id = EcoreUtil.generateUUID()
			z = 1000
		]
		jsonGraph.cells += nodeLabel

		val jsonEntry = new JsonEntry => [
			id = entry.ID
			entryKind = entry.kind.literal
			parent = parent?.id ?: null
			z = parent.z + 1
			embeds += nodeLabel.id

		]
		jsonGraph.cells += jsonEntry

		if (parent instanceof JsonRegion)
			parent.embeds += jsonEntry.id

		val node = findNotationView(entry, diagram, entry.kind.entryNodeName)
		if (node !== null) {
			jsonEntry.setBounds(node, jsonGraph)
			nodeLabel.position = new JsonPosition => [
				x = jsonEntry.position.x
				y = jsonEntry.position.y + 22
			]

		}
		entry.outgoingTransitions.forEach[toJSON(jsonEntry, diagram)]
	}

	def dispatch protected void toJSON(Exit exit, JsonCell parent, Diagram diagram) {
		val nodeLabel = new JsonNodeLabel => [
			!exit.name.nullOrEmpty ? name = exit.name
			it.parent = exit.ID
			id = EcoreUtil.generateUUID()
			z = 1000
		]
		jsonGraph.cells += nodeLabel

		val jsonExit = new JsonExit => [
			id = exit.ID
			!exit.name.nullOrEmpty ? name = exit.name
			parent = parent?.id ?: null
			z = parent.z + 1
			embeds += nodeLabel.id
		]
		jsonGraph.cells += jsonExit
		if (parent instanceof JsonRegion)
			parent.embeds += jsonExit.id

		val node = findNotationView(exit, diagram, "Exit")
		jsonExit.setBounds(node, jsonGraph)

		nodeLabel.position = new JsonPosition => [
			x = jsonExit.position.x
			y = jsonExit.position.y + 22
		]

		exit.outgoingTransitions.forEach[toJSON(jsonExit, diagram)]
	}

	def dispatch protected void toJSON(FinalState finalState, JsonCell parent, Diagram diagram) {
		val jsonFinal = new JsonFinal => [
			id = finalState.ID
			parent = parent?.id ?: null
			z = parent.z + 1
		]
		jsonGraph.cells += jsonFinal
		if (parent instanceof JsonRegion)
			parent.embeds += jsonFinal.id

		val node = findNotationView(finalState, diagram, FinalState.simpleName)
		jsonFinal.setBounds(node, jsonGraph)

	}

	def dispatch protected void toJSON(Choice choice, JsonCell parent, Diagram diagram) {
		val jsonChoice = new JsonChoice => [
			id = choice.ID
			parent = parent?.id ?: null
			z = parent.z + 1
		]
		jsonGraph.cells += jsonChoice
		if (parent instanceof JsonRegion)
			parent.embeds += jsonChoice.id

		val node = findNotationView(choice, diagram, Choice.simpleName)
		jsonChoice.setBounds(node, jsonGraph)

		choice.outgoingTransitions.forEach[toJSON(jsonChoice, diagram)]

	}

	def dispatch protected void toJSON(Synchronization sync, JsonCell parent, Diagram diagram) {
		val jsonSync = new JsonSynchronization => [
			id = sync.ID
			parent = parent?.id ?: null
			z = parent.z + 1
		]
		jsonGraph.cells += jsonSync
		if (parent instanceof JsonRegion)
			parent.embeds += jsonSync.id

		val node = findNotationView(sync, diagram, Synchronization.simpleName)
		jsonSync.setBounds(node, jsonGraph)

		sync.outgoingTransitions.forEach[toJSON(jsonSync, diagram)]
	}

	def dispatch protected void toJSON(Transition transition, JsonCell parent, Diagram diagram) {
		val jsonTransition = new JsonTransition => [
			id = transition.ID
			z = 1000
			specification = transition.specification
			source = new JsonTransitionSource => [
				id = transition.source.ID
			]
			target = new JsonTransitionTarget => [
				id = transition.target.ID
			]
		]

		val edge = findNotationEdge(transition, diagram, Transition.simpleName) as Edge
		jsonTransition.vertices = edge.transformRelativeBendpoints(parent)
		jsonGraph.cells += jsonTransition
	}

	def protected ID(EObject object) {
		return (object.eResource as XMIResource).getID(object);
	}
}
