/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.stext.validation

import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.ExpressionsPackage
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.RegularEventSpec
import com.yakindu.base.expressions.expressions.TimeEventSpec
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Direction
import com.yakindu.base.types.Event
import com.yakindu.base.types.TypesPackage
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.util.StatechartUtil
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.extensions.STextExtensions
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.model.stext.stext.InternalScope
import java.util.Collection
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EStructuralFeature.Setting
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.naming.IQualifiedNameProvider
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.CheckType

import static extension org.eclipse.xtext.EcoreUtil2.*

class EventValidator extends STextBaseValidator {

	@Inject
	ITypeSystemInferrer typeInferrer
	@Inject
	extension ExpressionExtensions
	@Inject
	extension IQualifiedNameProvider
	@Inject
	extension STextExtensions
	@Inject
	extension StatechartAnnotations
	@Inject
	extension StatechartUtil

	@Check(CheckType.FAST)
	def void checkExpression(TimeEventSpec expression) {
		typeInferrer.infer(expression, this)
	}

	@Check(CheckType.FAST)
	def void checkRaisingExpressionEvent(EventRaisingExpression expression) {
		var EObject element = expression.getEvent().featureOrReference
		if (element !== null && (!(element instanceof Event))) {
			var String elementName = ""
			if (element instanceof NamedElement) {
				elementName = element.getName()
			}
			error(String.format("'%s' is not an event.", elementName),
				ExpressionsPackage.Literals.EVENT_RAISING_EXPRESSION__EVENT, -1)
		}
	}

	public static final String OUT_EVENT_NEVER_RAISED = "Out event '%s' is never raised in this statechart. The transition trigger is therefore never active.";

	@Check(CheckType.FAST)
	def void checkNotRaisedOutEvent(EventDefinition event) {
		if (event.direction != Direction.OUT) {
			return;
		}
		var Collection<Setting> usages = EcoreUtil.UsageCrossReferencer.find(event, event.eResource());
		var isRaised = usages.exists[setting|setting.getEObject().eContainer instanceof EventRaisingExpression]
		if (!isRaised) {
			usages.filter [ setting |
				setting.EObject instanceof ElementReferenceExpression || setting.EObject instanceof FeatureCall
			].forEach [ setting |
				warning(String.format(OUT_EVENT_NEVER_RAISED, event.fullyQualifiedName), setting.EObject, null, -1)
			]
		}
	}


	public static final String LOCAL_DECLARATIONS = "Local declarations are not allowed in interface scope. Violated by: ";
	public static final String IN_OUT_DECLARATIONS = "In/Out declarations are not allowed in internal scope. Violated by: ";

	@Check(CheckType.FAST)
	def void checkEventDefinition(EventDefinition event) {
		if (event.eContainer() instanceof InterfaceScope && event.getDirection() === Direction.LOCAL) {
			error(LOCAL_DECLARATIONS + event.name, TypesPackage.Literals.EVENT__DIRECTION)
		}
		if (event.eContainer() instanceof InternalScope && event.getDirection() !== Direction.LOCAL) {
			error(IN_OUT_DECLARATIONS + event.name, TypesPackage.Literals.EVENT__DIRECTION)
		}
	}

	static final String UNRAISED_LOCAL_EVENT_MSG = "The local event '%s' is never raised.";
	public static final String UNRAISED_LOCAL_EVENT_CODE = "LocalEventNeverRaised";

	@Check(CheckType.FAST)
	def void checkInternalEventsAreRaised(InternalScope scope) {
		val allRaisedEvents = scope.eResource.statechart.eAllContents.filter(EventRaisingExpression).map [
			event.featureOrReference
		].filter(Event).toSet
		val localEvents = scope.declarations.filter(Event).toSet
		localEvents.removeIf[allRaisedEvents.map[name].toSet.contains(it.name)]
		localEvents.forEach [
			warning(String.format(UNRAISED_LOCAL_EVENT_MSG, it.name), it, null, UNRAISED_LOCAL_EVENT_CODE)
		]
	}
	
	public static final String EVENT_RAISED_AND_NOT_CONSUMED = "The unbuffered event '%s' is raised but not consumed downstream. Unbuffered events are only visible 'downstream' in subsequent orthogonal regions within the same run-to-completion step."
	public static final String EVENT_RAISED_AND_NOT_CONSUMED_CODE = "EventRaisedAndNotConsumed";
	
	@Check(CheckType.FAST)
	def void checkRaisedUnbufferedEventIsConsumedDownstream(EventRaisingExpression expression) {
		val element = expression.event.featureOrReference
		if (element === null || element.eContainer.isMultiSM) return;
		if (element instanceof Event) {
			if ((element.direction == Direction.IN || element.direction == Direction.LOCAL) && !element.isBuffered) {
				val downstreamRegions = expression.contextElement.downstreamRegions
				for (Region r : downstreamRegions) {
					if (r.eAllOfType(ArgumentExpression).exists[featureOrReference == element]) {
						return
					}
				}
				warning(String.format(EVENT_RAISED_AND_NOT_CONSUMED, element.name), expression, null, EVENT_RAISED_AND_NOT_CONSUMED_CODE)
			}
		}
	}
	
	public static final String UNBUFFERED_LOCAL_EVENT_NOT_RAISED = "The unbuffered local event '%s' is never raised. For unbuffered local events, event raising must happen before they are consumed, i.e. in preceding orthogonal regions."
	public static final String UNBUFFERED_LOCAL_EVENT_NOT_RAISED_CODE = "UnbufferedLocalEventNotRaised";
	
	@Check(CheckType.FAST)
	def void checkUnbufferedLocalEventIsRaisedUpstream(RegularEventSpec trigger) {
		val element = trigger.event.featureOrReference
		if (element === null || element.eContainer.isMultiSM) return;
		if (element instanceof Event) {
			if (element.direction == Direction.LOCAL && !element.isBuffered) {
				val upstreamRegions = trigger.contextElement.upstreamRegions
				for (Region r : upstreamRegions) {
					if (r.eAllOfType(EventRaisingExpression).exists[event.featureOrReference == element]) {
						return
					}
				}
				warning(String.format(UNBUFFERED_LOCAL_EVENT_NOT_RAISED, element.name), trigger, null, UNBUFFERED_LOCAL_EVENT_NOT_RAISED_CODE)
			}
		}
	}
	
	public static final String RAISING_SUBMACHINES_OUT_EVENT = "Out events of submachines can not be raised. Use an in event instead of: "
	public static final String RAISING_SUBMACHINES_OUT_EVENT_CODE = "RasingSubmachinesOutEvent"

	@Check(CheckType.FAST)
	def void checkNotRaisingSubmachinesOutEvent(EventRaisingExpression expression) {
		val event = expression.event.featureOrReference
		if (event instanceof EventDefinition) {
			if (event.eContainer.isMultiSM && event.direction == Direction.OUT) {
				error(RAISING_SUBMACHINES_OUT_EVENT + event.name, expression, null, RAISING_SUBMACHINES_OUT_EVENT_CODE)
			}
		}
	}


	public static final String RAISED_EVENT_NOT_SPECIFIED = "No valid event to raise is specified."
	public static final String RAISED_EVENT_NOT_SPECIFIED_CODE = "RaisedEventNotSpecified"

	@Check(CheckType.FAST)
	def void checkRaisedEventSpecified(EventRaisingExpression expression) {
		val event = expression.event.featureOrReference
		if (event === null) {
			error(RAISED_EVENT_NOT_SPECIFIED, expression, null, RAISED_EVENT_NOT_SPECIFIED_CODE)		
		} 
	}


	protected def isBuffered(Event ev) {
		val statechart = ev.statechart
		return 
			ev.direction == Direction.IN && statechart.isInEventBuffer ||
			ev.direction == Direction.LOCAL && statechart.isInternalEventBuffer
	}
	
	protected def getUpstreamRegions(EObject base) {
		base.allContainers.filter(Region).map[r | r.predecessorRegions].flatten.toList
	}
	
	protected def getPredecessorRegions(Region r) {
		val idx = r.composite.regions.indexOf(r)
		if (idx > 0) 
			r.composite.regions.subList(0, idx) 
		else 
			#[]
	}
	
	protected def getDownstreamRegions(EObject base) {
		base.allContainers.filter(Region).map[r | r.successorRegions].flatten.toList
	}
	
	protected def getSuccessorRegions(Region r) {
		val idx = r.composite.regions.indexOf(r)
		if (idx < r.composite.regions.size - 1) 
			r.composite.regions.subList(idx + 1, r.composite.regions.size) 
		else 
			#[]
	}
}
