/**
 * Copyright (c) 2021-2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.transformation

import com.google.inject.Inject
import com.yakindu.base.types.validation.IValidationIssueAcceptor
import com.yakindu.base.types.validation.IValidationIssueAcceptor.ListBasedValidationIssueAcceptor
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.concepts.CompletionEvent
import com.yakindu.sct.model.sexec.concepts.Customizations
import com.yakindu.sct.model.sexec.concepts.EnterMethod
import com.yakindu.sct.model.sexec.concepts.EventBuffer
import com.yakindu.sct.model.sexec.concepts.EventProcessing
import com.yakindu.sct.model.sexec.concepts.ExecutionGuard
import com.yakindu.sct.model.sexec.concepts.ExitMethod
import com.yakindu.sct.model.sexec.concepts.HaltBehavior
import com.yakindu.sct.model.sexec.concepts.MicroStep
import com.yakindu.sct.model.sexec.concepts.RunCycleMethod
import com.yakindu.sct.model.sexec.concepts.SuperStep
import com.yakindu.sct.model.sexec.concepts.TriggerWithoutEventMethod
import com.yakindu.sct.model.sexec.transformation.config.IFlowConfiguration
import com.yakindu.sct.model.sgraph.Statechart

/**
 * @author axel terfloth
 */
class ModelSequencer implements IModelSequencer {
	 
	@Inject extension SexecElementMapping mapping
	@Inject extension StructureMapping structureMapping
	@Inject extension BehaviorMapping behaviorMapping
	@Inject extension ReactionBuilder reactBuilder
	@Inject extension SequenceBuilder seqBuilder
	@Inject extension StateVectorBuilder svBuilder
	@Inject extension RetargetReferences retageting
	
	@Inject ExecutionGuard executionGuard
	@Inject EventProcessing eventProcessing
	@Inject extension ReactMethod reactMethod
	@Inject SuperStep superStep
	@Inject MicroStep microStep
	@Inject CompletionEvent completionEvent
	@Inject EnterMethod enterMethod
	@Inject ExitMethod exitMethod
	@Inject RunCycleMethod runCycleMethod
	@Inject extension EventBuffer eventBuffer
	@Inject TriggerWithoutEventMethod triggerWithoutEventMethod
	@Inject HaltBehavior haltBehavior
	@Inject Customizations customizations
	
	@Inject extension IFlowConfiguration config
	
	
	/* ==========================================================================
	 * TRANSFORMATION ROOT
	 */ 
	override ExecutionFlow transform(Statechart sc) {
		transform(sc, new ListBasedValidationIssueAcceptor)
	}

	override ExecutionFlow transform(Statechart sc, IValidationIssueAcceptor acceptor) {
		
		val ef = sc.create
		
		// derive configuration from statechart
		config.defineConfigurationForStatechart(sc)
		
		// during mapping the basic structural elements will be mapped from the source statechart to the execution flow
		sc.mapScopes(ef)
		sc.mapRegularStates(ef)
		sc.mapPseudoStates(ef)
		sc.mapRegions(ef)
		sc.mapTimeEvents(ef)

		// calculate state vectors
		ef.defineStateVector(sc)
		ef.defineHistoryVector(sc)
		
		completionEvent.defineFeatures(ef)
		
		// derive all additional information that is necessary for the execution
		sc.mapEntryActions(ef)
		sc.mapExitActions(ef)
		
		 
		ef.defineEnterSequences(sc)
		ef.defineStateExitSequences(sc)
		ef.defineDeepEnterSequences(sc)
		ef.defineShallowEnterSequences(sc)
		ef.defineStatechartEnterSequence(sc)
		ef.defineStatechartExitSequence(sc)

		ef.defineStatechartInitSequence(sc)
		ef.defineStatechartStaticInitSequence(sc)
		
		sc.mapLocalReactions(ef)
		
		ef.declareReactMethods
		
		sc.mapTransitions(ef)
		sc.mapEntries(ef)
		sc.mapChoiceTransitions(ef)
		sc.mapSyncTransitions(ef)

	
		sc.defineEntryReactions(ef)
		ef.definePseudoStateReactions(sc)
		
		
		
		ef.defineReactMethods
		
		// retarget declaration refs
		ef.retargetDeclRefs

		// map own out events to local events
		sc.mapLocalOutEvents(ef)
		
		// map submachine out events to shadow events
		sc.mapReferencedMachineOutEvents(ef)

		// transform structural extensions
		customizations.defineFeatures(ef)
		
		// define event buffer
		ef.defineEventBuffer(sc)
		eventProcessing.transformEventAccess(ef)
		
		
		haltBehavior.defineFeatures(ef)
		eventProcessing.defineFeatures(ef)
		executionGuard.defineFeatures(ef)
		microStep.defineFeatures(ef)
		superStep.defineFeatures(ef)
		runCycleMethod.defineFeatures(ef)
		enterMethod.defineFeatures(ef)
		exitMethod.defineFeatures(ef)
		triggerWithoutEventMethod.defineFeatures(ef)
		

		//clear create caches to avoid memory leak with repetetive generator cycles
		mapping.cleanup
		
		return ef
	}

	/************** retarget declaration refs **************/
	
		 	
}
