/**
 * Copyright (c) 2019-2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 *
 * Contributors:
 * 	Rene Beckmann - itemis AG
 * 	Mathias Alexander Schulte- itemis AG
 * 	Axel Terfloth - itemis AG
 *
 */
package com.yakindu.yet.sct.generator.c

import com.google.inject.Inject
import com.yakindu.sct.generator.c.extensions.FileNaming
import com.yakindu.sct.generator.c.extensions.GenmodelEntries
import com.yakindu.sct.generator.core.artifacts.IContentTemplate
import com.yakindu.sct.generator.core.artifacts.IGenArtifactConfigurations
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sgen.GeneratorEntry

class YetSource implements IContentTemplate<ExecutionFlow> {
	@Inject extension GenmodelEntries
	@Inject protected extension FileNaming
	@Inject extension YETNaming
	
	
	override content(ExecutionFlow it, GeneratorEntry entry, IGenArtifactConfigurations locations) {
		'''
			«entry.licenseText»
			
			«includes(entry, locations)»
			
			«defines»
			
			«declarations»
			
			«implementations»
		'''
	}
	
	def protected includes(ExecutionFlow flow, GeneratorEntry entry, extension IGenArtifactConfigurations locations){
		'''
			#include "«(yetModule.h).relativeTo(yetModule.c)»"
			#include "«(typesModule.h).relativeTo(yetModule.c)»"
			#include <inttypes.h>
			#include <stdio.h>
			#include <string.h>
			#include <stdlib.h>
		'''
	}
	
	def protected defines() '''
		#ifndef SC_UNUSED
		#define SC_UNUSED(P) (void)(P)
		#endif

		/* The minimum message size: start char '%' or '#', a comma, a newline character and a null terminator*/
		#define YET_MIN_MESSAGE_SIZE 4
	'''

	def protected declarations(){
		'''
			static void create_message(char* buf, struct yet_message * msg);
			static void create_init_message(char* buf, char* key, char* value);
			static void create_update_message(char* buf, yet_timestamp timestamp, char* key, char* value);
			
			static yet_msize get_msg_len_init(char* key, char* value);
			static yet_msize get_msg_len_update(yet_timestamp timestamp, char* key, char* value);
			static yet_msize get_char_count_timestamp(yet_timestamp data);
			
			static yet_error read_message(char* text, yet_message* msg);
			static yet_error read_timestamp(char* text, char** endptr, yet_message * msg);
			static yet_error read_key(char* text, char** endptr, yet_message * msg);
			static yet_error read_value(char* text, struct yet_message * msg);
		'''
	}
	
	def protected implementations(ExecutionFlow it){
		'''
			void yet_scope_send(yet_scope* self, struct yet_message * msg)
			{
				char buf[YET_SCOPE_SEND_BUF_LEN];
				yet_msize len = yet_message_len(msg);
				if (self->parent_scope != sc_null) {
					yet_scope_send(self->parent_scope, msg);
				} else {
					if ( len <= YET_SCOPE_SEND_BUF_LEN) {
						create_message(buf, msg);
						sc_observable_sc_string_next(&(self->trace_messages), buf);
					}
				}
			}
			
			static void create_init_message(char* buf, char* key, char* value)
			{
				sprintf(buf, "%c%s,%s\n", YET_MESSAGE_INIT_BEGIN, key, value);
			}
			
			static void create_update_message(char* buf, yet_timestamp timestamp, char* key, char* value)
			{
				if(value != NULL) {
					sprintf(buf, "%c%"PRIu64",%s,%s\n", YET_MESSAGE_UPDATE_BEGIN, timestamp, key, value);
				} else {
					sprintf(buf, "%c%"PRIu64",%s\n", YET_MESSAGE_UPDATE_BEGIN, timestamp, key);
				}
			}
			
			static void create_message(char* buf, struct yet_message * msg)
			{
				if(msg->type == INIT) {
					create_init_message(buf, msg->key, msg->value);
				} else if(msg->type == UPDATE) {
					create_update_message(buf, msg->timestamp, msg->key, msg->value);
				}
			}
			
			static yet_msize get_msg_len_init(char* key, char* value)
			{
				yet_msize count = YET_MIN_MESSAGE_SIZE;
				count += (yet_msize) strlen(key);
				if (value != sc_null) {
					count += (yet_msize) strlen(value);
				}
				return count;
			}
			
			static yet_msize get_msg_len_update(yet_timestamp timestamp, char* key, char* value)
			{
				yet_msize count = YET_MIN_MESSAGE_SIZE;
				count += (yet_msize) get_char_count_timestamp(timestamp);
				count += (yet_msize) strlen(key);
				if (value != NULL) {
					count += (yet_msize) (strlen(value) + 1); /* For needed additional comma separator */
				}
				return count;
			}
			
			yet_msize yet_message_len(yet_message* msg)
			{
				if(msg->type == INIT) {
					return get_msg_len_init(msg->key, msg->value);
				} else if(msg->type == UPDATE) {
					return get_msg_len_update(msg->timestamp, msg->key, msg->value);
				}
				return 0;
			}
			
			#define D2 10ul
			#define D3 100ul
			#define D4 1000ul
			#define D5 10000ul
			#define D6 100000ul
			#define D7 1000000ul
			#define D8 10000000ul
			#define D9 100000000ul
			#define D10 1000000000ul
			#define D11 10000000000ul
			#define D12 100000000000ul
			#define D13 1000000000000ul
			#define D14 10000000000000ul
			#define D15 100000000000000ul
			#define D16 1000000000000000ul
			#define D17 10000000000000000ul
			#define D18 100000000000000000ul
			#define D19 1000000000000000000ul
			#define D20 10000000000000000000ul
			
			static yet_msize get_char_count_timestamp(yet_timestamp n)
			{
				if (n<D11)
					if (n<D6)
						if (n<D3)
							if (n<D2) return 1;
							else return 2;
						else
							if (n<D4) return 3;
							else
								if (n<D5) return 4;
								else return 5;
					else
						if (n<D8)
							if (n<D7) return 6;
							else return 7;
							  else
							      if (n<D9) return 8;
							      else
							      	if (n<D10) return 9;
							      	else return 10;
				else
					if (n<D16)
						if (n<D13)
							if (n<D12) return 11;
							else return 12;
						else
							if (n<D14) return 13;
							else
								if (n<D15) return 14;
								else return 15;
					else
						if (n<D18)
							if (n<D17) return 16;
							else return 17;
						else
							if (n<D19) return 18;
							else
								if (n<D20) return 19;
								else return 20;
			}
			
			static yet_error read_message(char* text, yet_message* msg)
			{
				char* endptr = NULL;
				char begin;
				yet_error e;
			
				if(text == NULL || msg == NULL) {
					return YET_ERR_NULLPTR;
				}
			
				begin = *text;
				if(begin != YET_MESSAGE_INIT_BEGIN && begin != YET_MESSAGE_UPDATE_BEGIN) {
					return (YET_ERR_INVALID_MSG | YET_ERR_INVALID_BEGIN);
				}
			
				text++;
				if(begin == YET_MESSAGE_UPDATE_BEGIN) {
					msg->type = UPDATE;
					e = read_timestamp(text, &endptr, msg);
					if(e != 0) {
						return (YET_ERR_INVALID_MSG | e);
					}
					text = endptr + 1;
				} else {
					msg->type = INIT;
				}
			
				e = read_key(text, &endptr, msg);
				if(e != 0) {
					return (YET_ERR_INVALID_MSG | e);
				}
			
				text = endptr;
				if(begin == YET_MESSAGE_INIT_BEGIN && *text == '\0') {
					return (YET_ERR_INVALID_MSG | YET_ERR_MISSING_VALUE);
				}
			
				if(*text == YET_MESSAGE_SEPARATOR) {
					*text = '\0';
					e = read_value(text + 1, msg);
				} else {
					msg->value = NULL;
				}
				if(*text == '\n') {
					*text = '\0';
				}
				if(e != 0) {
					return (YET_ERR_INVALID_MSG | e);
				}
				return 0;
			}
			
			static yet_error read_timestamp(char* text, char** endptr, yet_message * msg)
			{
				msg->timestamp = strtoul(text, endptr, 10);
				/*
				 * strtoul saves the address of the first character that is not a digit
				 * in endptr. If endptr == text, something went wrong / no digits were found.
				 */
				if(*endptr != text) {
					return 0;
				}
			
				return YET_ERR_INVALID_TIMESTAMP;
			}
			
			static yet_error read_key(char* text, char** endptr, yet_message * msg)
			{
				int i;
				if(text == 0 || *text == '\0') {
					return YET_ERR_INVALID_KEY;
				}
				i = 0;
				msg->key = text;
				while(text[i] != YET_MESSAGE_SEPARATOR && text[i] != '\0' && text[i] != '\n') {
					i++;
				}
				if(i != 0) {
					*endptr = text + i;
					return 0;
				}
				return YET_ERR_INVALID_KEY;
			}
			
			static yet_error read_value(char* text, struct yet_message * msg)
			{
				int i;
				
				if(text == 0 || *text == '\0' || (*text == '\n' && *(text+1) == '\0')) {
					return YET_ERR_INVALID_VALUE;
				}
				i = 0;
				msg->value = text;
				while(text[i] != '\0') {
					i++;
				}
				if(text[i-1] == '\n') {
					/* Remove message delimiting newline character */
					text[i-1] = '\0';
				}
				if(i != 0) {
					return 0;
				}
				return YET_ERR_INVALID_VALUE;
			}
			
			void yet_scope_accept_message(yet_scope *scope, char *text)
			{
				struct yet_message msg;
				read_message(text, &msg);
				yet_scope_dispatch_message(scope, &msg, msg.key);
			}
			
			void yet_scope_dispatch_message(yet_scope* scope, yet_message* msg, char *fqn)
			{
				char *next;
				unsigned int prefix_len;
				
				if(scope->name != 0) {
					next = strchr(fqn, '.');
					if(next == fqn || next == 0) {
						return;
					}
					prefix_len = (unsigned int) (next - fqn);
					if(strncmp(scope->name, fqn, prefix_len) != 0) {
						return;
					}
					next++;
				} else {
					next = fqn;
				}
				
				if(scope->num_child_scopes > 0) {
					int i; 
					for(i = 0; i < scope->num_child_scopes; i++) {
						yet_scope_dispatch_message(scope->child_scopes[i], msg, next);
					}
				} else if(scope->handler != 0) {
					scope->handler(scope, msg, next);
				}
			}
			
			char* yet_scope_create_message(yet_scope* scope, char* msg, char* dst, sc_boolean meta)
			{
				if(scope->parent_scope != 0) {
					dst = yet_scope_create_message(scope->parent_scope, 0, dst, bool_false);
				}
				if(scope->name != 0 && strlen(scope->name) > 0) {
					unsigned long name_len;
					
					name_len = strlen(scope->name);
					strncpy(dst, scope->name, name_len);
					dst += name_len;
					if(meta){
						*dst = '@';
					}else{
						*dst = '.';
					}
					dst++;
				}
				if(msg != 0) {
					strcpy(dst, msg);
					dst += strlen(msg);
					*dst = 0;
				}
				return dst;
			}
			
			void yet_scope_init(yet_scope* scope, yet_scope* parent_scope, yet_scope** child_scopes, sc_integer num_child_scopes, char *name, yet_handler handler, void* instance)
			{
				scope->parent_scope = parent_scope;
				scope->name = name;
				scope->handler = handler;
				scope->instance = instance;
				scope->child_scopes = child_scopes;
				scope->num_child_scopes = num_child_scopes;
			
				sc_single_subscription_observer_sc_string_init(&(scope->message_receiver), scope, (sc_observer_next_sc_string_fp) &yet_scope_accept_message);
			
				sc_observable_sc_string_init(&(scope->trace_messages));
			}
			
			yet_value_serializer yet_int_serializer_function(size_t size, sc_boolean is_signed) {
				if (size == sizeof(int8_t)) {
					if (is_signed) {
						return yet_serialize_int_8;
					} else {
						return yet_serialize_uint_8;
					}
				} else if (size == sizeof(int16_t)) {
					if (is_signed) {
						return yet_serialize_int_16;
					} else {
						return yet_serialize_uint_16;
					}
				} else if (size == sizeof(int32_t)) {
					if (is_signed) {
						return yet_serialize_int_32;
					} else {
						return yet_serialize_uint_32;
					}
				} else if (size == sizeof(int64_t)) {
					if (is_signed) {
						return yet_serialize_int_64;
					} else {
						return yet_serialize_uint_64;
					}
				}
				return yet_serialize_int_32;
			}
			
			yet_value_serializer yet_real_serializer_function(size_t size) {
				if (size == sizeof(float)) {
					return yet_serialize_float;
				} else if (size == sizeof(double)) {
					return yet_serialize_double;
				}
				return sc_null;
			}
			
			char* yet_serialize_void(const void* from, char* to)
			{
				SC_UNUSED(from);
				SC_UNUSED(to);
				
				return sc_null;
			}
			
			char* yet_serialize_bool(const void* from, char* to) {
				bool boolValue;
				
				SC_UNUSED(to);
				
				boolValue = *((sc_boolean*)from);
				if (boolValue) return "t";
				else return "f";
			}
			
			char* yet_serialize_float(const void* from, char* to) {
				float floatValue = *((float*)from);
				sprintf(to, "%f", floatValue);
				return to;
			}
			
			char* yet_serialize_double(const void* from, char* to) {
				double doubleValue = *((double*)from);
				sprintf(to, "%f", doubleValue);
				return to;
			}
			
			char* yet_serialize_string(const void* from, char* to) {
				SC_UNUSED(to);
				return (sc_string)from;
			}
			
			char* yet_serialize_int_8(const void* from, char* to) {
				int8_t intValue = *((int8_t*)from);
				sprintf(to, "%"PRId8, intValue);
				return to;
			}
			
			char* yet_serialize_int_16(const void* from, char* to) {
				int16_t intValue = *((int16_t*)from);
				sprintf(to, "%"PRId16, intValue);
				return to;
			}
			
			char* yet_serialize_int_32(const void* from, char* to) {
				int32_t intValue = *((int32_t*)from);
				sprintf(to, "%"PRId32, intValue);
				return to;
			}
			
			char* yet_serialize_int_64(const void* from, char* to) {
				int64_t intValue = *((int64_t*)from);
				sprintf(to, "%"PRId64, intValue);
				return to;
			}
			
			char* yet_serialize_uint_8(const void* from, char* to) {
				uint8_t intValue = *((uint8_t*)from);
				sprintf(to, "%"PRIu8, intValue);
				return to;
			}
			
			char* yet_serialize_uint_16(const void* from, char* to) {
				uint16_t intValue = *((uint16_t*)from);
				sprintf(to, "%"PRIu16, intValue);
				return to;
			}
			
			char* yet_serialize_uint_32(const void* from, char* to) {
				uint32_t intValue = *((uint32_t*)from);
				sprintf(to, "%"PRIu32, intValue);
				return to;
			}
			
			char* yet_serialize_uint_64(const void* from, char* to) {
				uint64_t intValue = *((uint64_t*)from);
				sprintf(to, "%"PRIu64, intValue);
				return to;
			}
		'''
	}
}
