/**
 * Copyright (c) 2020-2026 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter

import com.google.common.collect.Sets
import com.google.inject.Inject
import com.google.inject.Singleton
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.AssignmentOperator
import com.yakindu.base.expressions.expressions.BitwiseAndExpression
import com.yakindu.base.expressions.expressions.BitwiseOrExpression
import com.yakindu.base.expressions.expressions.BitwiseXorExpression
import com.yakindu.base.expressions.expressions.BoolLiteral
import com.yakindu.base.expressions.expressions.ConditionalExpression
import com.yakindu.base.expressions.expressions.DoubleLiteral
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventValueReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.FloatLiteral
import com.yakindu.base.expressions.expressions.IntLiteral
import com.yakindu.base.expressions.expressions.LogicalAndExpression
import com.yakindu.base.expressions.expressions.LogicalNotExpression
import com.yakindu.base.expressions.expressions.LogicalOrExpression
import com.yakindu.base.expressions.expressions.LogicalRelationExpression
import com.yakindu.base.expressions.expressions.NullLiteral
import com.yakindu.base.expressions.expressions.NumericalAddSubtractExpression
import com.yakindu.base.expressions.expressions.NumericalMultiplyDivideExpression
import com.yakindu.base.expressions.expressions.NumericalUnaryExpression
import com.yakindu.base.expressions.expressions.ParenthesizedExpression
import com.yakindu.base.expressions.expressions.PostFixUnaryExpression
import com.yakindu.base.expressions.expressions.PrimitiveValueExpression
import com.yakindu.base.expressions.expressions.ShiftExpression
import com.yakindu.base.expressions.expressions.StringLiteral
import com.yakindu.base.expressions.expressions.TypeCastExpression
import com.yakindu.base.expressions.interpreter.base.IInterpreter
import com.yakindu.base.expressions.interpreter.base.SRuntimeFunction
import com.yakindu.base.expressions.interpreter.base.ValueSemantics
import com.yakindu.base.expressions.interpreter.context.IExecutionSlotResolver
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.Event
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Type
import com.yakindu.base.types.typesystem.GenericTypeSystem
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionEvent
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ExecutionVariable
import com.yakindu.sct.model.sruntime.ReferenceSlot
import java.util.Optional
import java.util.Set
import org.eclipse.emf.ecore.EObject

/**
 * 
 * @author andreas muelder - Initial contribution and API 
 * @author axel terfloth - additions
 * 
 */
@Singleton
class DefaultExpressionInterpreter extends AbstractExpressionInterpreter implements IExpressionInterpreter {

	@Inject
	protected extension ITypeSystem ts;
	@Inject
	protected extension IExecutionSlotResolver resolver
	@Inject
	protected extension ValueSemantics valueSemantics
	@Inject
	protected extension SRuntimeFunction
	

	@Inject(optional=true)
	protected Set<IOperationExecutor> operationExecutors = Sets.newHashSet

	@Inject(optional=true)
	protected ExecutionContext context

	@Inject
	protected extension ExpressionExtensions

	override evaluate(Expression statement, ExecutionContext context) {
		this.context = context
		statement.execute()
	}

	def dispatch Object execute(Expression statement) {
		null
	}

	def dispatch Object execute(ConditionalExpression expression) {
		if (expression.condition.value as Boolean) {
			return expression.trueCase.value
		} else {
			return expression.falseCase.value
		}
	}

	def protected Object value(Expression expression) {
		expression.execute.asValue	
	}
	
	def dispatch Object execute(BitwiseAndExpression expression) {
		executeBinaryCoreFunction(expression.leftOperand, expression.rightOperand, CoreFunction::BIT_AND)
	}

	def dispatch Object execute(BitwiseOrExpression expression) {
		executeBinaryCoreFunction(expression.leftOperand, expression.rightOperand, CoreFunction::BIT_OR)
	}

	def dispatch Object execute(BitwiseXorExpression expression) {
		executeBinaryCoreFunction(expression.leftOperand, expression.rightOperand, CoreFunction::BIT_XOR)
	}

	def dispatch Object execute(LogicalRelationExpression expression) {
		executeBinaryCoreFunction(expression.leftOperand, expression.rightOperand, expression.operator.getName())
	}

	def dispatch Object execute(NumericalAddSubtractExpression expression) {
		executeBinaryCoreFunction(expression.leftOperand, expression.rightOperand, expression.operator.literal)
	}

	def dispatch Object execute(NumericalMultiplyDivideExpression expression) {
		executeBinaryCoreFunction(expression.leftOperand, expression.rightOperand, expression.operator.getName())
	}

	def dispatch Object execute(ShiftExpression expression) {
		executeBinaryCoreFunction(expression.leftOperand, expression.rightOperand, expression.operator.getName())
	}

	def dispatch Object execute(NumericalUnaryExpression expression) {
		executeUnaryCoreFunction(expression.operand, expression.operator.getName())
	}

	def dispatch Object execute(PostFixUnaryExpression it) {
		var result = operand.value
		val slot = context.resolveSlot(operand).orElseThrow(SlotResolutionExceptionSupplier.forContext(operand))

		slot.value = evaluate(operator.getName(), result)
		result
	}

	def executeBinaryCoreFunction(Expression leftStatement, Expression rightStatement, String operator) {
		var leftResult = leftStatement.value
		var rightResult = rightStatement.value
		return evaluate(operator, leftResult, rightResult)
	}

	def dispatch Object execute(LogicalAndExpression expression) {
		var leftResult = value(expression.leftOperand)
		if (!leftResult as Boolean)
			return false
		var rightResult = value(expression.rightOperand)
		return leftResult as Boolean && rightResult as Boolean
	}

	def dispatch Object execute(LogicalOrExpression expression) {
		var leftResult = value(expression.leftOperand)
		if (leftResult as Boolean)
			return true
		var rightResult = value(expression.rightOperand)
		return leftResult as Boolean || rightResult as Boolean
	}

	def dispatch Object execute(LogicalNotExpression expression) {
		return ! (expression.operand.value() as Boolean)
	}

	protected def Object resolveReference(Object element) {
		if (element instanceof ReferenceSlot) {
			return element.reference
		}
		return element
	}

	def dispatch Object execute(AssignmentExpression assignment) {
		executeAssignment(assignment)
	}

	def dispatch Object execute(TypeCastExpression expression) {
		var operand = expression.operand.value
		typeCast(operand, expression.type.originType)
	}

	def Object cast(Object value, Type type) {
		if (type !== null) {
			typeCast(value, type.originType)
		} else {
			value
		}
	}

	def protected dispatch Object typeCast(Long value, Type type) {
		if(type instanceof EnumerationType) return type.enumerator.get(value.intValue);
		if(ts.isSuperType(type, ts.getType(GenericTypeSystem.INTEGER))) return value
		if(ts.isSuperType(type, ts.getType(GenericTypeSystem.REAL))) return Double.valueOf(value)
		throw new IllegalArgumentException("unknown type " + type.name)
	}

	def protected dispatch Object typeCast(Float value, Type type) {
		if(ts.isSuperType(type, ts.getType(GenericTypeSystem.INTEGER))) return value.longValue
		if(ts.isSuperType(type, ts.getType(GenericTypeSystem.REAL))) return Double.valueOf(value)
		throw new IllegalArgumentException("Invalid cast from Float to " + type.name)
	}

	def protected dispatch Object typeCast(Double value, Type type) {
		if(ts.isSuperType(type, ts.getType(ITypeSystem.INTEGER))) return value.longValue
		if(ts.isSuperType(type, ts.getType(ITypeSystem.REAL))) return Double.valueOf(value)
		throw new IllegalArgumentException("Invalid cast from Double to " + type.name)
	}

	def protected dispatch Object typeCast(Boolean value, Type type) {
		if(ts.isSuperType(type, ts.getType(ITypeSystem.BOOLEAN))) return value
		throw new IllegalArgumentException("Invalid cast from Boolean to " + type.name)
	}

	def protected dispatch Object typeCast(String value, Type type) {
		if(ts.isSuperType(type, ts.getType(ITypeSystem.STRING))) return value
		throw new IllegalArgumentException("Invalid cast from String to " + type.name)
	}

	def protected dispatch Object typeCast(Enumerator value, Type type) {
		if(ts.isSuperType(type, value.owningEnumeration)) return value
		if(ts.isSuperType(type, ts.getType(ITypeSystem.INTEGER))) return (value.literalValue as long)
		
		throw new IllegalArgumentException("Invalid cast from Enumerator to " + type.name)
	}

	def protected dispatch Object typeCast(Object value, Type type) {
		value
	}

	def Object executeAssignment(AssignmentExpression assignment) {
		val scopeVariable = context.resolveSlot(assignment.varRef).orElseThrow(
			SlotResolutionExceptionSupplier.forContext(assignment.varRef))
		var result = assignment.expression.value
		var targetProperty = assignment.varRef.featureOrReference

		if (scopeVariable instanceof CompositeSlot) {
			return setValue(scopeVariable, result, targetProperty)
		} else if (assignment.operator == AssignmentOperator::ASSIGN) {
			return setValue(scopeVariable, if(result !== null) cast(result, scopeVariable.type) else null, targetProperty)
		} else {
			var operator = assignFunctionMap.get(assignment.operator.getName())
			var value = if (result !== null)
				cast(evaluate(operator, scopeVariable.value, result), scopeVariable.type)
			else
				null
				
			return setValue(scopeVariable, value, targetProperty)
		}
	}

	def protected setValue(ExecutionSlot slot, Object value, EObject target) {
		valueSemantics.setValue(slot, value)
		value
	}
	
	def dispatch Object execute(ParenthesizedExpression e) {
		e.expression.execute()
	}

	def dispatch Object execute(PrimitiveValueExpression expression) {
		return expression.value.valueLiteral
	}

	def dispatch valueLiteral(IntLiteral literal) {
		return literal.value as long
	}

	def dispatch valueLiteral(BoolLiteral bool) {
		return bool.value
	}

	def dispatch valueLiteral(DoubleLiteral literal) {
		return literal.value
	}

	def dispatch valueLiteral(FloatLiteral literal) {
		return literal.value
	}

	def dispatch valueLiteral(StringLiteral literal) {
		return literal.value
	}

	def dispatch valueLiteral(NullLiteral literal) {
		return IInterpreter.NULL
	}

	def dispatch Object execute(ElementReferenceExpression expression) {
		executeElementReferenceExpression(expression)
	}

	def executeElementReferenceExpression(ElementReferenceExpression expression) {
		val slot = context.resolveSlot(expression)
		return doExecute(expression.reference, slot.orElse(null), expression)
	}
	
	def dispatch Object execute(EventValueReferenceExpression expression) {
		for (event : context.raisedEvents) {
			val slot = context.resolve(expression.value)
				.orElseThrow(SlotResolutionExceptionSupplier.forContext(expression.value))
			if (slot instanceof ExecutionEvent && slot.fqName == event.fqName) {
				return event.getValue;
			}
		}
		throw new UndefinedValueException("Undefined value of event '" + expression.value.eventName + "'\n" +
			"Event values only exist in the same cycle in which the event was raised.")
	}
	
	def dispatch protected getEventName(ElementReferenceExpression it) {
		if (reference instanceof Event) {
			return (reference as Event).name
		}
		return "null"
	}
	
	def dispatch protected getEventName(Expression it) {
		return "null"
	}

	def dispatch protected getEventName(FeatureCall it) {
		if (feature instanceof Event) {
			return (feature as Event).name
		}
		return "null"
	}

	def dispatch Object execute(FeatureCall call) {
		executeFeatureCall(call)
	}

	def executeFeatureCall(FeatureCall call) {
		var Object result
		for (ArgumentExpression exp : call.toCallStack) {
			val slot = context.resolveSlot(exp)
			result = doExecute(exp.featureOrReference, slot.orElse(null), exp)
		}
		return result
	}

	def dispatch doExecute(EObject feature, Void slot, ArgumentExpression exp) {
		// fall-back
		println("No implementation found for " + exp + " -> returning null")
		null
	}

	def dispatch doExecute(EObject feature, ExecutionVariable slot, ArgumentExpression exp) {
		slot.value
	}

	def dispatch doExecute(EObject feature, CompositeSlot slot, ArgumentExpression exp) {
		slot
	}

	def dispatch doExecute(EObject feature, ExecutionEvent slot, ArgumentExpression exp) {
		slot.raised
	}

	def dispatch doExecute(Operation feature, ExecutionEvent slot, ArgumentExpression exp) {
		slot.raised = true
	}

	def dispatch doExecute(Operation feature, Void slot, ArgumentExpression exp) {
		val executor = operationExecutors.findFirst[canExecute(exp, context)]
		if (executor !== null) {
			return executor.executeOperation(exp)
		}
	}

	def dispatch doExecute(Operation feature, ExecutionSlot slot, ArgumentExpression exp) {
		val executor = operationExecutors.findFirst[canExecute(exp, context)]
		if (executor !== null) {
			slot.value = executor.executeOperation(exp)
		}
		return slot.value
	}

	def dispatch doExecute(Operation feature, CompositeSlot slot, ArgumentExpression exp) {
		val executor = operationExecutors.findFirst[canExecute(exp, context)]
		if (executor !== null) {
			return executor.executeOperation(exp)
		}
		return slot
	}

	def dispatch doExecute(Enumerator feature, Void slot, ArgumentExpression exp) {
		return feature
	}

	def dispatch doExecute(Type feature, Void slot, ArgumentExpression exp) {
		null
	}

	def executeUnaryCoreFunction(Expression statement, String operator) {
		var result = statement.value()
		return evaluate(operator, result);
	}

	def executeOperation(IOperationExecutor executor, ArgumentExpression expression) {
		executor.execute(expression, context)
	}

	def protected Optional<ExecutionSlot> resolveSlot(ExecutionContext context, Expression expression){
		context.resolve(expression)	
	}
	
	protected static class UndefinedValueException extends IllegalStateException {

		new(String message) {
			super(message)
		}

	}
}
