/**
 * Copyright (c) 2020-2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter.base

import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionEvent
import com.yakindu.sct.model.sruntime.ExecutionVariable
import com.yakindu.sct.model.sruntime.ReferenceSlot
import com.yakindu.sct.model.sruntime.SRuntimeFactory
import java.util.ArrayDeque
import java.util.ArrayList
import java.util.Deque
import java.util.List
import org.eclipse.xtend.lib.annotations.Accessors
import org.eclipse.xtend.lib.annotations.Data
import org.eclipse.xtext.xbase.lib.Functions.Function0

import static extension com.yakindu.base.expressions.interpreter.base.InterpreterAdapter.*
import com.yakindu.sct.model.sruntime.ExecutionArgument
import com.yakindu.base.expressions.interpreter.base.IInterpreter.Result
import org.eclipse.xtend.lib.annotations.AccessorType
import com.yakindu.base.expressions.interpreter.base.IInterpreter.Context
import java.util.Optional

/**
 * Implements the core interpreter engine based on sruntime concepts.
 * It implements the actual processing loop for executions.
 * 
 * @author axel terfloth
 */
abstract class SRuntimeInterpreter implements IInterpreter, IInterpreter.Control, IInterpreter.Context, InterpreterControl.OperationCallback {

	public static final String HEAP_SLOT_NAME = "heap"
	public static final String JOBS_SLOT_NAME = "#jobs"
	public static final String GLOBALS_SLOT_NAME = "#globals"
	
	
	protected extension SRuntimeFactory runtimeFactory = SRuntimeFactory.eINSTANCE
	protected ExecutionContext heap
	protected CompositeSlot jobsSlot
	protected CompositeSlot globalsSlot
	
	protected boolean debug = false
	
	protected boolean isProcessing = false
	protected int externalCallDepth = 0
	
	protected InterpreterControl control
	
	
	new() {
		control = new InterpreterControl => [
			it.operationCallback = this
			it.enter
		]
	}
	
	/*********************************************************************************************
	 * 
	 * PUBLIC IInterpreter API 
	 * 
	 ***/
	
	
	override evaluate(Object program) {
		evaluate(null, program)
	}
	
	
	override evaluate(Object instance, Object program) {
		process("_evaluate_", [
			if(instance !== null) defineVariable(InstanceSemantics.SELF_NAME, instance)
			program.prepareExecution
		])
	}

	/**
	 * The interpreter requires that a program which was passed for evaluation will be prepared for execution. 
	 * This actually means that the program must be translated into executions which can be processed by the interpreter.
	 * This translation must be provided by subclasses.
	 */
	def abstract void prepareExecution(Object program);
	
	
	/*********************************************************************************************
	 * 
	 * PUBLIC IInterpreter.Control API 
	 * 
	 ***/
	
	override resume() {
		control.raiseResume
		doProcess()
		if ( suspended ) throw new InterpreterSuspendedException
	}
	
	override suspend() {
		control.raiseSuspend
	}
	
	override terminate() {
		control.raiseTerminate
	}
	
	
	/*********************************************************************************************
	 * 
	 * PUBLIC IInterpreter.Context API 
	 * 
	 ***/
	
	override popValue() {
		val v = stack.peek
		if (v instanceof Frame) {
			throw new InterpreterException("No value on stack.")			
		}
		stack.pop
		logDebug['''«ident»    < «v.description» «stack.map[description]»''']				
		return v 
	}
	
	
	override pushValue(Object value) {
		if ( value !== null ) {
			stack.push(value)
			if(value instanceof Frame) {
				logDebug['''«ident»>> «value.description»  «stack.map[description]»''']
				control.currentJob.frameCount = control.currentJob.frameCount+1
				
			} else {
				logDebug['''«ident»    > «value.description»  «stack.map[description]»''']		
			}
		} else {
			throw new InterpreterException("missing value")
		}
	}
	
	override void _launch(String description, Runnable r) {
		nextExecutions.add(_promise('''launch<«description»>''', [
			launch(description, r)
			null
		]))
	}
	
	override _async(String description, Runnable r) {
		val result = new JobResult
		nextExecutions.add(_promise('''async<«description»>''', [
			result.provider = launch(description, r)
			result.context = this
			null
		]))
		return result;
	}
	
	override _await(Result result, String description, (Object)=>void onResult) {
		if (result instanceof JobResult) {
			(result as JobResult).onResult = _promise('''result<«description»>''', [
				null
			])._then(onResult)
			
			result.receiver = control.currentJob
			
			nextExecutions.add(_promise('''await<«description»>''', [
				control.jobs.raiseWaiting(result.receiver)
				syncJobs
				null
			]))
		}
	}
	
	override _provide(Result result) {
		if (result instanceof JobResult) {
			if (result.receiver !== null) {
				nextExecutions.add(_promise('''provide<«description»>''', [
					control.jobs.raiseContinuing(result.receiver)
					syncJobs
					null
				]))
			}
		}
	}
	
	
	override void _execute(String description, Runnable r) {
		nextExecutions.add(_promise(description, [ 
			r.run 
			null
		]))
	}
		
	override void _return(String description, ()=>Object f) {
		nextExecutions.add( _promise(description, f)._then[ v | 
			v.pushValue 		
		] )
	}
	
	override void _try(String description, Runnable doTry, (Throwable)=>void doCatch) {
		nextExecutions.add(_promise("begin try", [ 
			enterTry(description, doCatch)
			_execute("try", doTry)
			_execute("end try", [
				exitTry
			])
			null
		]))
	} 

	override void _throw(Throwable throwable) {
		
		var Object stackValue;
		
		do {
			stackValue = popAny
		} while ((! (stackValue instanceof TryFrame)) && (!stack.empty))
		
		if ( stackValue instanceof TryFrame ) {
			val tryFrame = stackValue
			_execute("catch", [
				tryFrame.catcher.apply(throwable)
			])			
		} else {
			throw throwable
		}
	}

	override enterCall(String opName) {
		val frame = new StackFrame
		frame.name = opName
		pushFrame(frame)
	}
	
	override exitCall(Object returnValue) {
		do {} while (! (popAny instanceof StackFrame)) 
		if (returnValue !== null) pushValue(returnValue)
	}
	
	
	override <T extends Object> T externalCall(Function0<T> f) {
		try {
			externalCallDepth += 1
			return f.apply
		} finally {
			externalCallDepth -= 1
		}
	}
	
	def protected isInExternalCall() {
		externalCallDepth > 0
	}
	
	override  defineVariable(String name, Object value) {
		_variable(name, value) => [
			localScope.slots.add(it)
		]
	}
	
	def protected ExecutionArgument _variable(String name, Object value) {
		createExecutionArgument => [ v |
			v.name = name
			v.value = value
		]
	}
	
	
		
	
	/*********************************************************************************************
	 * 
	 * maintaining HEAP memory
	 * 
	 ***/

	override ExecutionContext getHeap(){
		return heap
	}

	def setHeap(ExecutionContext ctx){
		heap = ctx
	}	
	
	
	override cleanupMemory() {
		this.jobsSlot = createCompositeSlot => [
			name = JOBS_SLOT_NAME
			visible = true
		]
		this.globalsSlot = createCompositeSlot => [
			name = GLOBALS_SLOT_NAME
			visible = true
		]
		this.heap = createExecutionContext => [
			name = HEAP_SLOT_NAME
			fqName = HEAP_SLOT_NAME
			it.adaptInterpreter(this)
			slots += jobsSlot 
			slots += globalsSlot
		]
		resetStatistics
	}
	
	
	/*********************************************************************************************
	 * 
	 * MAIN PROCESSING
	 * 
	 ***/
	
	def synchronized Object process(String name, Runnable action) {
		
		val job = new Job => [
			it.name = name
			it.onStart = [			
				enterCall('''process<«name»>''')
				action.run
			]
		]
		control.jobs.raiseInvoked(job)
		doProcessJob
	}
	
	
	def synchronized Job launch(String name, Runnable action) {
		
		val job = new Job => [
			it.name = name
			it.onStart = [			
				enterCall('''launch<«name»>''')
				action.run
			]
		]
		control.jobs.raiseLaunched(job)
		syncJobs
		
		return job
	}
	
	
	def protected doProcessJob() {
		control.currentJob.start
		doProcess
		
		if ( suspended ) throw new InterpreterSuspendedException
		
		return control.jobsDone.peekLast.stack?.peek
	}
	
	
	def protected void doProcess() {
		if (isProcessing && ! isInExternalCall) throw new IllegalStateException("Invalid recursive invocation of processing loop.")
		
		try {
			isProcessing = true
			
			// main job loop
			var canProcessNextJob = canDoProcess
			while (canProcessNextJob) {
				
				// activate and prepare job execution
				val pickedJob = control.currentJob
				logDebug['''«ident»process job <«pickedJob.name»>''']
				pickedJob.start
				control.work.raiseBegin
				activateNextExecutions
				
				var switchedJob = false
				
				// main frame loop
				var canProcessNextFrame = currentFrame !== null
				while (canProcessNextFrame) {
					
					// execution loop
					// processes all executions of a frame until
					// no further executions exist or a termination condition applies
					var canProccesExecutions = hasExecutionItem
					while (canProccesExecutions) {
						
						// process next execution
						val head = currentFrame.executionQueue.pop
						logDebug['''«ident»«head.description»''']
						head.execute
						instructionCount++
						
						// if the execution control state switched to a non processing state
						// the execution must be suspended
						if ( ! canDoProcess ) {
							control.work.raiseSuspended
							return 
						}
						
						// handle a potential job change that occurred during execution
						switchedJob = pickedJob !== control.currentJob
						if(switchedJob) {
							logDebug['''«ident»interrupted job <«pickedJob.name»>''']
						}
						 
						if (switchedJob && isInExternalCall)
							throw new IllegalStateException("Invalid asynchronous job processing within synchronous processing loop.")
						
						if (!switchedJob)
							activateNextExecutions
						
						// determine if next execution can be processed
						canProccesExecutions = hasExecutionItem && !switchedJob
						
					}
					
					// if the execution o the current frame is complete exit the call
					if (currentFrame !== null && !hasExecutionItem) 
						exitCall(if (stack.peek instanceof StackFrame) null else popValue)
						
					// determine if next frame can be processed
					canProcessNextFrame = currentFrame !== null && !switchedJob
				}
				
				// mark current job as done
				if (!switchedJob) {
					control.work.raiseDone
				}
				
				// if this processing loop is executed within an external call
				// then it was indirectly recursively called from another execution of this loop. 
				// In this recursion only a single job must be executed.
				canProcessNextJob = canDoProcess && !isInExternalCall
			}
			
			syncJobs
			logDebug['''processed «instructionCount» instructions''']		
		} catch (Throwable e) {
			e.printStackTrace
			throw new InterpreterException('''
				Failed to execute '«currentFrame?.executionQueue?.head?.description»' 
				due to '«e.message»' 
				in «stack?.filter(StackFrame)?.map[description]»''', e)	
		} finally {
			isProcessing = false
		}
	}
	
	def protected boolean canDoProcess() {
		control.isStateActive(InterpreterControl.State.MAIN_BUSY_R_PROCESS_INVOCATION)
	}
	
	def protected boolean suspended() {
		control.isStateActive(InterpreterControl.State.MAIN_BUSY_R_SUSPENDED)
	}
	
	
	/*********************************************************************************************
	 * 
	 * DATA MODEL ACCESS
	 * 
	 ***/
	
	def protected currentJob() {
		control.currentJob
	}
	
	def protected stack() {
		currentJob?.stack
	}
	
	def protected nextExecutions() {
		currentJob?.executions
	}
	
	def protected void activateNextExecutions() {
		logDebug[ if (nextExecutions.empty) null else '''«ident»    «nextExecutions.map[description]» + «currentFrame.executionQueue.map[description]»''']

		nextExecutions.reverseView.forEach[currentFrame?.executionQueue?.push(it)]
		nextExecutions.clear	
	}
	
	def StackFrame currentStackFrame()  {
		(stack?.findFirst[ it instanceof StackFrame ] as StackFrame)	
	}
	
	def protected ExecutionContext localScope()  {
		currentStackFrame?.variables	
	}
	
	def CompositeSlot SELF() {
		localScope?.resolve(InstanceSemantics.SELF_NAME).value as CompositeSlot
	}
	
	override getSelf() {
		SELF
	}
	
	def Frame currentFrame()  {
		(stack?.findFirst[ it instanceof Frame ] as Frame)	
	}
	
	def protected pushFrame(Frame value) {
		if ( value !== null ) {
			stack.push(value)
			logDebug['''«ident»>> «value.description»  «stack.map[description]»''']
			currentJob.frameCount = currentJob.frameCount+1
		} else {
			throw new InterpreterException("missing value")
		}
	}
	
	def protected popAny() {
		val v = stack.pop
		if(v instanceof Frame) {
			control.currentJob.frameCount = control.currentJob.frameCount-1
			logDebug['''«ident»<< «v.description» «stack.map[description]»''']
		} else {
			logDebug['''«ident»    < «v.description» «stack.map[description]»''']		
		}
		return v
	}
	
	def protected enterTry(String opName, (Throwable)=>void doCatch) {
		val frame = new TryFrame
		frame.name = opName
		frame.catcher = doCatch
		pushFrame(frame)
	}
	
	def protected exitTry() {
		val frame = stack.findFirst[it instanceof TryFrame]
		if (frame !== null) {
			stack.remove(frame)

			control.currentJob.frameCount = control.currentJob.frameCount-1
			logDebug['''«ident»<< «frame.description» «stack.map[description]»''']
		}
	}	
	
	def protected hasExecutionItem() {
		(! (currentFrame ===null || currentFrame.executionQueue.empty))
	}
	
	def protected void syncJobs() {
		val List<Job> jobs = new ArrayList
		
		jobs += control.jobsDone
		if (control.currentJob !== null) jobs += control.currentJob
		jobs += control.jobsReady
		jobs += control.jobsWaiting
		
		jobsSlot.slots.clear
		jobs.forEach[ job |
			jobsSlot.slots += createExecutionVariable => [
				name = job.name
				value = job
			]
		]
	}
	
	
	/*********************************************************************************************
	 * 
	 * INTERNAL EXECUTION DATA MODEL 
	 * 
	 ***/
	
	/**
	 * Event instances represent concrete occurrences of events. At any point of time
	 * multiple instances for each defined event may exist. 
	 * 
	 */
	@Data static class EventInstance {

		public ExecutionEvent event;
		public Object value;

		new(ExecutionEvent ev, Object value) {
			this.event = ev
			this.value = value
		}
	}
	
	
	/**
	 * Promises are the basic executable entity. Every executable model code is wrapped into a promise. 
	 * So the the code is executed as a sequence of promises processed by the core processing loop.
	 */
	static class Promise {

		protected extension (Object)=>void then = null
		protected ()=>Object execution = null
		protected String description
		
		def String getDescription() {
			description
		}
		
		def void setThen((Object)=>void then) {
			this.then = then
		}		

		def (Object)=>void getThen() {
			return this.then
		}		
		
		def void setExecution(()=>Object execution) {
			this.execution = execution
		}
		
		def ()=>Object getExecution() {
			return this.execution
		}
		
		def execute() {
			val result = execution.apply
			if (then !== null) then.apply(result)
		}		
	}
	
	/** Factory for a promise instance. */
	def protected _promise(String desc, Function0<Object> f) {
		new Promise => [
			execution = f
			description = desc
		]
	}
	
	/** Factory for the promises then part. */
	def protected _then(Promise it, (Object)=>void then) {
		it.then = then
		return it
	}
	
	
	/**
	 * Frame is the base class required to keep track of hierarchical execution scopes. These execution frames are required 
	 * to handle e.g. function call hierarchies.
	 * 
	 * Each execution frame has a descriptive name and an execution queue of {@link SruntimeInterpreter$Promise} instances
	 */
	static class Frame {
		public String name
		protected Deque<Promise> executionQueue = new ArrayDeque<Promise>
	}
	
	/**
	 * A stack frame is created for each invoked function. 
	 */
	static class StackFrame extends Frame {
		public ExecutionContext variables = SRuntimeFactory.eINSTANCE.createExecutionContext 
	}

	/**
	 * TryFrame instances are used to maintain try catch frames on the call stack and as such enable exception handling
	 */
	static class TryFrame extends Frame {
		public (Throwable)=>void catcher
	}
	
	
	/**
	 * Jobs are asynchronously executed entities which are created on external execution requests as defined by 
	 * {@link IInterpreter}. Additionally Jobs can be launched and controlled using {@link IInterpreter$Context} methods:
	 * <ul>
	 * 	<li>{@link IInterpreter$Context#_launch}
	 * 	<li>{@link IInterpreter$Context#_async}
	 * 	<li>{@link IInterpreter$Context#_await}
	 * 	<li>{@link IInterpreter$Context#_provide}
	 * </ul>
	 * 
	 * Each job owns a stack which stores data and execution frames. So jobs are logically independent entities of execution
	 * which are comparable to threads. Nevertheless the default execution schema is the sequential execution of jobs.
	 * Additionally jobs may be chained via the <code>previous</code> reference for the case of job execution in the
	 * context of interpreter recursion which may occur if external objects are called by the interpreter which 
	 * again call the interpreter. 
	 * 
	 * Jobs may be require results from other jobs so it is possible that jobs a paused and continued when the required results 
	 * are available. So each job maintains its own state and also defined the possible state transitions. 
	 */
	static class Job {
		enum State {
			NEW,
			EXECUTING,
			AWAITING,
			READY,
			DONE
		}
		@Accessors String name
		@Accessors(AccessorType.PRIVATE_GETTER) State state = State.NEW
		@Accessors Runnable onStart
		@Accessors Runnable onContinue
		@Accessors Deque<Object> stack = new ArrayDeque<Object>
		@Accessors List<Promise> executions = new ArrayList<Promise>()
		@Accessors Job previous = null
		@Accessors int frameCount = 0
		
		def void start() {
			if ( state == State.NEW ) {
				state = State.EXECUTING
				onStart.run
			} else if (state == State.READY) {
				state = State.EXECUTING
				onContinue.run
			}
		}
		
		def void awaiting() {
			if (state == State.EXECUTING) {
				state = State.AWAITING
			}
		}
		
		def void done() {
			if (state == State.EXECUTING) {
				state = State.DONE
			}
		}
		
		def void ready() {
			if (state == State.AWAITING) {
				state = State.READY
			}
		}
		
		override toString() {
			state.name
		}
	}
	
	/**
	 * Implementation of {@link Result} instances of asynchronous executions which are based on the used {@link Job} concept.
	 * There are two job instances involved. First the which was started and is the 'provider' of the result and second
	 * the job that launched the first job and is the 'receiver' of the result. 
	 */
	static class JobResult implements Result {
		@Accessors Job provider
		@Accessors Job receiver
		@Accessors Promise onResult
		@Accessors IInterpreter.Context context
		
		protected Optional<Object> value = Optional.empty
		
		override provide(Object value) {
			this.value = Optional.of(value)
			if (receiver !== null) {
				receiver.onContinue = [
					// TODO : are this both steps redundant? Do we always want to pass the value via stack?
					context?.pushValue(value)
					onResult.then.apply(value)
				]
			}
			
			context._provide(this)
		}
		
		override value() {
			this.value
		}
		
	}
	
	
	/*********************************************************************************************
	 * 
	 * STATISTICS 
	 * 
	 ***/
	
	protected int instructionCount = 0

	override void resetStatistics() {
		instructionCount = 0			
	}
	
	
	/*********************************************************************************************
	 * 
	 * LOGGING
	 * 
	 ***/
	
	def dispatch description(Object it) {
		toString
	}
	
	def dispatch description(StackFrame it) {
		'''frame<«name»>'''
	}
	
	def dispatch description(TryFrame it) {
		'''try<«name»>'''
	}
	
	def dispatch description(ExecutionVariable it) {
		'''var<«name»>'''
	}
	
	def dispatch description(ExecutionEvent it) {
		'''event<«name»>'''
	}
	
	def dispatch description(CompositeSlot it) {
		'''obj<«name»>'''
	}
	
	def dispatch description(ReferenceSlot it) {
		'''ref<«name»>'''
	}
	
	def dispatch description(ExecutionArgument it) {
		'''arg<«name»>'''
	}
	
	
	def dispatch description(Null it) {
		'''null'''
	}
	
	def ident() {
		ident(control.currentJob.frameCount)
	} 
	
	def create new StringBuffer ident(int n) {
		for(i : 0..<n) it.append('''    ''')	
	}
	
	def void logDebug(()=>Object it) {
		if(debug) {
			val output = it.apply
			if (output !== null) System.out.println(it.apply)
		}
	}
}
