/**
 * Copyright (c) 2020-2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter.base

import com.google.inject.Inject
import com.yakindu.base.expressions.interpreter.base.IInterpreter.Null
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.typesystem.ITypeSemantics
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.model.sruntime.CompositeExecutionEvent
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionArgument
import com.yakindu.sct.model.sruntime.ExecutionEvent
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ExecutionVariable
import com.yakindu.sct.model.sruntime.ReferenceExecutionEvent
import com.yakindu.sct.model.sruntime.ReferenceSlot
import com.yakindu.sct.model.sruntime.SRuntimeFactory

/**
 * Defines how values are derived from object which may be values or value holders 
 * like execution slots.
 * 
 * TODO: we have to rework the execution slot structure. A major problem is the missing separation of 
 * slots and values especially for CompositeSlot structures.
 * 
 * @author axel terfloth - Initial contribution and API  * 
 */
class ValueSemantics {

	@Inject(optional=true) extension IInstanceFactory factory
	@Inject extension ITypeSystem
	@Inject extension ITypeSemantics

	def dispatch Object asValue(Object slot) {
		slot
	}

	def dispatch Object asValue(Enumerator slot) {
		slot
	}

	def dispatch Object asValue(ExecutionVariable slot) {
		slot.value
	}

	def dispatch Object asValue(ExecutionArgument slot) {
		slot.value
	}

	def dispatch Object asValue(ExecutionEvent slot) {
		slot.raised
	}

	def dispatch Object asValue(CompositeExecutionEvent slot) {
		if (slot.name.endsWith("@value") || slot.name.endsWith("@copy"))
			slot
		else
			slot.raised
	}

	def dispatch Object asValue(ReferenceExecutionEvent slot) {
		if (slot.name.endsWith("@value"))
			slot
		else
			slot.raised
	}

	/**
	 * TODO: this method is called whenever an instance is processed via the stack. 
	 * This leads to non required copies which will be copied later on. We have to review 
	 * and fix the semantics.
	 */
	def dispatch Object asValue(CompositeSlot slot) {
		if ( slot.type?.isValueType ) {
			assertFactoryAvailable
			slot.copyInstance
		} else {
			SRuntimeFactory.eINSTANCE.createReferenceSlot => [
				reference = slot
			]
		}
	}
	
	def dispatch Object asValue(ReferenceSlot slot) {
		assertFactoryAvailable
		slot.copyInstance
	}
	
	def dispatch void setValue(Object o, Object value) {
		throw new InterpreterException("Cannot assign value of type " + value.class.simpleName + " to slot of type " +
			o.class.name);
	}

	def dispatch void setValue(ExecutionVariable it, Object value) {
		it.value = value
	}

	def dispatch void setValue(ExecutionArgument it, Object value) {
		it.value = value
	}

	def dispatch void setValue(ExecutionVariable it, ExecutionSlot value) {
		it.value = value.value
	}

	def dispatch void setValue(ExecutionEvent it, Void value) {
		it.value = null
	}

	def dispatch void setValue(ExecutionEvent it, Object value) {
		it.value = value
	}

	def dispatch void setValue(ExecutionEvent it, ExecutionSlot value) {
		it.value = value.value
	}

	def dispatch void setValue(ExecutionEvent it, ExecutionEvent value) {
		it.value = value.value
	}

	def dispatch void setValue(CompositeExecutionEvent it, CompositeSlot value) {
		it.takeValuesFrom(value)
	}

	def dispatch void setValue(CompositeSlot it, CompositeSlot value) {
		it.takeValuesFrom(value)
	}

	protected def void takeValuesFrom(CompositeSlot it, CompositeSlot value) {

		if (it.type.isArray) {
			it.slots.clear
			value.slots.forEach [ s, i |
				it.slots += (factory.copyInstance(s) as ExecutionSlot) => [
					name = '''[«i»]'''
					fqName = name
				]
			]
		} else if (type.isMap) {
			slots.clear
			value.slots.forEach [ s, i |
				slots += (factory.copyInstance(s) as ExecutionSlot) => [
					name = s.name
					fqName = s.fqName
				]
			]
		} else
			it.slots.forEach [ s |
				val other = value.slots.findFirst[s.name == it.name || s.fqName == it.name]
				if (other !== null) {
					setValue(s, other)
				}
			]
	}

	def dispatch void setValue(ReferenceSlot it, ReferenceSlot value) {
		it.reference = value.reference
	}

	def dispatch void setValue(ReferenceSlot it, CompositeSlot value) {
		it.reference = value
	}

	def dispatch void setValue(ReferenceSlot it, Null value) {
		it.reference = null
	}

	def dispatch void setValue(ReferenceExecutionEvent it, ReferenceSlot value) {
		it.reference = value.reference
	}

	def dispatch void setValue(ReferenceExecutionEvent it, CompositeSlot value) {
		it.reference = value
	}

	def dispatch void setValue(ReferenceExecutionEvent it, Null value) {
		it.reference = null
	}

	def protected void assertFactoryAvailable() {
		if (factory === null) {
			throw new InterpreterException("Cannot copy complex values - factory is missing");
		}

	}
}
