/**
 * Copyright (c) 2025-2026 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.concepts

import com.google.inject.Inject
import com.google.inject.Singleton
import com.itemis.create.base.model.bindings.BindingsTypeLibrary
import com.itemis.create.base.model.bindings.PropertyChangedNotification
import com.itemis.create.base.model.bindings.ReactiveModel
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.InitializationExpression
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Property
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.TypedDeclaration
import com.yakindu.base.types.annotations.ClassificationAnnotations
import com.yakindu.base.types.annotations.TypeAnnotations
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.extensions.SexecTypeSemantics
import com.yakindu.sct.model.sexec.transformation.SexecElementMapping
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.stext.stext.VariableDefinition
import java.util.ArrayList
import java.util.HashMap
import java.util.List
import java.util.Map
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.EcoreUtil2
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.sct.model.stext.stext.ActiveStateReferenceExpression
import com.yakindu.base.expressions.expressions.ThisExpression

/**
 * Defines the concepts for property binding. It transforms 'bind' calls whithin an onInit hook.
 * - for each bind call a binder is created
 * - for each source element of all binders a notification mechanism will be added
 * - locally defined source properties directly trigger binder updates on change
 * - externally defined source properties must be out properties and observers will be added which trigger the binders.
 * 
 * TODO: refactor in two parts - one only dependent on types and the second specific for statecharts.
 * 
 * @author axel terfloth
 */
@Singleton
class PropertyBinder {
	
	@Inject protected extension TypeBuilder
	@Inject protected extension ExpressionBuilder
	@Inject protected extension BindingsTypeLibrary
	@Inject protected extension OnInitHook
	@Inject protected extension OnUpdateStateHook
	@Inject protected extension ExpressionExtensions
	@Inject protected extension SexecElementMapping
	@Inject protected extension ReactiveModel
	@Inject protected extension SexecOutProperty
	@Inject protected extension PropertyChangedNotification
	@Inject protected extension TypeAnnotations
	@Inject protected extension ClassificationAnnotations
	@Inject protected extension SexecTypeSemantics
	@Inject protected extension OriginTracing
	
	
	def create it : _complexType binderType() {
		name = "Binder"
		annotateAsReferenceType
		val lambda = _variable("lambda", _function)
		features += lambda
		features += _op("update", _void) => [
			_body(
				_block(
					_if (lambda._ref._notEquals(_null))._then(
						lambda._ref
							._call(_function_apply)
					)
				)	
			)
		]
		bindingPackage.member += it
	}
	
	def protected create it : _package("binding") bindingPackage() {
		scPackage.member += it
	}
	
	def protected create it : _package("sc") scPackage() {}
	
	def binderTypeUpdate() {
		binderType.features.filter(Operation).findFirst["update"==name]
	}
	
	def isBinderType(Type type) {
		EcoreUtil.equals(binderType, type)
	}
	
	
	def synchronized void defineFeatures(ExecutionFlow it) {
		if (appliesBinder) {
			defineBinderInstances
			defineBindingObserver
			defineLocalBindingUpdates
			defineOnUpdateStateUpdates
			defineOninitUpdates
		}
	}
	
	def appliesBinder(ExecutionFlow it) {
		onInitHook !== null
		&& !binderCalls.empty
	}
	
	def protected void defineBinderInstances(ExecutionFlow it) {
		binderCalls.forEach[ bc, i |
			if (bc.arguments.size == 2) {
				val target = bc.arguments.get(0).value.copy
				val expression = bc.arguments.get(1).value.copy
				val sources = new ArrayList<Expression>
				
				// collext all property sources
				sources.addAll(
					if (expression.featureOrReference instanceof Property)
						#[expression]
					else
						expression.eAllContents
							.filter(ArgumentExpression)
							.filter[
								! (eContainer instanceof ArgumentExpression)
								&& featureOrReference instanceof Property
							]
					.toList
					.map[copy])
				
				// if there are bindings which do not relate to properties an additional instance binding is required
				if (expression.requiresNonPropertyBinding) {
					sources += _this
				}
				
				// create a lambda which performs the update
				val lambda = _block(
					target.guardExpression(
						target.copy._assign(expression.copy)
					)
				)._lambda
				
				
				// create a binder instance using the calvulated values
				features += _part("_binder_" + i, binderType) => [
					_protected
					_synthetic
					initialValue = _initialization(#[
						lambda
					])
					metaFeatures += _variable("target", _any, target) => [
						_no_instance
					]
					metaFeatures += _variable(
						"sources"
						, _typeSpecifier(_array, _any._typeSpecifier)
						,_initialization(sources) 
					) => [
						_no_instance
					]
				]
			}
		]
	}
	
	def protected dispatch boolean requiresNonPropertyBinding(EObject it) {
		false
	}
	
	def protected dispatch boolean requiresNonPropertyBinding(Expression it) {
		eAllContents.exists[requiresNonPropertyBinding]
	}
	
	def protected dispatch boolean requiresNonPropertyBinding(ActiveStateReferenceExpression it) {
		true
	}
	
	def protected dispatch boolean requiresNonPropertyBinding(ArgumentExpression it) {
		!(eContainer instanceof ArgumentExpression) && !(featureOrReference instanceof Property)
	}
	
	def protected dispatch Expression guardExpression(Expression it, Expression guarded) {
		return guarded
	}
	
	def protected dispatch Expression guardExpression(ElementReferenceExpression it, Expression guarded) {
		if (it.reference.typedDeclaration?.type.isReferenceType) {
			_if (it.copy._notEquals(_null))
				._then(_block( guarded ))
		}
		else guarded
	}
	
	def protected dispatch Expression guardExpression(FeatureCall it, Expression guarded) {
		it.owner.guardExpression(
			if (it.feature.typedDeclaration?.type.isReferenceType) {
				_if (it.copy._notEquals(_null))
					._then(_block( guarded ))
			}
			else guarded
		)
	}
	
	def TypedDeclaration typedDeclaration(EObject it) {
		if (it instanceof TypedDeclaration) return it
		return null
	}
	
	def protected binderInstances(ExecutionFlow it) {
		features
			.filter(Property)
			.filter[type.isBinderType()]
			.toList
	}
	
	def protected binderTarget(Property it) {
		metaFeatures
			.filter(Property)
			.findFirst["target"==name]
			.initialValue
	}
	
	def protected binderSources(Property it) {
		metaFeatures
			.filter(Property)
			.findFirst["sources"==name]
			.initialValue
			.asList
	}
	
	def protected bindingObserver(ExecutionFlow it) {
		features
			.filter(Property)
			.filter[type.isSimpleObserverType && name.startsWith("_binding_observer_")]
			.toList
	}
	
	
	def protected dispatch List<Property> buildPath(Expression it) {
		emptyList
	}
	
	def protected dispatch List<Property> buildPath(ElementReferenceExpression it) {
		if (reference instanceof Property)
			#[reference as Property]
		else
			emptyList
	}
	
	def protected dispatch List<Property> buildPath(FeatureCall it) {
		val f = feature
		if (f instanceof VariableDefinition) {
			val f_flow = f.create =>[traceOrigin(f.origin)]
			val p = new ArrayList<Property>()
			p.addAll(owner.buildPath)
			p += f_flow
			return p
		}
		else
			emptyList
	}
	
	
	
	def protected void defineBindingObserver(ExecutionFlow flow) {
		val List<List<Property>> sourcePaths = new ArrayList
		val Map<List<Property>, List<Property>> sourcePathToBindersMap = new HashMap
		val Map<List<Property>, Expression> sourcePathToExpressionMap = new HashMap
		
		flow.binderInstances.forEach[ binder |
			binder.binderSources.forEach[ se |
				val source = se.featureOrReference
				val sourcePath = se.buildPath
				val sourceChart = EcoreUtil2.getContainerOfType(source, Statechart)
				if (sourceChart !== flow.sourceElement) {
					if (source instanceof VariableDefinition) {
						val binders = sourcePathToBindersMap.get(sourcePath)
						if (binders === null) {
							sourcePaths += sourcePath
							sourcePathToBindersMap.put(sourcePath, new ArrayList => [it+=binder])
							sourcePathToExpressionMap.put(sourcePath, se)
						} else {
							binders += binder
						}
					}
				}
			]
		]
		
		// create an observer for each property
		sourcePaths.forEach[ s, i |
			val binders = sourcePathToBindersMap.get(s)
			
			val lambda = _block(
				binders.map[ b |
					b._ref._call(binderTypeUpdate)
				]
			)._lambda => [
				parameters += _lambda_param("value", s.lastElement.typeSpecifier.copy)
			]
			
			flow.features += _part("_binding_observer_" + i) => [
				s.forEach[ pathElement |
					pathElement.origin === null ? traceOrigin(pathElement) : traceOrigin(pathElement.origin)
				]
				_protected
				_synthetic
				typeSpecifier = simpleObserverType._typeSpecifier(s.lastElement.typeSpecifier.copy)
				initialValue = _initialization(#[
					lambda
				])
			]
		]
		
		// enforce assumption: each source property is 'out property'
		sourcePaths.forEach[
			if (it.lastElement.outPropertyObservableGetter === null) {
				_sexecOutProperty.defineFeatures(it.lastElement)
			}
		]
		
		// subscribe binding observer
		val observer = flow.bindingObserver
		val block = _block
		observer.forEach[ o, i |
			val source = sourcePaths.get(i)
			val observerSubscribe = o._ref._call(
				simpleObserverTypeSubscribe
			)._with(
				sourcePathToExpressionMap.get(source).copy => [ 
					if (it instanceof FeatureCall) {
						it.feature = source.lastElement.outPropertyObservableGetter
						it.operationCall = true
					}
				]
			)
			block.expressions += observerSubscribe
		]
		// add to beginning of onInitHook implementation
		if (!block.expressions.empty) {
			flow.addOnInitAction(block)
		}
	}
	
	
	def protected <T> lastElement(List<T> l) {
		if (l.size > 0) {
			l.get(l.size-1)
		} else {
			null
		}
	}
	
	
	def protected void defineLocalBindingUpdates(ExecutionFlow flow) {
		flow.binderInstances.forEach[ binder |
			binder.binderSources.forEach[ se |
				val source = se.featureOrReference
				val sourceChart = EcoreUtil2.getContainerOfType(source, Statechart)
				if (sourceChart === flow.sourceElement) {
					if (source instanceof VariableDefinition) {
						val flowSource = source.create
						if (flowSource !== null) {
							flowSource.addChangeNotifictation(binder.name, _block(binder._ref._call(binderTypeUpdate)))
						}
					}
				}
			]
		]
	}
	
	def protected void defineOnUpdateStateUpdates(ExecutionFlow flow) {
		flow.binderInstances.forEach[ binder |
			if ( binder.binderSources.exists[it instanceof ThisExpression] ){
				flow.addOnUpdateStateBehavior(binder._ref._call(binderTypeUpdate))
			}
		]
	}
	
	
	def protected void 	defineOninitUpdates(ExecutionFlow it) {
		val binder = binderInstances
		binderCalls.forEach[ bc, i |
			if (bc.eContainer !== null) {
				val binderUpdate = binder.get(i)._ref._call(binderTypeUpdate)
				EcoreUtil.replace(bc, binderUpdate)
			}
		]
	}
	
	def protected List<ArgumentExpression> create l : new ArrayList binderCalls(ExecutionFlow it) {
		l.addAll(onInitHook.implementation.eAllContents
			.filter(ArgumentExpression)
			.filter[featureOrReference.isBindOperation]
			.toList)
	}
	
	def protected flow(EObject it) {
		EcoreUtil2.getContainerOfType(it, ExecutionFlow)
	}
	def protected dispatch List<Expression> asList(Expression it) {
		#[it]
	}
	
	def protected dispatch List<Expression> asList(InitializationExpression it) {
		it.arguments.map[value].toList
	}
	
	def protected <T extends EObject> T copy(T it) {
		EcoreUtil.copy(it)
	}
}