/**
 * Copyright (c) 2025-2026 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.concepts

import com.google.inject.Inject
import com.google.inject.Singleton
import com.itemis.create.base.generator.core.concepts.Documentation
import com.itemis.create.base.model.core.EventModel
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Event
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Part
import com.yakindu.base.types.Property
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.base.types.annotations.TypeAnnotations
import com.yakindu.base.types.libraries.AbstractTypeLibrary
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sexec.extensions.SexecBuilder
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.util.StatechartUtil
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.stext.SubmachineReferenceExpression
import java.util.List
import org.eclipse.emf.common.util.URI

import static com.itemis.create.base.model.core.ConceptModel.*

/**
 * This class defines the general sub machine concept. It covers structure and behaviour.
 * 
 * @author aterfloth - Initial contribution
 * @author laszlo kovacs - Extensions and implementations
 */
 @Singleton
class SubMachine {
	
	public static val CTX_POSTFIX = "ctx"
	public static val SUBMACHINE_CONTEXT_MEMBER = "_submachineContext_"
	public static val SUBMACHINE_CTX_TYPE = "SubmachineCtx"
	
	public static val TRACE_SUBCYCLE_START = "traceSubCycleStart"
	public static val TRACE_SUBCYCLE_END = "traceSubCycleEnd"
	
	
	@Inject protected extension ExpressionExtensions
	@Inject protected extension SExecExtensions
	@Inject protected SexecBuilder sexec
	
	
	@Inject protected extension StatechartAnnotations
	@Inject protected extension SubmachineTypeLibrary lib

	@Inject protected extension TypeBuilder
	@Inject protected extension ExpressionBuilder exprBuilder
	
	@Inject protected extension EventModel
	@Inject protected extension MicroStep
	@Inject protected extension Documentation
	@Inject protected extension RunCycleMethod
	@Inject protected extension StatechartUtil
	@Inject protected extension SuperStep
	@Inject protected extension OriginTracing
	@Inject protected extension EventProcessing
	@Inject protected extension EventBuffer
	@Inject protected extension EventQueue
	@Inject protected extension TypeAnnotations
	@Inject protected extension OnInitHook	
	
	/* =================================================================
	 * 
	 * QUERIES
	 * 
	 * ================================================================= */
	
	def boolean embedsSubMachine(State state) {
		state.embeddedSubMachines.size > 0
	}
	
	def List<SubmachineReferenceExpression> embeddedSubMachines(State state) {
		state
			.scopes
			.map[members]
			.flatten
			.filter(SubmachineReferenceExpression)
			.filter[
				submachine.featureOrReference.appliesSubMachine
			]
			.toList
	}
	
	def boolean hasReferencedSubmachine(ExecutionFlow it){
		referencedStatecharts.exists[appliesSubMachine]
	}
	
	def dispatch appliesSubMachine(Type it) {
		return false
	}
	
	def dispatch appliesSubMachine(Object it) {
		return false
	}
	
	def dispatch appliesSubMachine(ComplexType it) {
		if(origin instanceof Statechart)
			return (origin as Statechart).isSubMachine
		else
			return false
	}
	
	def dispatch appliesSubMachine(Property it) {
		if(type.origin !== null && type.origin instanceof Statechart)
			return (type.origin as Statechart).isSubMachine
		else
			return false
	}
	
	def dispatch appliesSubMachine(ExecutionFlow it) {
		statechart.isSubMachine
	}
	
	def dispatch appliesSubMachine(Statechart it) {
		isSubMachine
	}
	
	def allSubmachineOperations(ExecutionFlow it){
		#[it.runSubmachineCycle, it.enableNextEvent, it.setSubmachineContext]
	}
	
	def getSubmachineContextPoints(ExecutionFlow flow){
		flow.features.filter(Part).filter[type instanceof ComplexType].filter[submachineContextTypeConcept.appliesTo(type as ComplexType)]
	}
	
	
	/* =================================================================
	 * 
	 * TRANSFORMATIONS
	 * 
	 * ================================================================= */

	def void defineSubmachineFeatures(ExecutionFlow it) {
		if (appliesSubMachine) {
			adaptSubMachineInterface
			runSubmachineCycleConcept.define(it)
			enableNextEventConcept.define(it)
			subMachineContextMemberConcept.define(it)
			setSubmachineContextConcept.define(it)
		}
	}
	
	def void defineSubmachineOwnerFeatures(ExecutionFlow it) {
		if(hasReferencedSubmachine){
			addSubmachineEvents
			submachineContextTypeConcept.define(it)
			addCtxPoints
		}
		
	}
	
	
	def adaptSubMachineInterface(ComplexType it) {
		superTypes += submachineInterface._typeSpecifier
	}
	
	def protected addCtxPoints(ExecutionFlow flow){
		flow.eAllContents.filter(Property).filter[type.appliesSubMachine].forEach[ vd |
//			flow.features += flow.createCtxPointFor(vd)
			submachineContextPointConcept.define(vd)
		]
	}
	
	def protected addSubmachineEvents(ExecutionFlow flow){
		flow.eAllContents.filter(Property).filter[type.appliesSubMachine].forEach[ vd |
			submachineEventConcept.define(vd)
		]
	}

	/* =================================================================
	 * 
	 * CONCEPT DEFINITIONS
	 * 
	 * ================================================================= */
	
	/**
	 * SubmachineEvent concept
	 * 
	 * Each state machine which embeds sub machines defines an event for each sub machine.
	 * This event is used to handle sub machine events properly in the context of the parent machine.
	 */
	var submachineEventConcept = featureConcept(Property, Event, "submachine_event", 
		[ prop |
			_in(_event(prop.name + "Event")) => [
				_protected
				_synthetic
				activateEventConcept.define(event) => [
					_body(
						_block(
							prop._ref._call(submachineInterfaceEnableNextEvent)
						)
					)
				]
			]
		],
		[ flow.features ],
		[ prop, c | c.name == prop.name + "Event"]
	)
	
	def getSubmachineEvent(Property p) {
		submachineEventConcept.definition(p)
	}
	
	/**
	 * SubmachineContextPoint concept
	 * 
	 * For each embedded sub machine a sub machine context instance (context point) is created 
	 * as part of the host state machine.
	 * This event is used to handle sub machine events properly in the context of the parent machine.
	 */
	var submachineContextPointConcept = featureConcept(Property, Part, "submachine_event", 
		[ sub |
			_part(sub.name + "_" + CTX_POSTFIX, sub.flow.submachineContextType) => [
				_protected
				_synthetic
				val flow = sub.flow
				val submachineEvent = sub.submachineEvent
				val lambda = _block(
					submachineEvent._ref._meta(storeEventConcept.define(submachineEvent)),
					if (flow.appliesSubMachine)
						subMachineContextMemberConcept.define(flow)._ref._call(submachineContextInterfaceEventRaised) 
					else 
						_call(flow.runCycle)
				)._lambda
			
				initialValue = _initialization(#[lambda])
				
				val assignContext = sub._ref._call(submachineInterfaceSetSubmachineContext)._with(it._ref)
				flow.addOnInitAction(assignContext)
			]
		],
		[ flow.features ],
		[ prop, c | c.name == prop.name + "_" + CTX_POSTFIX]
	)
	
	def getSubmachineContextPoint(Property vd){
		submachineContextPointConcept.definition(vd)
	}
	
	/**
	 * SubMachineContextType concept
	 * 
	 * Each state machine which embeds sub machines defines a context type.
	 */
	var submachineContextTypeConcept = featureConcept(ExecutionFlow, ComplexType, SUBMACHINE_CTX_TYPE, [ flow |
		_complexType(SUBMACHINE_CTX_TYPE) => [
			_public
			annotateAsReferenceType
			superTypes += getSubmachineContextInterface._typeSpecifier
			
			val onEventRaised = _variable("onEventRaised", _function)
			features += onEventRaised
			features += _op(SubmachineTypeLibrary.EVENT_RAISED, _void) => [
				_body(
					_block(
						_if (onEventRaised._ref._notEquals(_null))._then(
							onEventRaised._ref
								._call(_function_apply)._with(_null)
						)
					)	
				)
			]
			
			features += _op(SUBMACHINE_CTX_TYPE) => [
				val lambdaParam = _lambda_param("onEventRaised_", _function)
				parameters += lambdaParam
				_no_return_type
				documentation("Constructor for SumachineContext that implements the interface 'SumachineContextInterface'" )
				implementation = _block(
					onEventRaised._ref._assign(lambdaParam._ref)
				)
			]
		]
	])
	
	def submachineContextType(ExecutionFlow flow){
		submachineContextTypeConcept.definition(flow)
	}
	
	def definesSubmachineContextType(ExecutionFlow flow){
		flow.submachineContextType !== null
	}
	
	def hasEventRaised(ExecutionFlow it) {
		eventRaised !== null
	}
	
	def dispatch eventRaised(ComplexType it) {
		features.filter(Operation).filter( m | m.name == SubmachineTypeLibrary.EVENT_RAISED).head		
	}
	
	def dispatch eventRaised(Type it) {
		null
	}
	
	/**
	 * SubMachineContextMember concept
	 * 
	 * Each sub machine owns a reference to its context which is of type SubMachineContextInterface
	 */
	var subMachineContextMemberConcept = featureConcept(ExecutionFlow, Property, SUBMACHINE_CONTEXT_MEMBER, [
		_variable(SUBMACHINE_CONTEXT_MEMBER, submachineContextInterface) => [
			_public
			_synthetic
			const = true
		]
	])
	
	def getSubmachineContextMember(ExecutionFlow it) {
		subMachineContextMemberConcept.definition(it)
	}
	
	/**
	 * runSubmachineCycle concept
	 * 
	 * Each sub machine implements a 'runSubmachineCycle method.
	 */
	var runSubmachineCycleConcept = featureConcept(ExecutionFlow, Operation, SubmachineTypeLibrary.RUN_SUBMACHINE_CYCLE, [ flow |
		SubmachineTypeLibrary.RUN_SUBMACHINE_CYCLE._op(_boolean) => [
			_public
			documentation("Performs a 'cycle' step in context of the submachine." )
			val impl = _block()
				var traceCycleStart = sexec._conceptSequence(TRACE_SUBCYCLE_START) => [steps += sexec._traceBeginRunCycle]
				impl.expressions += traceCycleStart._ref
				impl.expressions += flow.stateChangedProperty._ref._assign(_false)
				impl.expressions += flow._superStepLoop(
					if(flow.microStep.exists) flow.microStep._call else null
				)
				if(flow.clearInEvents !== null)
					impl.expressions += flow.clearInEvents._call
				var traceCycleEnd = sexec._conceptSequence(TRACE_SUBCYCLE_END) => [steps += sexec._traceEndRunCycle]
				impl.expressions += traceCycleEnd._ref
				impl.expressions += exprBuilder._if(flow.stateChangedProperty._ref)
					._then(_block(_return(_true)))
					._else(_block(_return(_false)))
			implementation = impl			
		]
	])

	def getRunSubmachineCycle(ExecutionFlow it) {
		runSubmachineCycleConcept.definition(it)
	}
	
	/**
	 * enableNextEvent concept
	 * 
	 * Each sub machine implements a 'enableNextEvent' method.
	 */
	var enableNextEventConcept = featureConcept(ExecutionFlow, Operation, SubmachineTypeLibrary.ENABLE_NEXT_EVENT, [ flow |
		SubmachineTypeLibrary.ENABLE_NEXT_EVENT._op(_void) => [
			_public
			documentation("Allow the parent to enable processing the next event of the submachine." )
			var impl = _block()
			//This has to be transformed to actual codemodel element when generating code
			if(flow.requiresEventQueue)
				impl.expressions += sexec._conceptSequence(EventProcessing.NEXT_EVENT)._ref => [operationCall = true]
			implementation = impl
		]
	])
	
	def getEnableNextEvent(ExecutionFlow it) {
		enableNextEventConcept.definition(it)
	}
	
	/**
	 * setSubMachineContext concept
	 * 
	 * Each sub machine implements a setter for its context.
	 */
	var setSubmachineContextConcept = featureConcept(ExecutionFlow, Operation, SubmachineTypeLibrary.SET_SUBMACHINE_CONTEXT, [ flow |
		SubmachineTypeLibrary.SET_SUBMACHINE_CONTEXT._op(_void) => [
			val param = _lambda_param("ctx_", submachineContextInterface._typeSpecifier)
			parameters += param
			_public
			documentation("Sets the parental submachine context." )
			implementation = _block(
				flow.submachineContextMember._ref._assign(param._ref)
			)
		]
	])
	
	def getSetSubmachineContext(ExecutionFlow it) {
		setSubmachineContextConcept.definition(it)	
	}
	
	
	@Singleton
	static class SubmachineTypeLibrary extends AbstractTypeLibrary {
		
		public static val LIBRARY = "create.submachine"
		public static val LIBRARY_URI = URI.createURI(LIBRARY)
		
		public static val SC_PACKAGE_NAME = "sc"
		public static val SUBMACHINE_INTERFACE = "SubmachineInterface"
		public static val RUN_SUBMACHINE_CYCLE = "runSubmachineCycle"
		public static val ENABLE_NEXT_EVENT = "enableNextEvent"
		public static val SET_SUBMACHINE_CONTEXT = "setSubmachineContext"
		
		public static val SUBMACHINE_CONTEXT_INTERFACE = "SubmachineContextInterface"
		public static val EVENT_RAISED = "eventRaised"
		
		@Inject protected extension TypeBuilder
		@Inject protected extension TypeAnnotations
				
		override protected createLibraryPackage() {
			SC_PACKAGE_NAME._package => [
				val subMachineContextInterface = SUBMACHINE_CONTEXT_INTERFACE._complexType => [
					_public
					annotateAsReferenceType
					features += EVENT_RAISED._op(_void)
				]
				
				val subMachineInterface = SUBMACHINE_INTERFACE._complexType => [
					_public
					annotateAsReferenceType
					features += RUN_SUBMACHINE_CYCLE._op(_boolean)
					features += ENABLE_NEXT_EVENT._op(_void)
					features += SET_SUBMACHINE_CONTEXT._op(_void)._param("subCtxIface",subMachineContextInterface._typeSpecifier)
				]
				
				member += subMachineInterface
				member += subMachineContextInterface
			]
		}
		
		def getSubmachineContextInterface() {
			getLibraryPackage().member.findFirst[name == SUBMACHINE_CONTEXT_INTERFACE] as ComplexType
		}

		def getSubmachineContextInterfaceEventRaised() {
			submachineContextInterface.features.findFirst[name == EVENT_RAISED] as Operation
		}
		
		def getSubmachineInterface() {
			getLibraryPackage().member.findFirst[name == SUBMACHINE_INTERFACE] as ComplexType
		}
		
		def getSubmachineInterfaceEnableNextEvent() {
			submachineInterface.features.filter(Operation).findFirst[name == ENABLE_NEXT_EVENT]
		}
		
		def getSubmachineInterfaceRunSubmachineCycle() {
			submachineInterface.features.filter(Operation).findFirst[RUN_SUBMACHINE_CYCLE == name]
		}
		
		def getSubmachineInterfaceSetSubmachineContext() {
			getSubmachineInterface.features.findFirst[name == SET_SUBMACHINE_CONTEXT] as Operation
		}
		
		override getLibraryURI() {
			LIBRARY_URI
		}
		
	} 
}