/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.transformation

import com.google.inject.Inject
import com.google.inject.Singleton
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.ExpressionsFactory
import com.yakindu.base.expressions.expressions.TimeEventSpec
import com.yakindu.base.expressions.expressions.TimeUnit
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.sct.model.sexec.Check
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.Reaction
import com.yakindu.sct.model.sexec.Sequence
import com.yakindu.sct.model.sexec.TimeEvent
import com.yakindu.sct.model.sexec.concepts.ExecutionDebugging
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sexec.extensions.SexecBuilder
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Statechart

/**
 * This class takes care of creation of the method and class member to store the value of time when scheduling a time event.
 * The class also takes care of the ExecFlow transformation to inject the time event related logging expressions.
 * 
 * @author laszlo kovacs - Initial contribution
 */
@Singleton
class MapTimeEventValue {
	@Inject extension SexecElementMapping mapping
	@Inject extension TypeBuilder typeBuilder
	@Inject extension SexecBuilder sexec
	@Inject extension ExpressionBuilder expr
	@Inject protected extension ExecutionDebugging
	@Inject protected extension StatechartExtensions sc
	@Inject protected extension SExecExtensions
	@Inject extension OriginTracing
	
	protected extension ExpressionsFactory expFactory = ExpressionsFactory.eINSTANCE
	
	def maptimeEventValues(Statechart statechart, ExecutionFlow flow){
		val allStates = statechart.eAllContents.filter(State).toList
		allStates.map[timeEventSpecs].flatten.map[createDerivedEvent].forEach[ t |
			flow.declareReactMethod(t)
		]
	}
	
	def addTimeEventLogs(ExecutionFlow flow){
		val timedConditions = flow.eAllContents.filter(Check)
		.filter[
			condition !== null &&
			condition instanceof ElementReferenceExpression &&
			(condition as ElementReferenceExpression).reference instanceof TimeEvent
		]
		timedConditions.forEach[ tc |
			if(tc.eContainer instanceof Reaction &&  (tc.eContainer as Reaction).effect instanceof Sequence)
				((tc.eContainer as Reaction).effect as Sequence).steps.add(0,((tc.condition as ElementReferenceExpression).reference as TimeEvent).logsFor)	
		]
	}
	
	def declareReactMethod(ExecutionFlow node, TimeEvent te) {
		node => [
			features.add(te.scheduleTimeValue)
			features.add(te.defineReactMethod)
		]
	}
	
	def Sequence logsFor(TimeEvent te){
		val loggs = te.debugLogInstance.logFor(te).expressions
		val logSeq = _sequence()
		loggs.forEach[ e |
			logSeq.steps.add(e.copy._statement)
		]
		logSeq
	}
	
	def create it : _variable("te_" + te.name.getLastSegment + "_last", _integer) scheduleTimeValue(TimeEvent te){
		_protected
	}
	
	def create method: sexecFactory.createMethod defineReactMethod(TimeEvent te) {
		method.name = "te_" + te.name.getLastSegment + "_value"
		method._type(_integer)
		method._param("value", _integer)
		// Value is in 'ms' (what the timer awaits)
		val value = method.param('value')
		val timeEventSpec = te.origin as TimeEventSpec
		method => [
			body = _sequence(
				te.scheduleTimeValue._ref._assign(value._ref._divide(timeEventSpec.unit.calculateProceedValue)._cast(_integer._typeSpecifier))._statement,
				_return(te.scheduleTimeValue._ref)._statement
			) => [comment = "Store time event value"]
		]
	}
	
	def static String getLastSegment(String fqn) {
        if (fqn.isNullOrEmpty) {
            return fqn
        }

        val lastDotIndex = fqn.lastIndexOf('.')

        if (lastDotIndex == -1) {
            return fqn
        }

        return fqn.substring(lastDotIndex + 1)
    }
    
    def calculateProceedValue(TimeUnit unit){
    	var s = 1.0
    	if(unit == TimeUnit.SECOND)
    		s = 1000
    	else if(unit == TimeUnit.MICROSECOND)
    		s = 0.001
    	else if(unit == TimeUnit.NANOSECOND)
    		s = 0.000001
    	return s._value
	}
}