/**
 * Copyright (c) 2022-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 
 */
package com.yakindu.sctunit.generator.cpp.extensions

import com.google.inject.Inject
import com.itemis.create.base.generator.core.types.Literals
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.AssignmentOperator
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.EventValueReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.LogicalRelationExpression
import com.yakindu.base.expressions.expressions.StringLiteral
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.TypedElement
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.sct.generator.c.types.CTypes
import com.yakindu.sct.generator.cpp.CppNaming
import com.yakindu.sct.generator.cpp.FeatureCallSeparator
import com.yakindu.sct.model.sexec.naming.INamingService
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.util.StatechartUtil
import com.yakindu.sct.model.stext.stext.ActiveStateReferenceExpression
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.model.stext.stext.VariableDefinition
import com.yakindu.sctunit.generator.c.extensions.CExpressionsExtensions
import com.yakindu.sctunit.sCTUnit.EnterExpression
import com.yakindu.sctunit.sCTUnit.ExitExpression
import com.yakindu.sctunit.sCTUnit.ProceedExpression
import com.yakindu.sctunit.sCTUnit.StatechartActiveExpression
import com.yakindu.sctunit.sCTUnit.StatechartFinalExpression
import com.yakindu.sctunit.sCTUnit.TriggerWithoutEventExpression
import org.eclipse.xtext.util.Strings

/**
 * 
 * @author florian antony 
 */
class CppExpressions extends CExpressionsExtensions {
	
	@Inject protected extension SCTUnitCppNaming
	@Inject protected extension INamingService
	@Inject protected extension ExpressionExtensions
	@Inject protected extension OriginTracing
	@Inject protected extension CppNaming naming
	@Inject protected extension FeatureCallSeparator
	@Inject protected extension StatechartUtil
	
	@Inject extension Literals

	override dispatch CharSequence code(EventValueReferenceExpression it) {
		'''«value.context»«eventDefinition.valueGetter»()'''
	}

	override dispatch CharSequence code(EventRaisingExpression it) {
		'''«event.context»«naming.asRaiser(eventDefinition)»(«IF value !== null»«value.code»«ENDIF»)'''
	}
		
	override dispatch CharSequence code(ActiveStateReferenceExpression it) '''«ref»«isStateActive»(«SCTUnitClass?.statechart.typeName»::«value.shortName.asEscapedIdentifier»)'''

	override generateNotLocalAssignment(AssignmentExpression it) {
		if (needsTempValueAssignment) {
			return tempValueAssignment
		} else {
			if (operator == AssignmentOperator.ASSIGN) {
				return '''«varRef.context»«varRef.valueSetter»(«expression.code»)'''
			} else {
				return '''«varRef.context»«varRef.valueSetter»(«varRef.code» «operator.toString» «expression.code»))'''
			}
			
		}
	}
	
	def protected needsTempValueAssignment(AssignmentExpression it) {
		val cp = varRef.getComplexType
		cp !== null && !(cp.type.isOriginStatechart || cp.type.originTraces.exists[it instanceof InterfaceScope])
	}
	
	def protected tempValueAssignment(AssignmentExpression it) {
		val fc = varRef as FeatureCall
		val index = indexOfAssignmentExpression
		val ownerRef = fc.toCallStack.head
		return
		'''
			«varRef.getComplexType.targetLanguageName» value«index» = «ownerRef.context»«ownerRef.valueGetter»();
			value«index».«fc.fqName» «operator» «expression.code»;
			«ownerRef.context»«ownerRef.valueSetter»(value«index»)'''
	}
		
	override dispatch TypeSpecifier getComplexType (FeatureCall it) {
		if (property !== null) {
			val ref = statechartLocalOwnerRef
			return (ref?.reference as TypedElement)?.typeSpecifier
 		}
	}

	def protected getStatechartLocalOwnerRef(FeatureCall it) {
		var owner = owner
		while (owner instanceof FeatureCall) {
			owner = owner.owner
		}
		if (owner instanceof ElementReferenceExpression) {
			val reference = owner.reference
			if (reference instanceof VariableDefinition) {
				return owner
			}
		}
	}

	override generateNotLocalElemRefExpr(ElementReferenceExpression it) {
		'''«context»«reference.getterName»()'''
	}
	
	def dispatch context(FeatureCall it) {
		'''«owner.code»«owner.callSep»'''
	}
	
	def dispatch context(Expression it) {
		'''«ref»'''
	}

	def dispatch CharSequence code(LogicalRelationExpression it, CTypes.PString pType) 
		'''(«leftOperand.code» «operator.literal» «rightOperand.code»)'''


	
	override dispatch CharSequence code(StatechartActiveExpression exp) {
		'''«ref»«isActiveFunctionName»()''' // TODO: sync with C syntax
	}

	override dispatch CharSequence code(StatechartFinalExpression exp) {
		'''«ref»«isFinalFunctionName»()''' // TODO: sync with C syntax
	}
	
	override dispatch code(Expression owner, EventDefinition feature){
		'''«owner.code»«owner.callSep»«feature.isRaised»()'''
	}
	
	override dispatch code(Expression owner, Enumerator feature){
		if (feature.eContainer.isOriginStateEnum) {
			return feature.stateEnumerator
		}
		return '''«feature.type.namespaceAccess»«feature.access»'''
	}
	
	
	def stateEnumerator(Enumerator it) {
		stateEnumAccess
	}
	
	override dispatch code(Expression owner, VariableDefinition feature){
		// these guys come from a statechart (interface) => access via getter
		'''«owner.code»«owner.callSep»«feature.valueGetter»()'''
	}
	
	override dispatch code(Expression owner, Property feature){
		if (!feature.eContainer.originTraces.filter(InterfaceScope).nullOrEmpty || 
			!feature.eContainer.originTraces.filter(Statechart).nullOrEmpty
		) {
			// these guys come from a statechart (interface) => access via getter
			'''«owner.code»«owner.callSep»«feature.valueGetter»()'''
		} else {
			'''«owner.code»«owner.callSep»«feature.code»'''
		}
	}
	
	override dispatch code(Expression owner, Operation feature) {
		'''«owner.code»«owner.callSep»«feature.name»(«owner.args»)'''
	}
	
	override dispatch code(ElementReferenceExpression owner, Operation feature) {
		'''«owner.code»«owner.callSep»«feature.name»(«owner.args»)'''
	}
	
	override CharSequence args(Expression owner)
		'''«FOR arg : (owner.eContainer as FeatureCall).expressions SEPARATOR ', '»«arg.code»«ENDFOR»'''
	
	override proceedTime(ProceedExpression it) {
		var time_ms = value.toMilliseconds(unit)
		'''runner->proceed_time(«time_ms.code»);'''
	}

	override dispatch code(EnterExpression it) {
		'''«ref»«enterFunctionName»()'''
	}
	
	override dispatch code(TriggerWithoutEventExpression it) {
		'''«ref»«triggerWithoutEventFunctionName»()'''
	}

	override dispatch code(ExitExpression it) {
		'''«ref»«exitFunctionName»()'''
	}

	override proceedCycles(ProceedExpression it) {
		'''runner->proceed_cycles(«value.code»);'''
	}
	
	override dispatch CharSequence code(ElementReferenceExpression it) {
		if (isExpressionVoidOperation) {
			return '''«referenceName»(«FOR expr : expressions SEPARATOR ', '»«expr.code»«ENDFOR»)'''
		}
		if (!reference.isLocal && reference.isStatechartLocal) {
			return generateNotLocalElemRefExpr
		}
		referenceName
	}	
	
	override dispatch code(StringLiteral it) { 
		Strings.convertToJavaString(value.escaped).asLiteral
	}
	
	/**
	 * Expression code generator for mock return expressions.
	 * It handles the mock instances access to test class members.
	 */
	static class MockReturn extends CppExpressions {
	
		override dispatch CharSequence code(ElementReferenceExpression it) 
		'''owner->«super._code(it)»'''
	}
	
}
