/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 	Jonathan Thoene - itemis AG
 */

package com.yakindu.sctunit.generator.java

import com.google.inject.Inject
import com.google.inject.name.Named
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.TypeCastExpression
import com.yakindu.base.types.PrimitiveType
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.sct.generator.core.extensions.StringHelper
import com.yakindu.sct.generator.core.types.ICodegenTypeSystemAccess
import com.yakindu.sct.generator.java.Naming
import com.yakindu.sct.generator.java.files.ITimed
import com.yakindu.sct.generator.java.files.ITimerService
import com.yakindu.sct.generator.java.files.VirtualTimer
import com.yakindu.sct.model.sgen.GeneratorEntry
import com.yakindu.sct.model.sgraph.Scope
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.stext.OperationDefinition
import com.yakindu.sct.types.resource.Statechart2TypeTransformation
import com.yakindu.sctunit.generator.base.ISCTUnitGenerator
import com.yakindu.sctunit.generator.base.extensions.BaseNavigationExtensions
import com.yakindu.sctunit.generator.base.extensions.StaticExtensions
import com.yakindu.sctunit.generator.java.extensions.CallbackExtensions
import com.yakindu.sctunit.generator.java.extensions.IMockingExtensions
import com.yakindu.sctunit.generator.java.extensions.JavaExpressionExtensions
import com.yakindu.sctunit.generator.java.extensions.JavaStatementExtensions
import com.yakindu.sctunit.generator.java.extensions.SCTUnitJavaNaming
import com.yakindu.sctunit.generator.java.features.JavaSCTUnitGenmodelEntries
import com.yakindu.sctunit.sCTUnit.SCTUnitClass
import com.yakindu.sctunit.sCTUnit.SCTUnitElement
import com.yakindu.sctunit.sCTUnit.SCTUnitOperation
import com.yakindu.sctunit.sCTUnit.SCTUnitSuite
import com.yakindu.sctunit.sCTUnit.TestPackage
import com.yakindu.sctunit.sCTUnit.VariableDefinitionStatement
import java.util.List
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtext.generator.IFileSystemAccess

import static com.yakindu.sct.generator.core.filesystem.ISCTFileSystemAccess.*

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * @author Johannes Dicks - Refactoring based on EFS encapsulation
 * @author jonathan thoene - Expanding templates to generate mocking statements 
 * 
 */
class JavaSCTUnitGenerator implements ISCTUnitGenerator {

	public val static String SCTUNIT_INFERRER = "com.yakindu.sctunit.generator.java.JavaSCTUnitGenerator.typeInferrer"

	@Inject extension JavaExpressionExtensions
	@Inject extension StatechartAnnotations
	@Inject extension SCTUnitJavaNaming
	@Inject extension StringHelper
	@Inject extension JavaStatementExtensions
	@Inject extension StaticExtensions
	@Inject extension IMockingExtensions
	@Inject extension ICodegenTypeSystemAccess
	@Inject extension CallbackExtensions
	@Inject extension BaseNavigationExtensions
	@Inject extension JavaSCTUnitGenmodelEntries
	@Inject extension Naming naming
	@Inject extension Statechart2TypeTransformation
	
	@Inject extension VirtualTimer timer
	@Inject extension ITimerService iTimerServiceTemplate
	@Inject extension ITimed iTimedTemplate
	
	@Inject
	@Named(SCTUNIT_INFERRER)
	extension ITypeSystemInferrer;
	
	@Inject extension JavaSubchartInitializer

	GeneratorEntry entry;
	
	override generate(GeneratorEntry entry, IFileSystemAccess fsa) {
		this.entry = entry;

		var testPackage = entry.testPackage
		var fileName = testPackage.fileName
		val element = testPackage.SCTUnitElement
		var content = element.toCode
		
		if (element instanceof SCTUnitClass) {
			if (element.statechart.needsTimer) {
				timer.generateWithStatechart(element.statechart, entry, fsa)
				val timerServiceFileName = entry.statechartLibraryPackage.toPath + '/' + naming.java(iTimerService)
				fsa.generateFile(timerServiceFileName, LIBRARY_TARGET_FOLDER_OUTPUT, iTimerServiceTemplate.content(entry.licenseText, entry.statechartLibraryPackage))
				val iTimedFileName = entry.statechartLibraryPackage.toPath + '/' + naming.java(iTimed)
				fsa.generateFile(iTimedFileName, LIBRARY_TARGET_FOLDER_OUTPUT, iTimedTemplate.content(entry.licenseText, entry.statechartLibraryPackage))
			}
		}
		fsa.generateFile(fileName, content)
	}

	def dispatch protected CharSequence toCode(SCTUnitSuite it) {
		'''
			«getLicenseText(entry)»
			
			package «classPackage»;
			import org.junit.runner.RunWith;
			import org.junit.runners.Suite;
			import org.junit.runners.Suite.SuiteClasses;
			
			@RunWith(Suite.class)
			@SuiteClasses({
				«FOR group : SCTUnitClasses SEPARATOR ','»
					«group.testClassName».class
				«ENDFOR»
				})
			public class «testClassName» {
			}
		'''
	}

	def dispatch protected CharSequence toCode(SCTUnitClass it) {
		'''
			«getLicenseText(entry)»
			
			«IF !classPackage.nullOrEmpty»package «classPackage»;«ENDIF»
			
			«mockingImports(entry)»
			import org.junit.*;
			import static org.junit.Assert.*;
			«IF classPackage != entry.statechartBasePackage»
				import «getStatechartBasePackage(entry).dot(statemachineClassName)»;
				import «getStatechartBasePackage(entry).dot(statemachineClassName)».State;
			«ENDIF»	
			«IF statechart.needsTimer»
				import «getStatechartLibraryPackage(entry)».VirtualTimer;
			«ENDIF»
			«IF statechart.isCycleBased»
				import «getStatechartLibraryPackage(entry)».VirtualTimer.VirtualTimeTask;
				import «getStatechartLibraryPackage(entry)».VirtualTimer.CycleTimeEventTask;
			«ENDIF»
			«IF classPackage != entry.statechartBasePackage»
				«FOR subchartImport : statechart.subchartImports.entrySet»
					import «getStatechartBasePackage(entry, subchartImport.key).dot(subchartImport.value.toString)»;
				«ENDFOR»
			«ENDIF»
			«FOR typeImport : it.importsForReferencedTypes»
				import «typeImport»;
			«ENDFOR»
			
			«author»
			@SuppressWarnings("all")
			public class «testClassName» {
				«FOR scope : statechart.scopes»
					«IF !scope.members.filter(OperationDefinition).nullOrEmpty»«scope.setCallBack»«ENDIF»
				«ENDFOR»
				
				private «statemachineClassName» statemachine;	
				«IF statechart.needsTimer»
					private VirtualTimer timer;
				«ENDIF»
				
				«FOR variable: variableDefinitions»
					protected «variable.definition.type.targetLanguageName» «variable.definition.name»;
				«ENDFOR»
				
				«setup»
			
				@After
				public void «name.toFirstLower»_tearDown() {
					«FOR Scope scope : statechart.scopes»
						«IF !scope.members.filter(OperationDefinition).nullOrEmpty»«scope.unsetOperationCallback»«ENDIF»
					«ENDFOR»
					statemachine = null;
					
					«IF statechart.needsTimer»
						timer = null;
					«ENDIF»
				}
				
				«FOR operation : SCTUnitOperations SEPARATOR "\n"»
					«operation.toCode»
				«ENDFOR»
			}
		'''
	}

	def dispatch protected CharSequence toCode(SCTUnitOperation it) {
		'''
			«IF isIgnore»@Ignore«ENDIF»
			«IF isTest»@Test«ENDIF»
			public «infer.type.targetLanguageName» «methodName»(«generateParameter») {
				«FOR statement : body.code»
					«statement.generate»
				«ENDFOR» 
			}
		'''
	}


	def dispatch protected CharSequence toCode(List<VariableDefinitionStatement> varDefs) '''
		«FOR varDef : varDefs»
			«varDef.toCode»
		«ENDFOR»
	'''		

	
	def dispatch protected CharSequence toCode(VariableDefinitionStatement it) '''
		«generate»
	'''
 	
	def protected setup(SCTUnitClass it) throws Exception {
		'''
			@Before
			public void «name.toFirstLower»_setUp() {
				statemachine = new «statemachineClassName»();
				«IF (statechart.needsTimer)»
					timer = new VirtualTimer(«cyclePeriod»);
				«ENDIF»
				«IF statechart.isCycleBased»
					timer.schedulePeriodicalTask(new CycleTimeEventTask(statemachine), «cyclePeriod», «cyclePeriod»);
				«ENDIF»
				«IF statechart.isTimed»
					statemachine.setTimerService(timer);
				«ENDIF»
				«FOR Scope scope : statechart.scopes»
					«IF !scope.members.filter(OperationDefinition).nullOrEmpty»«scope.mockOperationCallbacks»«ENDIF»
				«ENDFOR»
				
				«statechart.subchartInitSequence»
				
				«FOR variable: variableDefinitions.filter[definition.initialValue !== null]»
					«variable.definition.name» = «variable.definition.initialValue.code»;
				«ENDFOR»
			}
		'''
	}
	
	def protected getFileName(TestPackage it) {
		'''«entry.basePackage.toPath»/«IF !namespace.toString.nullOrEmpty»«namespaceFolder»/«ENDIF»«SCTUnitElement.testClassName».java'''.toString
	}
	
	def protected classPackage(SCTUnitElement it) {
		entry.basePackage.dot(namespace.toString)
	}
	
	
	
	def protected generateParameter(SCTUnitOperation it){
		'''«FOR param:parameters SEPARATOR ', '»«param.type.targetLanguageName» «param.name»«»«ENDFOR»'''
	}
	
	def importsForReferencedTypes(SCTUnitClass it) {
		val statechartImports = statechart.subchartImports.values.map[toString].toSet
		
		eAllContents
			.map[ 
				typesToImport.iterator
			]
			.flatten
			.toSet
			.filter[ 
				!(it instanceof PrimitiveType)
				&& !isStatechartType
				&& !isStatechartStateEnum
				&& !isNamedIface
			]
			.map[ 
				it.fqn
			]
			.filter[
				! statechartImports.contains(it)
			]
			.sort
	}
	
	def protected dispatch String fqn(NamedElement it) {
		val parentName = eContainer.fqn
		if ( parentName.nullOrEmpty ) {
			name
		} else {
			'''«parentName».«name»'''
		}
	}
	
	def protected dispatch String fqn(EObject it) {
		""
	}
	def protected dispatch String fqn(Void it) {
		""
	}
	
	def protected dispatch List<Type> typesToImport(EObject it) {
		#[]
	}

	def protected dispatch List<Type> typesToImport(TypeSpecifier it) {
		typeArguments
			.map[ typesToImport ]
			.flatten
			.toList => [ types | types += type]
	}
	
	def protected dispatch List<Type> typesToImport(ElementReferenceExpression it) {
		(reference instanceof Type) ? #[reference as Type] : #[]
	}
	
	def protected dispatch List<Type> typesToImport(FeatureCall it) {
		(feature instanceof Type) ? #[feature as Type] : #[]
	}
	
	def protected dispatch List<Type> typesToImport(TypeCastExpression it) {
		#[type]
	}
	
}
