/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.json.transformation.xmi2json

import com.yakindu.sct.json.transformation.model.JsonCell
import com.yakindu.sct.json.transformation.model.JsonChoice
import com.yakindu.sct.json.transformation.model.JsonEntry
import com.yakindu.sct.json.transformation.model.JsonExit
import com.yakindu.sct.json.transformation.model.JsonFinal
import com.yakindu.sct.json.transformation.model.JsonGraph
import com.yakindu.sct.json.transformation.model.JsonNodeLabel
import com.yakindu.sct.json.transformation.model.JsonNote
import com.yakindu.sct.json.transformation.model.JsonPosition
import com.yakindu.sct.json.transformation.model.JsonRegion
import com.yakindu.sct.json.transformation.model.JsonRoot
import com.yakindu.sct.json.transformation.model.JsonState
import com.yakindu.sct.json.transformation.model.JsonStatechart
import com.yakindu.sct.json.transformation.model.JsonSynchronization
import com.yakindu.sct.json.transformation.model.JsonTransition
import com.yakindu.sct.json.transformation.model.JsonTransitionAnchor
import com.yakindu.sct.json.transformation.model.JsonTransitionRouter
import com.yakindu.sct.json.transformation.model.JsonTransitionSource
import com.yakindu.sct.json.transformation.model.JsonTransitionTarget
import com.yakindu.sct.json.transformation.model.util.NodeModelFactory
import com.yakindu.sct.json.transformation.model.util.NotationUtil
import com.yakindu.sct.model.sgraph.Choice
import com.yakindu.sct.model.sgraph.Entry
import com.yakindu.sct.model.sgraph.Exit
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Synchronization
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.model.sgraph.Vertex
import java.util.HashMap
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.emf.ecore.xmi.XMLResource
import org.eclipse.gmf.runtime.notation.BooleanValueStyle
import org.eclipse.gmf.runtime.notation.ConnectorStyle
import org.eclipse.gmf.runtime.notation.Diagram
import org.eclipse.gmf.runtime.notation.Edge
import org.eclipse.gmf.runtime.notation.Routing
import org.eclipse.gmf.runtime.notation.Shape
import org.eclipse.gmf.runtime.notation.ShapeStyle
import org.eclipse.gmf.runtime.notation.StringValueStyle

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * @author axel terfloth - fixes and extensions
 * 
 */
class XMIToJSONTransformation {

	extension NotationUtil = new NotationUtil
	extension NodeModelFactory = new NodeModelFactory

	static class Context {
		val jsonGraph = new JsonGraph
		/**
		 * Id map wich is used on purpose if resource is no XMLResource.
		 */
		val idMap = new HashMap<EObject, String>()
		var Diagram diagram

		/**
		 * Provides the ID for an EObject. if it does not exist the id will be created.
		 */
		def protected ID(EObject object) {

			var id = object.eResource.getObjectId(object)

			if (id.nullOrEmpty) {
				id = EcoreUtil.generateUUID
				object.eResource.setObjectId(object, id)
			}

			return id
		}

		def protected dispatch getObjectId(XMLResource r, EObject object) {
			r.getID(object)
		}

		def protected dispatch getObjectId(Resource r, EObject object) {
			idMap.get(object)
		}

		def protected dispatch setObjectId(XMLResource r, EObject object, String id) {
			r.setID(object, id)
		}

		def protected dispatch setObjectId(Resource r, EObject object, String id) {
			idMap.put(object, id)
		}
	}

	def JsonRoot transform(Resource resource) {

		val extension context = new Context
		val statechart = resource.contents.filter(Statechart).head
		context.diagram = resource.contents.filter(Diagram).head

		val jsonStatechart = new JsonStatechart => [
			id = statechart.ID
			name = statechart.name
			specification = statechart.specification
			documentation = statechart.documentation
		]
		context.jsonGraph.cells += jsonStatechart
		statechart.regions.forEach[toJSON(null, context)]

		new JsonRoot => [
			graph = context.jsonGraph
		]
	}

	def protected toJSONNote(Shape note, JsonCell parent, extension Context context) {
		val jsonNote = new JsonNote => [
			name = note.description
			id = note.ID
			parent = parent?.id ?: null
		]
		jsonGraph.cells += jsonNote
		if (parent instanceof JsonRegion)
			parent.embeds += jsonNote.id
		jsonNote.setBounds(note, context.jsonGraph)
		jsonNote.z = 900
		jsonNote
	}

	def dispatch protected void toJSON(Region region, JsonCell parent, extension Context it) {
		val jsonRegion = new JsonRegion => [
			name = region.name
			id = region.ID
			parent = parent?.id ?: null
			z = if(parent !== null) parent.z + 1 else 1
		]
		jsonGraph.cells += jsonRegion
		if (parent instanceof JsonState)
			parent.embeds += jsonRegion.id

		val node = findNotationView(region, diagram, Region.simpleName)
		if (node !== null) {
			jsonRegion.setBounds(node, jsonGraph)
			node.persistedChildren.filter(Shape).head.persistedChildren.filter(Shape).filter[type == "Note"].forEach [ n |
				n.toJSONNote(jsonRegion, it)
			]
			// Shape Style
			val shapeStyle = node.styles.filter(ShapeStyle).head
			if (shapeStyle !== null) {
				// Defaults: fillColor="16448250" lineColor="12632256"
				val fillColor = shapeStyle.fillColor
				if (fillColor != 16448250)
					jsonRegion.fillColor = toHexColor(fillColor)
				val lineColor = shapeStyle.lineColor
				if (lineColor != 12632256)
					jsonRegion.strokeColor = toHexColor(lineColor)
			}
		}
		region.vertices.forEach[v|v.toJSON(jsonRegion, it)]
	}

	def dispatch protected void toJSON(State state, JsonCell parent, extension Context it) {
		val jsonState = new JsonState => [
			id = state.ID
			parent = parent?.id ?: null
			name = state.name
			specification = state.specification
			documentation = state.documentation
			z = parent.z + 1
		]
		jsonGraph.cells += jsonState
		if (parent instanceof JsonRegion)
			parent.embeds += jsonState.id
		// Region Alignment
		var node = findNotationView(state, diagram, State.simpleName)
		if (node !== null) {
			val layoutStyle = node.styles.filter(BooleanValueStyle).findFirst[it.name == "isHorizontal"]
			val isHorizontal = layoutStyle !== null ? layoutStyle.booleanValue : true
			jsonState.regionLayout = isHorizontal ? "horizontal" : "vertical"
			// Show Expression / Documentation
			val featureToShow = node.styles.filter(StringValueStyle).findFirst[it.name == "featureToShow"]
			val showSpecification = featureToShow !== null ? featureToShow.stringValue == "specification" : true
			jsonState.showSpecification = showSpecification
			jsonState.setBounds(node, jsonGraph)
			// Shape Style
			val shapeStyle = node.styles.filter(ShapeStyle).head
			if (shapeStyle !== null) {
				// Defaults: fillColor="15720400" lineColor="12632256"
				val fillColor = shapeStyle.fillColor
				if (fillColor != 15720400)
					jsonState.fillColor = toHexColor(fillColor)
				val lineColor = shapeStyle.lineColor
				if (lineColor != 12632256)
					jsonState.strokeColor = toHexColor(lineColor)
			}
		} // End Notation Model 
		state.regions.forEach[r|r.toJSON(jsonState, it)]
		state.outgoingTransitions.forEach[t|t.toJSON(jsonState, it)]

	}

	def dispatch protected void toJSON(Entry entry, JsonCell parent, extension Context it) {

		val nodeLabel = new JsonNodeLabel => [
			!entry.name.nullOrEmpty ? name = entry.name
			parent = entry.ID
			id = EcoreUtil.generateUUID()
			z = 1000
		]
		jsonGraph.cells += nodeLabel

		val jsonEntry = new JsonEntry => [
			id = entry.ID
			entryKind = entry.kind.getName.toFirstUpper
			parent = parent?.id ?: null
			z = parent.z + 1
			embeds += nodeLabel.id

		]
		jsonGraph.cells += jsonEntry

		if (parent instanceof JsonRegion)
			parent.embeds += jsonEntry.id

		val node = findNotationView(entry, diagram, entry.kind.entryNodeName)
		if (node !== null) {
			jsonEntry.setBounds(node, jsonGraph)
			nodeLabel.position = new JsonPosition => [
				x = jsonEntry.position.x
				y = jsonEntry.position.y + 22
			]

		}
		entry.outgoingTransitions.forEach[t|t.toJSON(jsonEntry, it)]
	}

	def dispatch protected void toJSON(Exit exit, JsonCell parent, extension Context it) {
		val nodeLabel = new JsonNodeLabel => [
			!exit.name.nullOrEmpty ? name = exit.name
			it.parent = exit.ID
			id = EcoreUtil.generateUUID()
			z = 1000
		]
		jsonGraph.cells += nodeLabel

		val jsonExit = new JsonExit => [
			id = exit.ID
			!exit.name.nullOrEmpty ? name = exit.name
			parent = parent?.id ?: null
			z = parent.z + 1
			embeds += nodeLabel.id
		]
		jsonGraph.cells += jsonExit
		if (parent instanceof JsonRegion)
			parent.embeds += jsonExit.id

		val node = findNotationView(exit, diagram, "Exit")
		if (node !== null) {
			jsonExit.setBounds(node, jsonGraph)
		}
		nodeLabel.position = new JsonPosition => [
			x = jsonExit.position.x
			y = jsonExit.position.y + 22
		]

		exit.outgoingTransitions.forEach[t|t.toJSON(jsonExit, it)]
	}

	def dispatch protected void toJSON(FinalState finalState, JsonCell parent, extension Context it) {
		val jsonFinal = new JsonFinal => [
			id = finalState.ID
			parent = parent?.id ?: null
			z = parent.z + 1
		]
		jsonGraph.cells += jsonFinal
		if (parent instanceof JsonRegion)
			parent.embeds += jsonFinal.id

		val node = findNotationView(finalState, diagram, FinalState.simpleName)
		if (node !== null) {
			jsonFinal.setBounds(node, jsonGraph)
		}
	}

	def dispatch protected void toJSON(Choice choice, JsonCell parent, extension Context it) {
		val jsonChoice = new JsonChoice => [
			id = choice.ID
			parent = parent?.id ?: null
			z = parent.z + 1
		]
		jsonGraph.cells += jsonChoice
		if (parent instanceof JsonRegion)
			parent.embeds += jsonChoice.id

		val node = findNotationView(choice, diagram, Choice.simpleName)
		if (node !== null) {
			jsonChoice.setBounds(node, jsonGraph)
		}
		choice.outgoingTransitions.forEach[t|t.toJSON(jsonChoice, it)]

	}

	def dispatch protected void toJSON(Synchronization sync, JsonCell parent, extension Context context) {
		val jsonSync = new JsonSynchronization => [
			id = sync.ID
			parent = parent?.id ?: null
			z = parent.z + 1
		]
		jsonGraph.cells += jsonSync
		if (parent instanceof JsonRegion)
			parent.embeds += jsonSync.id

		val node = findNotationView(sync, diagram, Synchronization.simpleName)
		if (node !== null) {
			jsonSync.setBounds(node, jsonGraph)
		}
		sync.outgoingTransitions.forEach[t|t.toJSON(jsonSync, context)]
	}

	def dispatch protected void toJSON(Transition transition, JsonCell parent, extension Context context) {
		val jsonTransition = new JsonTransition => [
			id = transition.ID
			z = 1000
			specification = transition.specification
			priority = "" + (transition.priority + 1)
			documentation = transition.documentation
			source = new JsonTransitionSource => [
				id = transition.source.ID
			]
			target = new JsonTransitionTarget => [
				id = transition.target.ID
			]
		]
		// https://gitlab.com/itemis/solutions/crt/create-rcp/-/issues/1806
		// Set the common anchestor of source and target as transitions parent.
		val commonAncestor = commonAncestor(transition.source, transition.target)
		if (commonAncestor !== null && !(commonAncestor instanceof Statechart)) {
			jsonTransition.parent = commonAncestor.ID;
			val parentContainer = context.jsonGraph.cells.findFirst[id == commonAncestor.ID]
			if (parentContainer instanceof JsonRegion)
				parentContainer.embeds += jsonTransition.id
			else if (parentContainer instanceof JsonState)
				parentContainer.embeds += jsonTransition.id

		}
		// Notation View Transformation
		val edge = findNotationEdge(transition, diagram, Transition.simpleName) as Edge
		if (edge !== null) {
			// Show Expression / Documentation
			val featureToShow = edge.styles.filter(StringValueStyle).findFirst[it.name == "featureToShow"]
			val showSpecification = featureToShow !== null ? featureToShow.stringValue == "specification" : true
			jsonTransition.showSpecification = showSpecification
			// Routing style
			val routingStyle = edge.styles.filter(ConnectorStyle).head
			jsonTransition.router = if ((routingStyle !== null &&
				routingStyle.routing == Routing.RECTILINEAR_LITERAL)) {
				new JsonTransitionRouter("orthogonal")
			} else {
				new JsonTransitionRouter("normal")
			}
			setLayoutFromAnnotation(jsonTransition, edge)
		}
		jsonGraph.cells += jsonTransition
	}

	def setLayoutFromAnnotation(JsonTransition transition, Edge edge) {
		// Bendpoints
		val vertices = edge.styles.filter(StringValueStyle).findFirst[name == "joint:vertices"]?.stringValue?.split(",")
		if (!vertices.nullOrEmpty) {
			transition.vertices = vertices.map[JsonPosition.fromString(it)]
		}
		// Source anchor
		val source = edge.styles.filter(StringValueStyle).findFirst[name == "joint:source:anchor"]?.stringValue?.
			split(";")
		if (!source.nullOrEmpty && source.length == 2) {
			transition.source.anchor = new JsonTransitionAnchor => [
				name = "topLeft"
				args.put("dx", source.get(0))
				args.put("dy", source.get(1))
				args.put("rotate", true)
			]
		}
		// Target anchor
		val target = edge.styles.filter(StringValueStyle).findFirst[name == "joint:target:anchor"]?.stringValue?.
			split(";")
		if (!target.nullOrEmpty && target.length == 2) {
			transition.target.anchor = new JsonTransitionAnchor => [
				name = "topLeft"
				args.put("dx", target.get(0))
				args.put("dy", target.get(1))
				args.put("rotate", true)
			]
		}
		// Label position
		val distance = edge.styles.filter(StringValueStyle).findFirst[name == "joint:label:distance"]?.stringValue
		val offset = edge.styles.filter(StringValueStyle).findFirst[name == "joint:label:offset"]?.stringValue
		if (!distance.isNullOrEmpty && !offset.isNullOrEmpty) {
			transition.labelDistance = Double.parseDouble(distance)
			transition.labelOffset = offset
		}

	}

	def protected priority(Transition it) {
		(eContainer as Vertex).outgoingTransitions.indexOf(it)
	}

	def static protected EObject commonAncestor(EObject a, EObject b) {
		for (var x = a?.eContainer; x !== null; x = x.eContainer) {
			for (var y = b?.eContainer; y !== null; y = y.eContainer) {
				if (x === y)
					return x
			}
		}
		return null
	}

}
