/**
 * Copyright (c) 2026 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.itemis.create.base.model.core

import com.yakindu.base.types.Declaration
import com.yakindu.base.types.AnnotatableElement
import com.yakindu.base.types.AnnotationType
import com.yakindu.base.types.TypesFactory
import com.yakindu.base.types.ComplexType
import java.util.List

/**
 * This class defines the core mechanisms to access and provide concepts. 
 * It does not define concrete concepts itself.
 * 
 * This implementation should use no injection.
 * 
 * @author axel terfloth - initial contribution
 */
class ConceptModel {
	
	def <R extends Declaration> R getConcept(ComplexType it, Class<R> clazz, AnnotationType at) {
		features.filter(clazz).findFirst[ 
			if (it instanceof AnnotatableElement)
				it.isAnnotatedWith(at)
			else
				false
		]
	}
	
	def <R extends Declaration> R getMetaConcept(Declaration it, Class<R> clazz, AnnotationType at) {
		metaFeatures.filter(clazz).findFirst[ 
			if (it instanceof AnnotatableElement)
				it.isAnnotatedWith(at)
			else
				false
		]
	}
	
	/**
	 * gets or creates a declaration
	 */
	def <S extends Declaration, R extends Declaration> R provideConcept(S s, (S)=>R query, (S)=>R factory) {
		if (s !== null) {
			val defined = query.apply(s)
			if (defined !== null) return defined
		}
		val created = factory.apply(s)
		return created
	}
	
	def void annotateWith(AnnotatableElement it, AnnotationType type) {
		if (it.getAnnotationOfType(type.name) === null) {
			val annotation = TypesFactory.eINSTANCE.createAnnotation
			annotation.type = type			
			annotations += annotation
		}
	}
	
	def boolean isAnnotatedWith(AnnotatableElement it, AnnotationType type) {
		return (it.getAnnotationOfType(type.name) !== null)
	}
	
	static class ConceptDefinition<S extends Declaration, C extends Declaration> {
		
		protected Concept<S, C> concept
		protected S subject
		
		new(Concept<S, C> concept, S subject) {
			this.concept = concept
			this.subject = subject
		}
		
		def C definition() {
			concept?.definition(subject)
		}
		
		def C define() {
			concept?.define(subject)
		}
		
		def boolean isDefined() {
			concept?.definition(subject) !== null
		}
		
	}
	
	static def <S extends Declaration, C extends Declaration> Concept<S,C> metaConcept(Class<S> sc, Class<C> cc, String i, (S)=>C p) {
		new Concept(sc, cc, i, p)
	}
	
	static def <S extends Declaration, C extends Declaration> Concept<S,C> featureConcept(Class<S> sc, Class<C> cc, String i, (S)=>C p) {
		val concept = new Concept(sc, cc, i, p)
		concept.anchor = [ if (it instanceof ComplexType) features else metaFeatures ]
		return concept
	}
	
	static def <S extends Declaration, C extends Declaration> Concept<S,C> featureConcept(
		Class<S> sc, 
		Class<C> cc, 
		String i, 
		(S)=>C p, 
		(S)=>List<Declaration> a, 
		(S,C)=>boolean f
	) {
		val concept = new Concept(sc, cc, i, p)
		concept.anchor = a
		concept.filter = f
		return concept
	}
	
	static  class Concept<S extends Declaration, C extends Declaration> extends ConceptModel {
		protected Class<S> subjectType
		protected Class<C> conceptType
		protected String identifier
		protected AnnotationType conceptAnnotationType
		protected (S,C)=>boolean filter
		protected (S)=>C provider
		protected (S)=>List<Declaration> anchor
		
		new(Class<S> sc, Class<C> cc, String i, (S)=>C p, (S)=>List<Declaration> a) {
			subjectType = sc
			conceptType = cc
			identifier = i
			filter = null
			provider = p
			anchor = a
			
			conceptAnnotationType = TypesFactory.eINSTANCE.createAnnotationType	=> [
				name = '''_«sc.class.simpleName»_«identifier»_concept_'''
			]
			
			if (anchor === null) {
				anchor = [metaFeatures]
			}
			
			if (filter === null) {
				filter = [ true ]
			}
		}
		
		new(Class<S> sc, Class<C> cc, String i, (S)=>C p) {
			this(sc, cc, i, p, null)
		}
		
		
		def C definition(S s) {
			anchor.apply(s).filter(conceptType).filter[ filter.apply(s,it) ].findFirst[isAnnotatedWith(conceptAnnotationType)]
		}
		
		def boolean appliesTo(C it) {
			it.isAnnotatedWith(conceptAnnotationType)
		}
		
		def synchronized C define(S s) {
			if (s !== null) {
				val defined = s.definition
				if (defined !== null) return defined
			}
			val created = provider.apply(s)
			created.annotateWith(conceptAnnotationType)
			connect(s, created)
			return created
		}
		
		def protected void connect(S s, C c) {
			anchor.apply(s).add(c)
		}
		
	}
	
}