/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Mülder - itemis AG
 * 
 */
package com.yakindu.sct.domain.scxml.simulation

import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.interpreter.IExpressionInterpreter
import com.yakindu.base.expressions.interpreter.context.IExecutionSlotResolver
import com.yakindu.base.types.Expression
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.impl.ExecutionVariableImpl
import java.util.Optional
import org.eclipse.xtext.naming.IQualifiedNameProvider
import org.eclipse.xtext.nodemodel.util.NodeModelUtils

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
class SCXMLExecutionSlotResolver implements IExecutionSlotResolver {

	@Inject
	extension IQualifiedNameProvider nameProvider
	@Inject
	extension IExpressionInterpreter interpreter;

	override resolve(ExecutionContext context, Expression expression) {
		try {
			val resolvedObject = expression.resolve
			if (resolvedObject !== null) {
				val slot = context.getSlot(resolvedObject)
				if (slot !== null)
					return Optional.of(slot)
			}
		} catch (IllegalArgumentException ex) {
			// Not found, simply ignore and try with text
		}
		val node = NodeModelUtils.getNode(expression)
		val text = if(node !== null) node.text.trim else expression.resolve

		if (text.contains("Math.floor")) {
			return Optional.of(mathFloor(expression, context))
		}
		var result = context.getSlot(text)
		if (result === null) {
			// TODO: How can we handle this?
			result = context.getSlot(text.replaceAll("_", "."))
		}
		if (result === null)
			throw new IllegalStateException("No slot found for '" + text + "'")

		Optional.of(result)
	}

	def mathFloor(Expression expression, ExecutionContext context) {
		val result = (expression as ArgumentExpression).arguments.head.value.evaluate(context) as Double
		return new ExecutionVariableImpl() {
			override getValue() {
				return Math.floor(result)
			}

		}
	}

	def dispatch String resolve(ElementReferenceExpression it) {
		reference.resolve
	}

	def dispatch String resolve(FeatureCall it) {
		feature.resolve
	}

	def dispatch String resolve(NamedElement it) {
		getFullyQualifiedName().toString
	}
}
