/**
* Copyright (c) 2025 itemis AG - All rights Reserved
* Unauthorized copying of this file, via any medium is strictly prohibited
* 
* Contributors:
* 	axel terfloth - itemis AG
*
*/
package com.yakindu.sct.types.concepts

import com.google.inject.Inject
import com.yakindu.base.types.ComplexType
import com.yakindu.sct.model.sgraph.Statechart
import com.google.inject.Provider
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.sct.model.stext.stext.VariableDefinition
import com.yakindu.sct.types.resource.Statechart2TypeTransformation
import com.yakindu.base.types.Part
import com.yakindu.sct.model.stext.concepts.InstancePath
import java.util.List
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Property
import com.yakindu.base.types.Declaration
import org.eclipse.emf.ecore.EObject
import com.yakindu.base.types.TypesUtil

/**
 * This class defines the concept of a statechart group. 
 * A statechart group is a type which contains a set of statecharts and is responsible
 * for orchestrating those. Thsi includes instantiation and connecting statechart references.
 * For this purpose a setup method is defined by the group.
 * 
 * This implementation takes a root statechart and determines the instances and their connections 
 * in the same way as multi state machine handling is implemented by SCTUnit and the interpreter. 
 * 
 * @author axel terfloth - Initial contribution and API
 */
class StatechartGroup {
	
	public static final String SETUP_METHOD_NAME = "setUp";
	public static final String ROOT_NAME = "root";
	
	@Inject protected Provider<com.yakindu.sct.model.stext.concepts.SubchartTraversal> subchartsProvider
	@Inject protected extension TypesUtil
	@Inject protected extension TypeBuilder
	@Inject protected extension ExpressionBuilder
	@Inject protected extension Statechart2TypeTransformation
	
	def ComplexType create group : createSubchartGroup(sc) subchartGroup(Statechart sc) {
	} 
	
	def protected createSubchartGroup(Statechart sc) {
		val group = _complexType
		val extension subcharts = subchartsProvider.get
		sc.createTypeDescription
		
		group.name = sc.name + "Group"
		
		val root = _part(ROOT_NAME, sc.statechartType)
		group.features += root
		
		sc.subchartPaths.forEach[ path |
			path.type.createTypeDescription
			_part(path.toName, path.type.statechartType) => [
				group.features += it
			]
		]		
		val groupSCs = group.groupStatecharts
		
		_op(SETUP_METHOD_NAME, _void) => [
			_public
			_body(_block(
				sc.subchartPaths.map[ path |
					val child = groupSCs.findFirst[name == path.toName]
					val parent = groupSCs.findFirst[name == path.path.parent.toName] ?: root
					if (child !== null && parent !== null) {
						val parentChildRef = parent.memberWithName(path.path.lastOrNull.name)
						parent._ref._dot(parentChildRef)._assign(child._ref())
					} else {
						_block
					}
				].toList
			))
			group.features += it
		]
		
		return group
	}
	
	def protected toName(InstancePath<VariableDefinition, Statechart> instance) {
		instance.path.toName
	}
	
	def protected toName(List<VariableDefinition> path) {
		path.map[name].reduce[n1, n2| n1+"_"+n2]
	}
	
	def protected parent(List<VariableDefinition> path) {
		val count = (path.size > 0) ? path.size-1 : 0
		path.take(count).toList
	}
	
	def protected dispatch Declaration memberWithName(Property it, String name) {
		type.memberWithName(name)
	}
	
	def protected dispatch Declaration memberWithName(ComplexType it, String name) {
		features.findFirst[ f | f.name == name]
	}
	
	def protected dispatch Declaration memberWithName(EObject it, String name) {
		null
	}
	
	def groupStatechartTypes(ComplexType it) {
		it.features
			.filter(Part)
			.map[type]
			.filter(ComplexType)
			.filter[isStatechartType]
			.toList
			.stream.distinct.toList
	}
		
	def groupStatecharts(ComplexType it) {
		it.features
			.filter(Part)
			.filter[type.isStatechartType]
			.toList
	}

	def groupRoot(ComplexType it) {
		it.features
			.filter(Part)
			.filter[type.isStatechartType]
			.findFirst[name == ROOT_NAME]
	}

	def groupSetUpMethod(ComplexType it) {
		it.features
			.filter(Operation)
			.findFirst[name == SETUP_METHOD_NAME]
	}
	
}
