/**
 * Copyright (c) 2026 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.generator.c.codemodel

import com.google.inject.Inject
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.BlockExpression
import com.yakindu.base.expressions.expressions.LambdaExpression
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Parameter
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Expression
import java.util.List
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.sct.generator.c.extensions.Naming

/**
 * Transformation which handles unused parameters in functions.  
 * For each unused parameter a 'SC_UNUSED' directive is added to the top of the function body.
 * Currently only lambda functions are covered.
 * 
 * @author Axel Terfloth - initial contribution
 */
class Unused {
	
	@Inject protected extension Naming
	@Inject protected extension ExpressionExtensions
	@Inject protected extension ExpressionBuilder
	 
	def handleUnusedParameters(ExecutionFlow flow) {
		flow.eAllContents.filter(LambdaExpression).forEach[handle]
		flow.eAllContents.filter(Operation).forEach[handle]
	}
	
	def handleUnusedHandle(ExecutionFlow flow) {
		flow.eAllContents.filter(Operation).forEach[handleUnusedHandle(flow)]
	}
	
	def protected void handle(LambdaExpression le) {
		le.expression = le.expression.handleUnusedParameters(le.parameters)
	}
	
	def protected handle(Operation op) {
		if (op.implementation !== null) {
			op.implementation = op.implementation.handleUnusedParameters(op.parameters)
		}
	}
	
	def protected handleUnusedParameters(Expression body, List<Parameter> parameters) {
		val allUsedParams = body
								.eAllContents
								.filter(ArgumentExpression)
								.map[featureOrReference]
								.filter(Parameter)
								.toSet
		val unusedParams = parameters
			.filter[ p | ! allUsedParams.contains(p)]
			.toList
		val unused = unusedParams.map[ _verbatim('''SC_UNUSED(«it.name»)''') as Expression ].toList
		
		return body.addToFront(unused)
	}
	
	def protected addToFront(Expression body, List<Expression> expressions) {
		if (expressions.size > 0) {
			val unusedHandlerBlock = _block
			expressions.forEach[
				unusedHandlerBlock.expressions += it
			]
			if (body instanceof BlockExpression) {
				unusedHandlerBlock.expressions += (body as BlockExpression).expressions
			} else {
				unusedHandlerBlock.expressions += body
			}
			return unusedHandlerBlock
		}
		return body
	}
	
	def protected handleUnusedHandle(Operation op, ExecutionFlow flow) {
		if (op.implementation !== null) {
			op.implementation = op.implementation.handleUnusedHandle(flow)
		}
	}
	
	def protected handleUnusedHandle(Expression body, ComplexType instanceType) {
		val allDeclarations = instanceType.eAllContents.filter(Declaration).reject[it instanceof Parameter].toSet
		val usedDeclarations = body
								.eAllContents
								.filter(ArgumentExpression)
								.map[featureOrReference]
								.filter(Declaration)
								.reject[ it instanceof Parameter ]
								.filter[ allDeclarations.contains(it) ]
								.toSet
		
		if (usedDeclarations.size == 0) {
			return body.addToFront(#[ _verbatim('''SC_UNUSED(«scHandle»)''' )])
		}
		return body
	}
	
}