/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.itemis.create.base.generator.core

import org.eclipse.emf.common.notify.impl.AdapterImpl
import org.eclipse.xtext.xbase.lib.Functions.Function0
import org.eclipse.emf.ecore.EObject

/**
 * Used to assign a generator to an element of the formal (type based) class declarations and definitions.
 *  
 * @author axel terfloth
 */

class GeneratorAssignment {
	
	
	def <T extends EObject> T generateDeclarationWith(T it, Function0<CharSequence> g) {
		return assignGenerator(new DeclarationAdapter(g))
	}	
	
	def <T extends EObject> T generateDefinitionWith(T it, Function0<CharSequence> g) {
		return assignGenerator(new DefinitionAdapter(g))
	}	
	
	def <S extends Adapter, T extends EObject> T assignGenerator(T it, S adapter) {
		val existingAdapter = it?.eAdapters.filter(adapter.class).head
		
		if (existingAdapter !== null) {
			it.eAdapters.remove(existingAdapter);
		}
		
		it.eAdapters.add(adapter)	

		return it
	}	
	
	def Function0<CharSequence> assignedGenerator(EObject it, Class<? extends Adapter> clazz) {
		it?.eAdapters.filter(clazz).head?.generator
	}	
	
	def Function0<CharSequence> getDeclarationGenerator(EObject it) {
		assignedGenerator(DeclarationAdapter)
	}	

	def Function0<CharSequence> getDefinitionGenerator(EObject it) {
		assignedGenerator(DefinitionAdapter)
	}	

	/** 
	 * Returns the generated declaration code for the model element.
	 */
	def CharSequence declarationCode(EObject it) {
		declarationGenerator?.apply
	}
	
	/** 
	 * Returns the generated declaration code for the model element and applies the default generator if no generator is assigned.
	 */
	def CharSequence declarationCode(EObject it, Function0<CharSequence> g) {
		(declarationGenerator?:g).apply
	}
	
	/** 
	 * Returns the generated declaration code for the model element.
	 */
	def CharSequence definitionCode(EObject it) {
		definitionGenerator?.apply
	}
	
	/** 
	 * Returns the generated declaration code for the model element and applies the default generator if no generator is assigned.
	 */
	def CharSequence definitionCode(EObject it, Function0<CharSequence> g) {
		(definitionGenerator?:g).apply
	}
	
	static class Adapter extends AdapterImpl {
		protected Function0<CharSequence> generator
		
		new(Function0<CharSequence> g) {
			generator = g
		}
		
		def Function0<CharSequence> getGenerator() {
			return generator
		}
	}
	
	static class DeclarationAdapter extends Adapter {
	
		new(Function0<CharSequence> g) {
			super(g)
		}
			
	}
	
	static class DefinitionAdapter extends Adapter {
	
		new(Function0<CharSequence> g) {
			super(g)
		}
			
	}
	
	
}