/**
 * Copyright (c) 2018 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Robert Rudi - itemis AG
 * 
 */
package com.yakindu.sct.domain.c.runtime.generator

import com.google.inject.Inject
import com.yakindu.base.types.scoping.IPackageImport2URIMapper
import com.yakindu.base.types.scoping.IPackageImport2URIMapper.PackageImport
import com.yakindu.sct.commons.EMFHelper
import com.yakindu.sct.commons.PathHelper
import com.yakindu.sct.domain.c.runtime.resource.IncludePathImport
import com.yakindu.sct.domain.common.ExternalIncludes
import com.yakindu.sct.generator.c.IncludeProvider
import com.yakindu.sct.generator.c.extensions.FileNaming
import com.yakindu.sct.generator.c.extensions.GenmodelEntries
import com.yakindu.sct.generator.core.artifacts.IGenArtifactConfigurations
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sgen.GeneratorEntry
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.stext.stext.ImportScope
import java.nio.file.LinkOption
import java.util.Optional
import java.util.Set
import org.eclipse.core.runtime.Path

class CDomainIncludeProvider implements IncludeProvider {

	@Inject extension FileNaming
	@Inject extension ExternalIncludes
	@Inject extension PathHelper
	@Inject extension GenmodelEntries
	@Inject extension GeneratorEntry entry

	@Inject
	IPackageImport2URIMapper includeMapper;

	override getIncludes(ExecutionFlow flow, IGenArtifactConfigurations artifactConfigs) {
		val includes = newArrayList
		val references = flow.includeFiles
		
		for(PackageImport p : flow.packageImports) {
			val path = usedPackageImportPath(p, references)
			if(path.isPresent) {
				includes += createImport(p, path.get.toFile.toPathString, artifactConfigs, flow)
			}
		}

		includes
	}
	
	/**
	 * For a given PackageImport, try to check if it is used by searching the given references for two versions
	 * of the PackageImport's path - the original one (which may be a symlink) and a resolved one.
	 * 
	 * Returns the version it found or an empty optional if this import was not used at all.
	 */
	protected def Optional<java.nio.file.Path> usedPackageImportPath(PackageImport pkgImport, Set<String> references) {
		val symLinkPath = pkgImport.toPathString.toPath(LinkOption.NOFOLLOW_LINKS)
		val resolvedPath = pkgImport.toPathString.toPath()
		if(entry.generateAllIncludes){
			return Optional.of(symLinkPath)
		}		
		if(references.exists[it.asKey == symLinkPath.toFile.toPathString.asKey]) return Optional.of(symLinkPath)
		if(references.exists[it.asKey == resolvedPath.toFile.toPathString.asKey]) return Optional.of(resolvedPath)
		else return Optional.empty		
	}

	protected def getPackageImports(ExecutionFlow flow) {
		val statechart = flow.sourceElement as Statechart
		val statechartImports = statechart.scopes.filter(ImportScope).map[imports].flatten
		return statechartImports.map[
			includeMapper.findPackageImport(statechart.eResource, it)
		].filter[isPresent].map[get]
	}

	protected def dispatch createImport(PackageImport include, String headerFile,
		extension IGenArtifactConfigurations artifactConfigs, ExecutionFlow flow) {
		var path = new Path(headerFile.relativeTo(flow.module.h))
		'''
			#include "«path.toString»"
		'''
	}

	protected def dispatch createImport(IncludePathImport include, String headerFile,
		extension IGenArtifactConfigurations artifactConfigs, ExecutionFlow flow) {
		'''
			#include <«include.name»>
		'''
	}

	protected def toPathString(PackageImport value) {
		var absPath = "NO_KEY";
		val uri = value.fileURI
		if (uri.isPlatform) {
			val iFile = EMFHelper.getIFileFromEMFUri(uri)
			absPath = iFile.getLocation().toString()
		} else if (uri.isFile) {
			absPath = uri.toFileString()
		} else {
			throw new IllegalArgumentException("Expecting a file OR platform URI but was : " + value.uri.toString);
		}
		return absPath
	}

	/**
	 * remove file separators because during conversion between EMF, EFS & java.io the string representation changes...
	 */
	protected def asKey(String value) {
		return value.replaceAll("\\\\", "").replaceAll("\\/", "")
	}

}
