/** 
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Contributors:
 * Andreas Muelder - itemis AG
 */
package com.yakindu.sct.domain.c.runtime.validator

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.InitializationExpression
import com.yakindu.base.expressions.expressions.IntLiteral
import com.yakindu.base.expressions.expressions.PrimitiveValueExpression
import com.yakindu.base.expressions.expressions.StringLiteral
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Parameter
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeAlias
import com.yakindu.base.types.TypedDeclaration
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.generator.c.types.CTypeAnnotations
import com.yakindu.sct.generator.c.typesystem.CTypeSystem
import com.yakindu.sct.model.sgraph.util.StatechartUtil
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.OperationDefinition
import com.yakindu.sct.model.stext.stext.VariableDefinition
import com.yakindu.sct.model.stext.validation.STextValidator
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.ComposedChecks

/** 
 * Invokes checks to assure that type inferrer marks usages of unsupported
 * types.
 * @author thomas kutz - Initial contribution and API
 */
@ComposedChecks(validators=#[CBuiltInTypesValidator,
	ShortIdentifiersValidator]) class CSTextJavaValidator extends STextValidator implements CSTextValidationMessages {
	@Inject extension ITypeSystemInferrer
	@Inject ITypeSystem typeSystem
	@Inject extension ExpressionExtensions
	@Inject extension StatechartUtil
	@Inject protected extension CTypeAnnotations

	@Check
	def checkExpression_OperationDefinition(OperationDefinition it) {
		if(type === null || type.eIsProxy()) return;
		infer(it, this)
	}

	@Check
	def checkExpression_EventDefinition(EventDefinition it) {
		if(type === null || type.eIsProxy()) return;
		infer(it, this)
	}

	@Check
	def checkExpression_Parameter(Parameter it) {
		if(type === null || type.eIsProxy()) return;
		infer(it, this)
	}

	@Check
	def checkExpression_TypeAlias(TypeAlias it) {
		if(type === null || type.eIsProxy()) return;
		infer(it, this)
	}

	@Check
	def checkVariableOfTypeArray_VariableDefinition(VariableDefinition it) {
		if(type === null || type.eIsProxy()) return;
		if (typeSystem.isSame(type.getOriginType(), typeSystem.getType(ITypeSystem.ARRAY))) {
			error(String.format(ARRAY_TYPE_SUPPORTED, name), it, null)
		}
	}

	@Check
	def void checkUniquePointerVariable(VariableDefinition it) {
		if (typeSpecifier.type.name.equals(CTypeSystem.UNIQUE_POINTER)) warning(String.format(UNIQUE_PTR_VARIABLE_MSG), it, null)
	}

	@Check
	def checkPointerAssignmentToLocalVariable(AssignmentExpression it) {
		if (varRef instanceof FeatureCall && expression instanceof FeatureCall) {
			val varRefFeature = (varRef as FeatureCall).feature
			if(!varRefFeature.eContainer.multiSM) return;
			if(!(expression as FeatureCall).owner.featureOrReference?.eContainer.multiSM) return;
			
			val feature = (expression as FeatureCall).feature
			if (feature instanceof TypedDeclaration) {
				if (CTypeSystem.POINTER_TYPES.exists[t | typeSystem.isSame(feature.type.originType, typeSystem.getType(t))] ) {
					error(String.format(POINTER_NOT_ALLOWED, ((varRef as FeatureCall).feature as Declaration).name), it, null)
				}
			}
		}
	}

	@Check
	def checkPropertyAssignmentInComplexTypeToLocalVariable(AssignmentExpression it) {
		val varRef = it.varRef
		if (varRef instanceof FeatureCall) {
			// Check if the access is to ct.property
			if(!(varRef.feature instanceof Property)) return;
			if(!(varRef.feature.eContainer instanceof ComplexType)) return;
			if(varRef.feature.eContainer.multiSM) return;

			// get first part of the callStack -> submachine.ct.ct.ct....ct.property
			val typedDeclaration = varRef.toCallStack.map[featureOrReference].filter(TypedDeclaration).head
			if (typedDeclaration.type.isMultiSM) {
				error(String.format(ASSIGN_TO_COMPLEX_TYPE_NOT_ALLOWED, (varRef.feature as Declaration).name, typedDeclaration.name), it,
					null)
			}
		}
	}
	
	@Check
	def checkArrayAssignment(AssignmentExpression it) {
		val varRef = varRef
		val ref = varRef.featureOrReference
		if (ref instanceof Property) {
			if ((typeSystem.isSame(ref.type.getOriginType(), typeSystem.getType(CTypeSystem.ARRAY)))) {
				if (varRef instanceof ElementReferenceExpression) {
					if(varRef.arrayAccess) return;
				}
				if (varRef instanceof FeatureCall) {
					if(varRef.arrayAccess) return;
				}
				error(String.format(ASSIGN_TO_ARRAY_NOT_ALLOWED, ref.name, ref.name), it, null)
			}			
			if(ref.hasCharArrayAnnotation){
				if(expression instanceof PrimitiveValueExpression){
					val expressionValue = (expression as PrimitiveValueExpression).value
					if(expressionValue instanceof StringLiteral){
						val stringLength = expressionValue.value.length
						val charArraySize = ((ref.getCharArrayAnnotation.arguments.head.value as PrimitiveValueExpression).value as IntLiteral).value
						if(charArraySize !== 0 && stringLength >= charArraySize){
							error(String.format(ASSIGN_TO_CHAR_ARRAY_OUT_OF_BOUND, ref.name, ref.name), it, null)
						}	
					}
				}
			}
		}
	}
}
