/**
 * Copyright (c) 2018 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 
 */
package com.yakindu.sct.generator.scxml.modifications

import com.yakindu.base.expressions.expressions.AdditiveOperator
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.AssignmentOperator
import com.yakindu.base.expressions.expressions.ExpressionsFactory
import com.yakindu.base.expressions.expressions.MultiplicativeOperator
import com.yakindu.base.expressions.expressions.ShiftOperator
import com.yakindu.sct.model.sgraph.Statechart

import static extension org.eclipse.emf.ecore.util.EcoreUtil.*

/**
 * @author andreas muelder - Initial contribution and API
 * @author robert rudi	- Added several replacement implementations for operator assignments (i.e. *= , /= , etc.)
 */
class AssignmentModification implements IModification {

	protected static extension ExpressionsFactory = ExpressionsFactory.eINSTANCE

	override modify(Statechart statechart) {
		statechart.eAllContents.filter(AssignmentExpression).forEach [ assignment |
			switch (assignment.operator) {
				case ADD_ASSIGN,
				case SUB_ASSIGN:
					assignment.replaceAddSubstractAssigment(assignment.operator)
				case DIV_ASSIGN,
				case MULT_ASSIGN,
				case MOD_ASSIGN:
					assignment.replaceMultiplicativeAssignment(assignment.operator)
				case LEFT_SHIFT_ASSIGN,
				case RIGHT_SHIFT_ASSIGN:
					assignment.replaceShiftAssignment(assignment.operator)
				case AND_ASSIGN,
				case XOR_ASSIGN,
				case OR_ASSIGN:
					assignment.replaceBitwiseAssignment(assignment.operator)
				default: {
				}
			}
			assignment.operator = AssignmentOperator.ASSIGN
		]
	}

	def void replaceBitwiseAssignment(AssignmentExpression assignment, AssignmentOperator op) {
		switch (op) {
			case AND_ASSIGN: {
				assignment.expression = createBitwiseAndExpression => [
					leftOperand = assignment.varRef.copy
					rightOperand = assignment.expression.copy
				]
			}
			case OR_ASSIGN: {
				assignment.expression = createBitwiseOrExpression => [
					leftOperand = assignment.varRef.copy
					rightOperand = assignment.expression.copy
				]
			}
			case XOR_ASSIGN: {
				assignment.expression = createBitwiseXorExpression => [
					leftOperand = assignment.varRef.copy
					rightOperand = assignment.expression.copy
				]
			}
		}
	}

	def void replaceShiftAssignment(AssignmentExpression assignment, AssignmentOperator op) {
		var shiftExpression = createShiftExpression
		switch (op) {
			case LEFT_SHIFT_ASSIGN:
				shiftExpression.operator = ShiftOperator.LEFT
			case RIGHT_SHIFT_ASSIGN:
				shiftExpression.operator = ShiftOperator.RIGHT
		}
		shiftExpression.leftOperand = assignment.varRef.copy
		shiftExpression.rightOperand = assignment.expression.copy
		assignment.expression = shiftExpression
	}

	def void replaceAddSubstractAssigment(AssignmentExpression assignment, AssignmentOperator op) {
		val addSubstractExpression = createNumericalAddSubtractExpression
		switch (op) {
			case ADD_ASSIGN:
				addSubstractExpression.operator = AdditiveOperator.PLUS
			case SUB_ASSIGN:
				addSubstractExpression.operator = AdditiveOperator.MINUS
		}
		addSubstractExpression.leftOperand = assignment.varRef.copy
		addSubstractExpression.rightOperand = assignment.expression.copy
		assignment.expression = addSubstractExpression
	}

	def void replaceMultiplicativeAssignment(AssignmentExpression assignment, AssignmentOperator op) {
		val multiplicativeExpression = createNumericalMultiplyDivideExpression
		switch (op) {
			case DIV_ASSIGN:
				multiplicativeExpression.operator = MultiplicativeOperator.DIV
			case MULT_ASSIGN:
				multiplicativeExpression.operator = MultiplicativeOperator.MUL
			case MOD_ASSIGN:
				multiplicativeExpression.operator = MultiplicativeOperator.MOD
		}
		multiplicativeExpression.leftOperand = assignment.varRef.copy
		multiplicativeExpression.rightOperand = assignment.expression.copy
		assignment.expression = multiplicativeExpression
	}
}
