/**
 * Copyright (c) 2023-2026 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.itemis.create.statechart.generator.csharp.transformation

import com.google.inject.Inject
import com.itemis.create.base.generator.core.Transformation
import com.itemis.create.base.generator.core.concepts.TimerService
import com.itemis.create.base.generator.csharp.codemodel.CsharpClass
import com.itemis.create.base.generator.csharp.codemodel.CsharpNaming
import com.itemis.create.base.generator.csharp.codemodel.CsharpStatemachineLibrary
import com.itemis.create.base.generator.csharp.codemodel.CsharpTypeBuilder
import com.itemis.create.base.generator.csharp.transformation.ClearOutEventsTransformation
import com.itemis.create.base.generator.csharp.transformation.EventBufferTransformation
import com.itemis.create.base.generator.csharp.transformation.RaiseEventModification
import com.itemis.create.statechart.generator.csharp.codemodel.CsharpNamedInterfaceClasses
import com.itemis.create.statechart.generator.csharp.codemodel.CsharpStatemachineEvents
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventValueReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.PrimitiveValueExpression
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Argument
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Part
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.TypedDeclaration
import com.yakindu.base.types.TypesFactory
import com.yakindu.sct.generator.core.codemodel.StatemachineClass
import com.yakindu.sct.generator.core.codemodel.StatemachineEvents
import com.yakindu.sct.generator.core.concepts.EventMembers
import com.yakindu.sct.generator.core.extensions.EventQueueExtension
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.TimeEvent
import com.yakindu.sct.model.sexec.concepts.EventProcessing
import com.yakindu.sct.model.sexec.concepts.PropertyBinder
import com.yakindu.sct.model.sexec.concepts.StateMachineBehaviorConcept
import com.yakindu.sct.model.sexec.concepts.SubMachine
import com.yakindu.sct.model.sexec.concepts.SubMachine.SubmachineTypeLibrary
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sexec.extensions.SexecBuilder
import com.yakindu.sct.types.resource.Statechart2TypeTransformation
import java.util.HashMap
import java.util.Iterator
import java.util.Map
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.EcoreUtil2

/**
 * As we still depend on the exec flow (mostly due to the flow code) we require to transform it.
 * This class is responsible to update the references to refer the new constructed types throughout expressions, operations, methods...
 * 
 * @author laszlo kovacs
 * 
 */
class ExecFlow2stmClassTransformation extends Transformation<ExecutionFlow, ExecutionFlow> {

	@Inject protected extension EventBufferTransformation
	@Inject protected extension StatemachineClass
	@Inject protected extension StatemachineEvents
	@Inject protected extension RaiseEventModification
	@Inject protected extension EventMembers
	@Inject protected extension SExecExtensions
	@Inject protected extension ExpressionExtensions
	@Inject protected extension ExpressionBuilder
	@Inject protected extension CsharpNaming
	@Inject protected extension CsharpTypeBuilder
	@Inject protected extension CsharpClass
	@Inject protected extension Statechart2TypeTransformation
	@Inject protected extension CsharpNamedInterfaceClasses
	@Inject protected extension StateMachineBehaviorConcept
	@Inject protected extension EventQueueExtension
	@Inject protected extension SexecBuilder
	@Inject protected extension ClearOutEventsTransformation
	@Inject protected extension CsharpStatemachineEvents
	@Inject protected extension SubMachine
	@Inject protected extension SubmachineTypeLibrary
	@Inject protected extension CsharpStatemachineLibrary
	@Inject protected extension PropertyBinder
	@Inject protected extension TimerService
	
	
	protected extension TypesFactory tFactory = TypesFactory.eINSTANCE
	

	override protected toTarget(ExecutionFlow source) {
		
		source.substituteEventValueReferenceExpressionsByEventValue
		source.submachineContextMember?.type?.applyCsharpNamingConventions
		
		source.stateMachineClass.substitueMethodMember(csharpSubStatemachineInterface)
		source.stateMachineClass.substitueMethodMember(csharpSubmachineContextInterface)
		
		source.substitueMethodMember(source.stateMachineClass)
		source.substitueStateMachineConcepts
		
		source.substituteEventRaisingExpressionsByEventRaisers
		source.stateMachineClass.substituteEventRaisingExpressionsByEventRaisers
		
		source.substitueEventRaisedExpression
		source.transformTimeEventReferences
		
		source.setOperationCall(source.stateMachineClass)
		
		source.transformSubmachineTypes
		source.stateMachineClass.transformBinderTypes
		source.stateMachineClass.transformSubmachineTypes
		
		source.transformClearOutEventOperation
		source.stateMachineClass.transformedRootFrom(source)
		source.stateMachineClass.applyCsharpNamingConventions
		
		source
	}

	override protected modifyTarget(ExecutionFlow target) {
		target.substitueOperationCalls
	}
	
	def protected transformTimeEventReferences(ExecutionFlow flow){
		val containers = #[flow, flow.stateMachineClass]
		containers.forEach[
			eAllContents
				.filter(ElementReferenceExpression)
				.filter[reference instanceof TimeEvent]
				.forEach[
					arraySelector += flow.getTimeEvents.indexOf(reference)._integer
					arrayAccess = true
					reference = flow.timeEventsVariable(flow.timeEvents.size)
				]
		]
	}
	
	def protected transformBinderTypes(ComplexType stmClass) {
		val csharpBinderType = stmClass.features.filter(ComplexType).filter[origin === binderType].head
		stmClass.eAllContents.filter(TypeSpecifier).filter[type === binderType].forEach[
			val container = eContainer
			if(container instanceof TypedDeclaration)
				container.typeSpecifier = createTypeSpecifier => [
					type = csharpBinderType
				]
			//Means it's a superType
			else if(container instanceof ComplexType)
				container.superTypes.set(container.superTypes.indexOf(it), createTypeSpecifier => [
					type = csharpBinderType
				]) 
		]
	}
	
	def protected transformSubmachineTypes(ComplexType flow) {
		flow.eAllContents.filter(TypeSpecifier).filter[type === getSubmachineContextInterface].forEach[
			val container = eContainer
			if(container instanceof TypedDeclaration)
				container.typeSpecifier = createTypeSpecifier => [
					type = getSubmachineContextInterface.asCsharpDeclaration
				]
			//Means it's a superType
			else if(container instanceof ComplexType)
				container.superTypes.set(container.superTypes.indexOf(it), createTypeSpecifier => [
					type = getSubmachineContextInterface.asCsharpDeclaration
				]) 
		]
	}

	def protected substituteEventValueReferenceExpressionsByEventValue(ExecutionFlow flow) {
		flow.eAllContents.filter(EventValueReferenceExpression).toList.forEach [
			if(it.value instanceof FeatureCall){
				(it.value as FeatureCall).feature = flow.eventValueReference(it)
				EcoreUtil.replace(it, it.value)
			}	
			else
				EcoreUtil.replace(it, flow.eventValueReference(it)._ref)
		]
	}
	
	def protected substitueEventRaisedExpression(ExecutionFlow flow) {
		flow.eAllContents.filter(ElementReferenceExpression).filter[!(reference instanceof Operation)].toList.forEach [ eleRef |
				//It is a getter defined in the Iface -> Needs iface access
				val parent = eleRef.reference?.eContainer
				if(parent !== null && parent.isCsharpInterface){
					EcoreUtil.replace(eleRef, eleRef.reference.stateMachineClass.features.filter(ComplexType).filter[
						superTypes.head?.type === eleRef.reference.eContainer
					].head.features.filter(Part).head._ref._dot(eleRef.reference))	
				}
			]
	}
	
	def protected eventValueReference(ExecutionFlow flow, EventValueReferenceExpression evr) {
		val evValMember = (evr.value.featureOrReference as TypedDeclaration).eventValueMemberType
		if(evValMember.isInNamedIface || evValMember.eContainer.childStatechart)
			evValMember.eventSignature
		else
			evValMember
	}

	def substitueStateMachineConcepts(ExecutionFlow flow) {
		flow.eAllContents.filter(ElementReferenceExpression).filter [
			reference instanceof NamedElement && (reference as NamedElement).isStateMachineConcept
		].forEach [
			if ((reference as NamedElement).name == EventProcessing.NEXT_EVENT)
				it.reference = flow.nextEventMethod
		]
	}

	override protected void transformReferences() {

		val transformedTargetDeclarations = targetScopes.map [
			eAllContents.filter(NamedElement).filter[origin !== null].toSet
		].flatten.toSet
		val allSourceDeclarations = sourceScopes.map[eAllContents.filter(NamedElement).toSet].flatten.toSet

		val sourceToTargetMap = new HashMap<NamedElement, NamedElement>
		transformedTargetDeclarations.forEach [
			val origin = it.origin
			if (origin instanceof NamedElement) {
				if (allSourceDeclarations.contains(origin)) {
					sourceToTargetMap.put(origin, it)
				}
			}
		]

		// We require this because we still rely on sexec for the flow code, thus we want to change references/features contained in it to point to the newly constructed codemodel elements
		sourceScopes.forEach[eAllContents.forEach[retargetDeclarationReferences(sourceToTargetMap)]]
		targetScopes.forEach[eAllContents.forEach[retargetDeclarationReferences(sourceToTargetMap)]]
	}

	def substitueOperationCalls(ExecutionFlow flow) {
		flow.eAllContents.filter(ArgumentExpression).filter [
			featureOrReference instanceof Operation && featureOrReference.eContainer.isCsharpInterface
		].toList.forEach [

			// That means we have the oc in an named iface
			if (it instanceof FeatureCall) {
				val ifaceOcbCall = it.owner.featureOrReference._ref._dot(
					featureOrReference.scope.scopeClass.features.filter(TypedDeclaration).filter [ td |
						td.type === featureOrReference.eContainer
					].head)._dot(featureOrReference) => [ fc |
					fc.arguments += arguments
				]

				EcoreUtil.replace(it, ifaceOcbCall)
			} else
				EcoreUtil.replace(it, featureOrReference.scope.scopeClass.features.filter(TypedDeclaration).filter [ td |
					td.type === featureOrReference.eContainer
				].head._ref._dot(featureOrReference) => [ fc |
					fc.arguments += arguments
				])
		]
	}

	override dispatch retargetDeclarationReferences(ElementReferenceExpression it,
		Map<NamedElement, NamedElement> map) {
		val prop = EcoreUtil2.getContainerOfType(it, Property)
		if (prop === null || prop.initialValue instanceof PrimitiveValueExpression)
			map.target(it.reference as NamedElement).ifPresent[target|it.reference = target]
	}
	
	def protected setOperationCall(Declaration method, ComplexType smClass){
		method.eAllContents.filter(FeatureCall).filter[feature instanceof Operation].toList.forEach[ source |
			val target = smClass.eAllContents.filter(Declaration).getTarget(source)
			val newSource = source.makeOperationCall
			if(target !== source.feature)
				EcoreUtil.replace(source, newSource)
			if (target !== null)
				EcoreUtil.replace(newSource, newSource.replaceSource(target))
		]		
	}

	def protected substitueMethodMember(Declaration method, ComplexType smClass) {
		method.eAllContents.filter(Argument).map[value].filter(ArgumentExpression).toList.forEach [ source |
			val target = smClass.eAllContents.filter(Declaration).getTarget(source)
			if (target !== null)
				EcoreUtil.replace(source, source.replaceSource(target))
		]

		// Have to do this separately and in order to be able to replace both ref and owner
		method.eAllContents.filter(ElementReferenceExpression).toList.forEach [ source |
			val target = smClass.eAllContents.filter(Declaration).getTarget(source)
			if (target !== null)
				EcoreUtil.replace(source, source.replaceSource(target))
		]

		method.eAllContents.filter(FeatureCall).toList.forEach [ source |
			val target = smClass.eAllContents.filter(Declaration).getTarget(source)
			if (target !== null)
				EcoreUtil.replace(source, source.replaceSource(target))
		]
	}

	def dispatch protected makeOperationCall(ArgumentExpression it) {
		return it
	}

	def dispatch protected makeOperationCall(FeatureCall it) {
		return it.copy => [ s |
			s.operationCall = true
			s.feature = (feature as NamedElement).copy => [
				name = name.toFirstUpper
			]
		]
	}

	def protected Declaration getTarget(Iterator<Declaration> content, ArgumentExpression source) {
		content.filter [ c |
			if(c instanceof ComplexType) c.eAllContents.filter(Declaration).getTarget(source)
			c.origin !== null && source.featureOrReference !== null && (c.origin === source.featureOrReference || 
			((c instanceof Operation) && c.origin === source.featureOrReference.origin
				
			))
		].head
	}

	def protected dispatch ArgumentExpression replaceSource(ElementReferenceExpression it, Declaration refer) {
		it.copy => [ e |
			if (refer instanceof ComplexType)
				e.reference = refer.features.filter[p|
					p.origin === reference
				].head
			else
				e.reference = refer

		]
	}

	def protected dispatch ArgumentExpression replaceSource(FeatureCall it, Declaration refer) {
		it.copy => [
			if(owner instanceof FeatureCall) owner = (owner as FeatureCall).swapOwners(refer)
			feature = refer 
		]
	}

	def FeatureCall swapOwners(FeatureCall it, Declaration refer) {
		it.copy => [
			if (owner instanceof FeatureCall) {
				feature = (refer.eContainer.eContainer as ComplexType).features.filter [ p |
					p.origin === feature
				].head
				owner = (owner as FeatureCall).swapOwners(refer.eContainer as Declaration)
			} else if (owner instanceof ElementReferenceExpression) {
				feature = (refer.eContainer.eContainer as ComplexType).features.filter [ p |
					p.origin === feature
				].head
			}
		]
	}

}
