/**
 * Copyright (c) 2023-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.itemis.create.base.generator.csharp.codemodel

import com.google.inject.Inject
import com.itemis.create.base.generator.core.concepts.InternallyDefinedTypeAnnotation
import com.yakindu.base.base.NamedElement
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Property
import com.yakindu.sct.types.resource.Statechart2TypeTransformation
import org.eclipse.emf.ecore.EObject

class CsharpNaming {

	public static final String[] CSHARP_KEYWORDS = #["abstract", "as", "base", "bool", "break", "byte", "case", "catch",
		"char", "checked", "class", "const", "continue", "decimal", "default", "delegate", "do", "double", "else",
		"enum", "event", "explicit", "extern", "false", "finally", "fixed", "float", "for", "foreach", "goto", "if",
		"implicit", "in", "int", "interface", "internal", "is", "lock", "long", "namespace", "new", "null", "operator",
		"out", "override", "params", "private", "protected", "public", "readonly", "ref", "return", "sbyte", "sealed",
		"short", "sizeof", "stackalloc", "static", "string", "struct", "switch", "this", "throw", "true", "try",
		"typeof", "uint", "ulong", "unchecked", "unsafe", "ushort", "using", "virtual", "void", "volatile", "while",
		"value", "object", "_semaphore", "await", "async"]

	public static final String[] CSHARP_SYSTEM_KEYWORDS = #["System.Threading.Thread.CurrentThread.ManagedThreadId"]

	public static final String[] STATECHART_KEYWORDS = #["parent", "timerService", "timeEvents", "stateConfVector",
		"stateConfVectorPosition", "current", "isExecuting", "operationCallback"]

	val protected char separator = '_'

	@Inject protected extension CsharpVisibility
	@Inject protected extension InternallyDefinedTypeAnnotation
	@Inject protected extension Statechart2TypeTransformation

	def applyCsharpNamingConventions(EObject it) {
		it.eAllContents.filter(NamedElement).forEach[applyCsharpName]
	}

	def protected dispatch void applyCsharpName(NamedElement it) {
	}

	def protected dispatch void applyCsharpName(Operation it) {
		name = csharpName.isCsharpKeyword ? csharpName + separator + 'ID' : csharpName
	}

	def protected dispatch void applyCsharpName(ComplexType it) {
		if(it.isNamedIface)
			name = name.replaceFirst("Type","")
		name = csharpName.isCsharpKeyword ? csharpName + separator + 'ID' : csharpName
	}

	def protected dispatch void applyCsharpName(Property it) {
		name = (csharpName.isCsharpKeyword || csharpName.isStatechartKeyword) && !isInternallyDefinedType ? csharpName +
			separator + 'ID' : csharpName
	}

	def csharpName(ComplexType it) {
		name.asIdentifier.toFirstUpper
	}

	def csharpName(Operation it) {
		name.asIdentifier.toFirstUpper
	}

	def csharpName(Property it) {
		if(name.isCsharpSystemKeyword)
			name
		else if (isPrivate || isInternal)
			name.asIdentifier.toFirstLower
		else
			name.asIdentifier.toFirstUpper
	}
	
	def boolean isCsharpSystemKeyword(String name) {
		return !CSHARP_SYSTEM_KEYWORDS.findFirst[it.equalsIgnoreCase(name)].nullOrEmpty
	}

	def boolean isCsharpKeyword(String name) {
		return !CSHARP_KEYWORDS.findFirst[it.equalsIgnoreCase(name)].nullOrEmpty
	}

	def boolean isStatechartKeyword(String name) {
		return !STATECHART_KEYWORDS.findFirst[it.equalsIgnoreCase(name)].nullOrEmpty
	}

	def private asIdentifier(String string) {
		if (string !== null)
			string.replaceAll('[[^a-z]&&[^A-Z]&&[^0-9]]', separator.toString)
	}

}
