/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.itemis.create.base.generator.csharp.concepts

import com.google.inject.Inject
import com.google.inject.Singleton
import com.itemis.create.base.generator.core.GeneratorAssignment
import com.itemis.create.base.generator.core.codepattern.IVariableCode
import com.itemis.create.base.generator.csharp.codemodel.CsharpTypeBuilder
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.BlockExpression
import com.yakindu.base.expressions.expressions.ExpressionsFactory
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.TypedDeclaration
import com.yakindu.base.types.annotations.VisibilityAnnotations
import com.yakindu.sct.generator.core.codemodel.InterfaceContext
import com.yakindu.sct.generator.core.codemodel.NamedInterfaceClasses
import com.yakindu.sct.generator.core.codemodel.StatemachineClass
import com.yakindu.sct.model.sexec.concepts.EventQueue
import com.yakindu.sct.model.sexec.concepts.RunCycleMethod

/**
 * 
 * This concept is responsible to create the required model objects for the statechart
 * for it to be able to satisfy thread-safe, serialized access.
 * This class also responsible for providing Expression transformation for existing
 * expression based implementations. The transformation does the required 
 * modifications to guard the existing execution with semaphore, waiting, try catch
 * 
 * @author laszlo kovacs - Initial contribution
 *
 */
@Singleton
class AsyncAwait {
	
	static String WAIT_ASYNC = "WaitAsync"
	static String WAIT = "Wait"
	static String RELEASE = "Release"
	
	@Inject protected extension TypeBuilder
	@Inject protected extension CsharpTypeBuilder
	@Inject protected extension GeneratorAssignment
	@Inject protected extension VisibilityAnnotations
	@Inject protected extension ExpressionBuilder
	@Inject protected extension StatemachineClass
	@Inject protected extension NamedInterfaceClasses
	@Inject protected extension InterfaceContext
	@Inject protected extension EventQueue
	@Inject protected extension IVariableCode
	@Inject protected extension RunCycleMethod
	@Inject protected extension ExpressionExtensions
	
	protected extension ExpressionsFactory = ExpressionsFactory.eINSTANCE
	
	def addAsyncAwait(ComplexType stm, boolean requiresEventQueue){
		stm.features += stm.semaphore
		if(requiresEventQueue){
			stm.features += stm.createThreadID
			stm.features += managedThreadID
		}			
	}
	
	def create m : _variable("await") await(){
		
	}
	
	def create m : _complexType("async Task") asyncTaskType(){
		
	}
	
	def create m : _complexType("Task") taskType(){
		
	}
	
	def create it : _variable("_semaphore", semaphoreType) semaphore(ComplexType stm){
		_private
		_readonly
		generateDefinitionWith[ '''«visibility.visibilityName» readonly «type.name» «name» = new «type.name»(1, 1);''' ]
		
	}
	
	def create it : _variable("_ownerThreadId", _integer) createThreadID(ComplexType stm){
		_private
		initialValue = (-1)._value
		generateDefinitionWith[ variableDeclarationCode ]
		
	}
	
	def threadID(ComplexType it){
		_createCache_createThreadID.get(#[it])
	}
	
	def create it : _variable("System.Threading.Thread.CurrentThread.ManagedThreadId", _any) managedThreadID(){	}
	
	def create it : _complexType("SemaphoreSlim") semaphoreType(){
		features += _op(WAIT_ASYNC)
		features += _op(WAIT)
		features += _op(RELEASE)
	}
	
	def waitAsyncOp(ComplexType semaphore){
		semaphore.features.filter(Operation).filter[name === WAIT_ASYNC].head
	}
	
	def waitOp(ComplexType semaphore){
		semaphore.features.filter(Operation).filter[name === WAIT].head
	}
	
	def releaseOp(ComplexType semaphore){
		semaphore.features.filter(Operation).filter[name === RELEASE].head
	}
	
	def guardWithSemaphore(TypedDeclaration prop, Expression execBody, boolean sync){
		_block() => [
			if(prop.stateMachineClass.threadID !== null)
				expressions +=_if(prop.propThreadId._equals(managedThreadID._ref))._then(execBody.threadIsDifferent(prop))
			expressions += prop.propWaitFor(sync)
			if(prop.stateMachineClass.threadID !== null)
				expressions += prop.propThreadId._assign(managedThreadID._ref)
			expressions += 
				_try(execBody)
				._finally(
					_block() => [ finalBlock |
						if(prop.stateMachineClass.threadID !== null){
							finalBlock.expressions += prop.propThreadId._assign((-1)._value)
						}
						finalBlock.expressions += prop.propRelease
					]				
			)
		]
	}
	
	def Expression threadIsDifferent(Expression execBody, TypedDeclaration prop){
		if(execBody instanceof BlockExpression){
			val execBodyCopy = execBody.copy
			if(execBodyCopy.expressions.get(execBodyCopy.expressions.size-1).featureOrReference === prop.stateMachineClass.runCycle)
				execBodyCopy.expressions.remove(execBodyCopy.expressions.size-1)
			execBodyCopy.expressions += _return(null)
			return execBodyCopy
		}
		else
			return _block(execBody, _return(null))
	}
	
	def propThreadId(TypedDeclaration prop){
		if(prop.needsParent)
			parentFor(prop)._ref._dot(prop.stateMachineClass.threadID)
		else
			prop.stateMachineClass.threadID._ref
	}
	
	def propRelease(TypedDeclaration prop){
		if(prop.needsParent)
			parentFor(prop)._ref._dot(prop.stateMachineClass.semaphore)._dot(semaphoreType.releaseOp)
		else
			prop.stateMachineClass.semaphore._ref._dot(semaphoreType.releaseOp)
	}
	
	def propWaitFor(TypedDeclaration prop, boolean sync){
		if(prop.needsParent)
			parentFor(prop)._ref._dot(prop.stateMachineClass.semaphore).waitForSemaphore(sync)
		else
			prop.stateMachineClass.semaphore._ref.waitForSemaphore(sync)
	}
	
	def waitForSemaphore(ArgumentExpression base, boolean sync){
		if(sync)
			base._dot(semaphoreType.waitOp)
		else
			base._dot(semaphoreType.waitAsyncOp)._meta(await)
	}
	
	def changeReturnTypeToTask(Operation op){
		op.typeSpecifier = _typeSpecifier(taskType)
	}
	
	def changeReturnTypeToAsyncTask(Operation op){
		op.typeSpecifier = _typeSpecifier(asyncTaskType)
	}
}