/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 *
 */
package com.yakindu.sct.domain.java.resource.trafo.types

import com.google.inject.Inject
import com.google.inject.Singleton
import java.util.Collection
import org.eclipse.emf.common.util.URI
import org.eclipse.jdt.core.IField
import org.eclipse.jdt.core.IJavaElement
import org.eclipse.jdt.core.IJavaProject
import org.eclipse.jdt.core.ILocalVariable
import org.eclipse.jdt.core.IMethod
import org.eclipse.jdt.core.IType
import org.eclipse.jdt.core.ITypeParameter
import org.eclipse.jdt.core.Signature
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypesFactory
import com.yakindu.base.types.typesystem.ITypeSystem

/**
 * @author Rene Beckmann - Initial API and contribution
 */
@Singleton
class TypeLookup {
	@Inject protected ITypeSystem ts
	@Inject protected TypeScope typeScope
	@Inject protected ProxyBuilder proxyBuilder

	protected URI uri

	def Type fromTypeSystem(String name) {
		ts.getType(name)
	}
	
	/** 
	 * Adds a new transparent layer to the typeScope.
	 * Should be used before a method's type parameters are
	 * registered and pop() should be called when the method is 
	 * done (deletes the type parameters)
	 */
	def void push() {
		typeScope.push
	}

	def void pop() {
		typeScope.pop
	}

	def void setURI(URI uri) {
		this.uri = uri
	}

	def void registerType(IJavaProject project, String name, Type type) {
		var projectQualifiedName = '''«project.project.name»#«name»'''
		typeScope.put(projectQualifiedName, type)
	}

	/**
	 * Resolves the type from a signature using the given type as resolving context.
	 */
	def Type fromSignature(IType type, String typeSignature) {
		doResolve(type.javaProject, Signature.toString(typeSignature), type)
	}
	
	def Iterable<String> superTypeSignatures(IType type) {
		val ret = newArrayList
		val scTs = type.superclassTypeSignature
		if(scTs !== null) {
			ret.add(scTs)
		}
		ret += type.superInterfaceTypeSignatures.toList
		ret
	}

	def dispatch Type resolveType(IType it) {
		loadType(javaProject, it.getFullyQualifiedName)
	}

	def dispatch Type resolveType(IMethod it) {
		doResolve(typeSignature(it), declaringType(it))
	}

	def dispatch Type resolveType(ILocalVariable it) {
		doResolve(typeSignature(it), declaringType(it))
	}

	def dispatch Type resolveType(IField it) {
		doResolve(typeSignature(it), declaringType(it))
	}

	def dispatch Type resolveType(ITypeParameter it) {
		typeScope.get(it.elementName)
	}

	def dispatch int arrayDepth(ILocalVariable it) {
		Signature.getArrayCount(it.typeSignature)
	}

	def dispatch int arrayDepth(IField it) {
		Signature.getArrayCount(it.typeSignature)
	}

	def dispatch int arrayDepth(IMethod it) {
		Signature.getArrayCount(it.returnType)
	}

	def protected Type doResolve(IJavaElement it, String signature, IType declaringType) {
		val fqn = resolveFullyQualifiedName(javaProject, declaringType, signature)
		loadType(javaProject, fqn)
	}

	def Collection<Type> getTypeArguments(IField field) {
		getTypeArguments(field.javaProject, field.typeSignature, field.declaringType)
	}
	
	def Collection<Type> getTypeArguments(ILocalVariable variable) {
		getTypeArguments(variable.javaProject, variable.typeSignature, variable.declaringType)
	}
	
	def Collection<Type> getTypeArguments(IMethod method) {
		getTypeArguments(method.javaProject, method.returnType, method.declaringType)
	}

	def Collection<Type> getTypeArguments(IJavaProject project, String signature, IType declaringType) {
		Signature.getTypeArguments(signature).map[
			var pretty = Signature.toString(it)
			if(Signature.getTypeSignatureKind(it) == Signature.WILDCARD_TYPE_SIGNATURE && it.startsWith(Signature.C_EXTENDS.toString)) {
				pretty = pretty.replace("? extends ", "")
				val namedType = doResolve(project, pretty, declaringType)
				val type = TypesFactory.eINSTANCE.createTypeParameter
				type.name = "?"
				val ts = TypesFactory.eINSTANCE.createTypeSpecifier
				ts.type = namedType
				type.superTypes += ts
				type
			} else {
				val namedType = doResolve(project, pretty, declaringType)
				namedType
			}
		].toList
	}

	def protected String resolveFullyQualifiedName(IJavaProject project, IType declaringType, String signature) {
		/* Check if signature is primitive type, like int, char, boolean */
		if(ts.getType(signature) !== null) {
			return signature
		}
		/* If the declaring type is binary, all type signatures _should_ be fully qualified already */
		if(declaringType.isBinary) {
			return Signature.getTypeErasure(signature)
		}
		
		return buildResolvedName(project, declaringType, signature)
	}

	/**
	 * Calls JDT's function IType.resolveType(String) to resolve the given type name signature relative to the
	 * given declaringType.
	 */
	def protected String buildResolvedName(IJavaProject project, IType declaringType, String signature) {
		var resolved = declaringType.resolveType(signature)
		if (resolved === null || resolved.length == 0 || resolved.get(0) === null) {
			return signature
		} else {
			val result = resolved.get(0).filter[!it.empty].join('.')
			val type = project.findType(result)
			return type.fullyQualifiedName
		}
	}

	def protected dispatch String typeSignature(IMethod it) {
		var signature = Signature.getTypeErasure(it.returnType)
		signature = Signature.getElementType(signature)
		signature = Signature.toString(signature)
		signature
	}

	def protected dispatch String typeSignature(IField it) {
		Signature.toString(Signature.getElementType(it.typeSignature))
	}

	def protected dispatch String typeSignature(ILocalVariable it) {
		Signature.toString(Signature.getElementType(it.typeSignature))
	}

	def protected dispatch IType declaringType(IMethod it) {
		it.declaringType
	}

	def protected dispatch IType declaringType(IField it) {
		it.declaringType
	}

	def protected dispatch IType declaringType(ILocalVariable it) {
		it.declaringMember.declaringType
	}

	def protected Type loadType(IJavaProject project, String typeName) {
		var projectQualifiedName = '''«project.project.name»#«typeName»'''
		var type = ts.getType(typeName)
		if (type !== null) {
			return type
		}
		type = typeScope.get(projectQualifiedName)
		if (type !== null) {
			return type
		}
		type = proxyBuilder.build(typeName, project)
		typeScope.put(projectQualifiedName, type)
		return type
	}
	
}
