/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Axel Terfloth - itemis AG
 * 
 */
package com.yakindu.base.types.adapter

import org.eclipse.emf.ecore.EObject

import static com.yakindu.base.types.adapter.Adapters.*

/**
 * This extension provides access to all information required for tracing from a statechart type to the original statechart. 
 * 
 * This extension is stateless. It encapsulates the storage of traces information. This is currently based on 'eAdapters'. 
 * 
 * @author terfloth
 */
class OriginTracing {

	/** 
	 * Adds a structural trace to an element. 'origin' is the original element in the statechart. These kind of traces 
	 * can be used for navigation purpose.
	 */
	def <T extends EObject> T traceOrigin(T it, Object origin) {
		it.eAdapters.add(new OriginTraceAdapter(origin))
		return it
	}

	/** 
	 * Returns all origins that are traces for a given slang model element. 
	 * The result also includes execution trace origins.
	 */
	def getOriginTraces(EObject it) {
		it.eAdapters.filter(OriginTraceAdapter).map[getOrigin()].toList
	}

	/**
	 * Returns the first origin if it is defined. 
	 */
	def Object origin(EObject it) {
		it?.eAdapters.filter(OriginTraceAdapter).head?.origin
	}

	/**
	 * Sets the first defined origin trace. This method replaces the first origin if it exists.
	 */
	def void setOrigin(EObject it, EObject origin) {
		assignAdapter(it, new OriginTraceAdapter(origin))
	}

	def EObject originSource(EObject it) {
		if (origin === null || !(origin instanceof EObject)) {
			it
		} else
			(origin as EObject).originSource
	}

}
